"""IO utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import math
import os
from pathlib import Path
import tempfile

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs
from xms.coverage.xy.xy_io import XyReader
from xms.coverage.xy.xy_series import XySeries
from xms.guipy import file_io_util
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.file_io.text_file_context_manager import TextFileContextManager
from xms.mf6.misc import util

# Constants (shouldn't be changed)
mftab = '  '
"""How much to indent blocks"""
mfsep = ' '  # Unfortunately flopy can't handle commas (but MODFLOW 6 can)
"""Separator used in cleaned external files."""
comment_lines = 6  # Number of lines in the comment edit field
"""Number of lines in the comment edit field"""


def _random_string(length: int) -> str:
    """Return a random string of given length.

    String can include upper case letters, lower case letters, digits, and underscores.

    Args:
        length:

    Returns:
        See description.
    """
    from random import Random
    import string
    characters = string.ascii_letters + string.digits + '_'
    random_obj = Random()
    random_string = ''.join(random_obj.choice(characters) for _ in range(length))
    return random_string


def get_temp_filename(folder: str | Path = None, suffix: str = '') -> str:
    """Returns a temporary filename, in the XMS temp directory by default.

    Args:
        folder: If provided, filename will be in folder. Otherwise, folder will be get_xms_temp_directory() result.
        suffix: The suffix to use for the file. You must include a '.' if you want it to be an extension.

    Returns:
        See description.
    """
    filename = f'tmp{_random_string(8)}{suffix if suffix else ""}'
    if folder:  # If we call the next line with folder == '', it seems to use the working directory.
        path = str(Path(folder) / filename)
    else:
        temp_dir = get_xms_temp_directory()
        path = str(temp_dir / filename)
    return path


def is_begin_x_line(line, x):
    """Returns true if the line is a 'BEGIN <x>' line where x is given.

    Args:
        line (str): A line read from a file.

    Returns:
        See description.
    """
    words = line.split(maxsplit=2)
    return words and len(words) > 1 and words[0].upper() == 'BEGIN' and words[1].upper() == x.upper()


def is_begin_line(line):
    """Returns true if the line starts with 'BEGIN'.

    Args:
        line (str): A line read from a file.

    Returns:
        (bool): True or False
    """
    words = line.split(maxsplit=1)
    return words and words[0].upper() == 'BEGIN'


def is_end_block_line(line, block_name):
    """Returns true if the line starts with 'END'.

    Args:
        line (str): A line read from a file.
        block_name (str): What comes after BEGIN and END.

    Returns:
        (bool): True or False
    """
    words = line.split()
    if words and words[0].upper() == 'END':
        if block_name:
            if len(words) > 1 and words[1].upper() == block_name.upper():
                return True

    return False


def skip_block(file):
    """Reads and ignores lines in the file until the 'END' of the block is found.

    Args:
        file: A file.
    """
    # for line in file:
    # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
    while True:
        line = file.readline()
        if not line:
            break

        if is_end_line(line):
            return


def is_comment(line):
    """Returns true if the line is a comment.

    Args:
        line (str): A line of text from a file.

    Returns:
        (bool): True or False.
    """
    if not line:
        return False

    # Find the first non whitespace character
    pos = -1
    for idx, chr in enumerate(line):
        if not chr.isspace():
            pos = idx
            break

    if pos >= 0:
        if line[pos] == '#' or line[pos] == '!' or\
                (pos + 1 < len(line) and line[pos:pos + 2] == '//'):
            return True
    return False


def is_gms_comment(line):
    """Returns true if the line is a GMS comment (starts with #GMSCOMMENT.

    Args:
        line (str): A line of text from a file.

    Returns:
        (bool): True or False.
    """
    return line.startswith('#GMSCOMMENT')


def is_comment_or_blank(line):
    """Returns true if the line is a comment line or a blank line.

    Args:
        line (str): A line of text read from a file.

    Returns:
        (bool): True if the line is a comment or is blank.
    """
    if not line.strip():
        return True
    return is_comment(line)


def is_end_line(line):
    """Returns true if the line starts with 'END'.

    Args:
        line (str): A line read from a file.

    Returns:
        (bool): True or False
    """
    words = line.split(maxsplit=1)
    return words and words[0].upper() == 'END'


def file_has_readasarrays(filepath: str | Path):
    """Returns True if the file contains the 'READASARRAYS' option.

    Args:
        filepath: The file path.

    Returns:
        (bool): True or False
    """
    # with open(filepath, 'r') as package_file:
    with TextFileContextManager(filepath) as package_file:
        # for line in package_file:
        # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
        while True:
            line = package_file.readline()
            if not line:
                break

            # if is_comment_or_blank(line):
            #     continue
            if is_begin_x_line(line, 'OPTIONS'):
                # for option_line in package_file:
                # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
                while True:
                    option_line = package_file.readline()
                    if not option_line:
                        break

                    # if is_comment_or_blank(option_line):
                    #     continue
                    # elif is_end_line(option_line):
                    if is_end_line(option_line):
                        return False
                    else:
                        words = option_line.split()
                        if words and words[0].upper() == 'READASARRAYS':
                            return True


def is_temporary_file(filename: Path | str):
    """Returns True if the filename is a temporary file (in the system temp directory).

    Args:
        filename: A filename

    Returns:
        (bool): See description.
    """
    # temp_dir = get_xms_temp_directory()
    # dirname = os.path.realpath(os.path.dirname(filename))
    # return fs.paths_are_equal(temp_dir, dirname)

    # system_temp = Path(tempfile.gettempdir())
    # return system_temp in Path(filename).parents

    return Path(filename).suffix == '.mf6_tmp'


def read_array_values(file, nvalues):
    """Reads through the lines/values of the array.

    Args:
        file (_io.TextIOWrapper):
        nvalues (int): Number of values to read.

    Returns:
        (tuple): tuple containing:
            - all_words (list of str): List of all array words (empty if not externalizing).
            - extra_words_after_array (str): The end of the line after last array word. Typically blank.
    """
    all_words = []
    words_read = 0
    extra_words_after_array = ''
    done = False
    while not done:
        line = file.readline()
        line = line.strip().replace(',', ' ')
        words = line.split()
        words_read += len(words)
        if words_read > nvalues:
            remainder = words_read - nvalues
            words = words[:-remainder]
            done = True
        elif words_read == nvalues:
            done = True

        all_words.extend(words)

    return all_words, extra_words_after_array


def write_array_values(values, shape, file):
    """Writes array values to the file using the shape.

    Args:
        values: List of values
        shape: Tuple of array shape
        file: The file, already opened, to write to
    """
    i = 0
    for _row in range(shape[0]):
        for _col in range(shape[1]):
            file.write(f'{values[i]} ')
            i += 1
        file.write('\n')


def get_external_filename(stress_period, nper, filename, name=''):
    """Returns the external filename as a tuple: the full filename, the relative filename.

    Args:
        stress_period (int): The stress period.
        nper (int): Number of stress periods
        filename (str): The package filename.
        name (str): Optional. Name to be appended to filename.

    Returns:
        (tuple): tuple containing:
            - full_filename (str): The full filename.
            - relative_filename (str): The relative filename.
    """
    sp_word = ''
    if stress_period is not None and stress_period != -1:
        ndigits = int(math.log10(nper)) + 1
        number_str = str(stress_period).zfill(ndigits)  # Use leading zeros
        sp_word = f'_{number_str}'
    name_word = '' if not name else f'_{name}'
    full_filename = f'{filename}{sp_word}{name_word}.txt'
    output_dir = os.path.dirname(filename)
    relative_filename = fs.compute_relative_path(output_dir, full_filename)
    return full_filename, relative_filename


def write_file_internal(fp, filename):
    """Writes list data internally to the file.

    Args:
        fp (_io.TextIOWrapper): the file
        filename (str): external file name
    """
    with open(filename, "r") as external_file:
        for line in external_file:
            fp.write(f'{mftab}{line}')


def write_option(options, key, file):
    """Writes an options block line to the file.

    Args:
        options (dict): The options dict.
        key (str): The key in the dict. Also gets written to the file.
        file (_io.TextIOWrapper): The file written to.
    """
    if key in options:
        value = options[key]
        if value:
            if isinstance(value, list) and len(value) > 0:
                word2 = f' {" ".join(value)}'
            else:
                word2 = f' {value}'
        else:
            word2 = ''
        file.write(f'{mftab}{key}{word2}\n')


def remove_trailing_comment(line):
    """Returns the line minus any comment (or the original line if no comment found).

    Args:
        line (str): A line from a file.

    Returns:
        (str): The line without the trailing comment if there was one.
    """
    for i in range(len(line)):
        if line[i] == '#' or line[i] == '!' or (line[i] == '/' and i < len(line) - 1 and line[i + 1] == '/'):
            return line[0:i]
    return line


def count_max_line(files):
    """Counts the non-blank lines in the list of files and returns the max count.

    Args:
        files (list of str): List of files.

    Returns:
        int: The maximum number of non-blank lines in any file.
    """
    max_line = 0
    for file in files:
        if not file:
            continue

        count = 0
        # with open(file, "r") as fh:
        #     for line in fh:
        # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
        with TextFileContextManager(file) as fh:
            while True:
                line = fh.readline()
                if not line:
                    break

                # if is_comment_or_blank(line):
                #     continue
                if line.strip():
                    count += 1
        if count > max_line:
            max_line = count
    return max_line


def grid_info_from_dis_file(dis_filename: Path | str):
    """Reads the dis file (DIS or DISV) and returns nrow, ncol and ncpl.

    If ftype is 'DIS6', nrow and ncol are meaningful and ncpl is -1.
    If ftype is 'DISV6', ncpl is meaningful and nrow and ncol are -1.

    Args:
        dis_filename: Full path to dis filename.

    Returns:
        (GridInfo): The grid dimensions.
    """
    grid_info = GridInfo()
    try:
        with TextFileContextManager(dis_filename) as dis_file:
            # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
            while True:
                line = dis_file.readline()
                if not line:
                    break

                if is_begin_line(line):
                    if is_begin_x_line(line, 'DIMENSIONS'):
                        # Use "while True" syntax and not "for line in file" cause the latter doesn't work with TextFile
                        while True:
                            dimension_line = dis_file.readline()
                            if not dimension_line:
                                break

                            if is_end_block_line(dimension_line, 'DIMENSIONS'):
                                break
                            _read_dimension_line_into_grid_info(dimension_line, grid_info)
                    else:
                        skip_block(dis_file)
        return grid_info
    except FileNotFoundError:
        return None


def _read_dimension_line_into_grid_info(dimension_line, grid_info):
    """Reads the line from the file and stores it in the proper place in the GridInfo object.

    Args:
        dimension_line: A line from the file.
        grid_info: The GridInfo object.
    """
    words = dimension_line.split()
    if words and len(words) > 1:
        if words[0].upper() == 'NLAY':
            grid_info.nlay = int(words[1])
        elif words[0].upper() == 'NROW':
            grid_info.dis_enum = DisEnum.DIS
            grid_info.nrow = int(words[1])
        elif words[0].upper() == 'NCOL':
            grid_info.dis_enum = DisEnum.DIS
            grid_info.ncol = int(words[1])
        elif words[0].upper() == 'NCPL':
            grid_info.dis_enum = DisEnum.DISV
            grid_info.ncpl = int(words[1])
        elif words[0].upper() == 'NVERT':
            grid_info.nvert = int(words[1])
        elif words[0].upper() == 'NODES':
            grid_info.dis_enum = DisEnum.DISU
            grid_info.nodes = int(words[1])
        elif words[0].upper() == 'NJA':
            grid_info.dis_enum = DisEnum.DISU
            grid_info.nja = int(words[1])


def clean_up_temp_files(file_list):
    """Deletes all temp files."""
    for filename in file_list:
        if is_temporary_file(filename) and os.path.exists(filename):
            os.remove(filename)


def copy_file_to_temp(filename):
    """Copies an original file to a temporary file and returns the temp filename.

    Args:
        filename (str): The filename to copy.

    Returns:
        The temp filename.
    """
    with open(get_temp_filename(get_xms_temp_directory()), mode='wt') as temp:
        fs.copyfile(filename, temp.name)
        temp_filename = temp.name
        temp.close()
    return temp_filename


def get_xms_temp_directory() -> Path:
    """Returns the path of the XMS temp directory in the system temp directory.

    If the XMS temp directory isn't found (not running XMS?), returns the system temp directory.

    Returns:
        (str): See description.
    """
    xms_temp = os.environ.get('XMS_PYTHON_APP_TEMP_DIRECTORY', 'unknown')
    if xms_temp == '' or xms_temp == 'unknown':
        xms_temp = tempfile.gettempdir()
    return Path(xms_temp)


def get_xms_version():
    """Returns the path of the XMS temp directory in the system temp directory.

    If the XMS temp directory isn't found (not running XMS?), returns the system temp directory.

    Returns:
        (str): See description.
    """
    version = os.environ.get('XMS_PYTHON_APP_VERSION')
    return version


def type_caster_from_string(string_type):
    """Returns the type to cast the string to given the column_type string.

    Args:
        string_type (str): Type as a string, e.g. 'int', 'double', 'string'...

    Returns:
        The type, e.g. int, double, str
    """
    if string_type == 'int':
        return int
    elif string_type in ['double', 'float']:
        return float
    elif string_type == 'string':
        return str
    else:
        raise RuntimeError('Type string not recognized.')


def get_gms_options_filepath(parent_dir: str | Path) -> Path:
    """Returns the filepath to the GMS options file.

    Args:
        parent_dir (str | Path): Path to the directory where the gms options file is located.

    Returns:
        See description.
    """
    return Path(parent_dir) / 'gms_options.json'


def write_gms_options(parent_dir: Path | str, gms_options) -> None:
    """Writes the GMS options to disk.

    Args:
        parent_dir: Directory where the mfsim.nam file is located.
        gms_options: The GMS options.
    """
    gms_options_filepath = get_gms_options_filepath(parent_dir)
    file_io_util.write_json_file(gms_options, gms_options_filepath)


def quote(filepath: str) -> str:
    """Wraps filepath in single quotes if necessary (has a space and hasn't been already wrapped).

    "Although spaces within a file name are not generally recommended, they can be specified if the entire file name is
    enclosed within single quotes, which means that the file name itself cannot have a single quote within it."
    (mf6io.pdf)

    Args:
        filepath (str): A file path.
    """
    return f"'{filepath}'" if ' ' in filepath and not filepath.startswith("'") else filepath


def uuid_from_path(filepath: Path | str | None) -> str:
    """Returns the uuid string given a filepath.

    Checks stem first, then parent name.

    Args:
        filepath: A file path.

    Returns:
        See description.
    """
    if util.null_path(filepath):
        return ''

    path = Path(filepath)
    if util.is_valid_uuid(path.name):  # If path is to a directory with a uuid name
        return path.name
    elif util.is_valid_uuid(path.parent.name):  # If path is to a file in a directory with a uuid name
        return path.parent.name
    return ''


def file_is_binary(filepath: Path) -> bool:
    """Returns True if the file is a binary file.

    https://stackoverflow.com/questions/898669/how-can-i-detect-if-a-file-is-binary-non-text-in-python

    Args:
        filepath:

    Returns:
        (bool): See description.
    """
    textchars = bytearray({7, 8, 9, 10, 12, 13, 27} | set(range(0x20, 0x100)) - {0x7f})
    # is_binary_string = lambda bytes: bool(bytes.translate(None, textchars))
    with filepath.open('rb') as file:
        # return is_binary_string(file.read(1024))
        return bool(file.read(1024).translate(None, textchars))


def is_filename(path_string: Path | str) -> bool:
    """Return true if path_string is just a filename without any parent directory.

    Args:
        path_string: A path.

    Returns:
        See description.
    """
    path = Path(path_string)
    return path.parent == Path('.')


def find_existing_parent(path: Path | str) -> Path | None:
    """Traverses up the directory tree from the given path until an existing directory is found.

    Args:
        path: The starting path as a string.

    Returns:
        A pathlib.Path object representing the first existing parent directory, or None if no parent exists.
    """
    path = Path(path).resolve()
    while path.parent != path:  # Stop when reaching the root directory
        if path.is_dir():
            return path
        path = path.parent
    return None  # No existing parent directory found


def read_xy_series_file(
    att_file: Path | str, start_date_time: datetime | None, time_units: str, date_times_to_floats: bool
) -> dict[int, XySeries]:
    """Reads the XY series file and returns it as a dict.

    Args:
        att_file: Att table filename.
        start_date_time: Starting date/time
        time_units: Time units: 'UNKNOWN', 'SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS'
        date_times_to_floats: If true, and a start_date_time and time_units is given, converts date/time
         strings to relative offsets from the start_date_time.

    Returns:
        (dict): Dict of curve ID -> XySeries.
    """
    xy_dict: dict[int, XySeries] = {}
    xy_filename = str(att_file) + '.xy'
    reader = XyReader(date_times_to_floats, start_date_time, time_units)
    xy_list = reader.read_from_text_file(xy_filename)
    for series in xy_list:
        if series.x and series.x[0] is None:
            raise RuntimeError(
                'Could not parse time value. If using dates/times, ensure TDIS has a'
                ' valid START_DATE_TIME and TIME_UNITS set.\n'
            )
        xy_dict[series.series_id] = series
    return xy_dict


def write_lines_to_temp_file(lines: list[str], filepath: Pathlike = None, append: bool = False) -> str:
    """Write the lines to a temporary file and return the filepath.

    Args:
        lines: List of strings.
        filepath: The filepath.
        append: True to append, false to overwrite.

    Returns:
        See description.
    """
    if append and not filepath:
        raise ValueError('append can only be true if filepath is provided')
    if not filepath:
        filepath = get_temp_filename(suffix='.mf6_tmp')
    mode = 'at' if append else 'wt'
    with open(filepath, mode=mode) as file:
        for line in lines:
            file.write(f'{line}\n')
        file.close()
    return file.name
