"""CellPropertiesDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QSizePolicy, QTreeWidgetItem

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.resources import resources_util
from xms.guipy.resources.merged_svg_icon_engine import MergedSvgIconEngine

# 4. Local modules
from xms.mf6.components import dmi_util
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.cell_properties_dialog_ui import Ui_cell_properties_dialog
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.gwf.evt_list_dialog import EvtListDialog
from xms.mf6.gui.gwf.lak_dialog import LakDialog
from xms.mf6.gui.gwf.maw_dialog import MawDialog
from xms.mf6.gui.gwf.sfr_dialog import SfrDialog
from xms.mf6.gui.gwf.uzf_dialog import UzfDialog
from xms.mf6.gui.package_dialog import PackageDialog
from xms.mf6.gui.resources import mfqrc  # noqa W503 imported but unused (needed for icon paths)


class CellPropertiesDialog(XmsDlg):
    """A dialog for all 'list' (CHD, DRN, EVT, GHB, HFB, RCH, RIV, WEL) packages."""

    supported_ftypes = [
        # GWF
        'CHD6',
        'DRN6',
        'EVT6',
        'GHB6',
        'HFB6',
        'LAK6',
        'MAW6',
        'RCH6',
        'RIV6',
        'SFR6',
        'WEL6',
        'UZF6',
        # GWT
        'CNC6',
        'SRC6',
        # GWE
        'CTP6',
        'ESL6',
    ]
    """List of ftypes this dialog can handle. EVT6 and RCH6 packages must be list-based."""
    def __init__(self, dlg_input: DialogInput, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.cell_properties_dialog')

        self.dlg_input = dlg_input
        self.help_getter = gui_util.help_getter(dlg_input.help_id)
        self.ui = None
        self.uix = {}
        self.dialogs = {}  # Dict of package uuid -> dialog class
        self._current_uuid = ''  # TreeNode uuid from ProjectTree

        self.setup_ui()

    def setup_ui(self):
        """Sets up the GUI."""
        self.ui = Ui_cell_properties_dialog()
        self.ui.setupUi(self)

        self.setup_tree()

        # Signals
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def setup_tree(self):
        """Sets up the tree of dialogs."""
        current_item: QTreeWidgetItem | None = None
        first_item: QTreeWidgetItem | None = None
        for model in self.dlg_input.mfsim.models:

            # Add an item for the model
            model_item = create_tree_item(model, False)
            self.ui.tree_packages.addTopLevelItem(model_item)
            model_item.setExpanded(True)

            for package in model.packages:
                if package.ftype not in self.supported_ftypes:
                    continue

                locked = dmi_util.is_locked(package.tree_node, self.dlg_input.query)
                if self._load_dialog(package, locked):

                    # Add an item for the package
                    item = create_tree_item(package, locked)
                    model_item.addChild(item)

                    # Save the first and current item
                    if package == self.dlg_input.data:
                        current_item = item
                    if first_item is None:
                        first_item = item

            # Remove the model item if it has no children
            if model_item.childCount() == 0:
                self.ui.tree_packages.removeItemWidget(model_item, 0)

        extra_width = 20  # For some reason, self.ui.tree_packages.sizeHintForColumn(0) just isn't wide enough
        self.ui.tree_packages.setMinimumWidth(self.ui.tree_packages.sizeHintForColumn(0) + extra_width)

        self.ui.tree_packages.currentItemChanged.connect(self._on_current_item_changed)

        # Set the current item
        if current_item is None:
            current_item = first_item
        self.ui.tree_packages.setCurrentItem(current_item)
        self._current_uuid = current_item.data(0, Qt.UserRole)

    def _on_current_item_changed(self, current_item: QTreeWidgetItem, previous_item: QTreeWidgetItem):
        """Called when the current item changes.

        Args:
            current_item: Current item.
            previous_item: Previous item.
        """
        # Hide the previous dialog
        if self._current_uuid in self.dialogs:
            self.dialogs[self._current_uuid].hide()

        # Get new package uuid
        self._current_uuid = current_item.data(0, Qt.UserRole)
        if self._current_uuid not in self.dialogs:  # If it's a model node, show the first package below it
            current_item = current_item.child(0)
            if current_item:
                self._current_uuid = current_item.data(0, Qt.UserRole)

        # Show the new dialog
        if self._current_uuid in self.dialogs:
            self.dialogs[self._current_uuid].show()
            self.ui.lbl_package_path.setText('Package: ' + current_item.parent().text(0) + '/' + current_item.text(0))
            self.dialogs[self._current_uuid].updateGeometry()
        else:
            self.ui.lbl_package_path.setText('Model has no packages that have cell properties.')

    def _load_dialog(self, data: type[BaseFileData], locked: bool):
        """Loads a sub-dialog.

        Args:
            data: The data class.
            locked: True if the dialog is to be locked.

        Returns:
            True if a dialog was found and loaded, otherwise False.
        """
        # Create dialog input
        dlg_input = _create_dlg_input(self.dlg_input, data, locked)

        # Add dialog
        dlg = None
        if data.ftype in ['CHD6', 'CNC6', 'CTP6', 'DRN6', 'ESL6', 'GHB6', 'HFB6', 'RIV6', 'SRC6', 'WEL6']:
            dlg = PackageDialog(dlg_input, self)
        elif data.ftype == 'RCH6' and isinstance(data, ListPackageData):
            dlg = PackageDialog(dlg_input, self)
        elif data.ftype == 'EVT6' and isinstance(data, ListPackageData):
            dlg = EvtListDialog(dlg_input, self)
        elif data.ftype == 'LAK6':
            dlg = LakDialog(dlg_input, self)
        elif data.ftype == 'MAW6':
            dlg = MawDialog(dlg_input, self)
        elif data.ftype == 'SFR6':
            dlg = SfrDialog(dlg_input, self)
        elif data.ftype == 'UZF6':
            dlg = UzfDialog(dlg_input, self)

        if not dlg:
            return False

        dlg.setWindowFlags(dlg.windowFlags() & ~Qt.Dialog)

        # Hide the left side of the dialog and the buttonbox
        dlg.ui.wid_left.hide()
        dlg.ui.buttonBox.hide()

        dlg.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.ui.vlay_right.layout().addWidget(dlg)
        dlg.hide()
        uuid = data.tree_node.uuid if data.tree_node else io_util.uuid_from_path(data.filename)
        self.dialogs[uuid] = dlg
        return True

    def current_package_uuid(self):
        """Returns the current package."""
        current_item = self.ui.tree_packages.currentItem()
        if not current_item:
            return None
        return current_item.data(0, Qt.UserRole)

    def accept(self):
        """OK button clicked."""
        for dialog in self.dialogs.values():
            dialog._save_geom = False
            dialog.accept()
        super().accept()

    def reject(self):
        """Cancel button clicked."""
        for dialog in self.dialogs.values():
            dialog._save_geom = False
            dialog.reject()
        super().reject()


def create_tree_item(base: BaseFileData, locked: bool) -> QTreeWidgetItem:
    """Creates a QTreeWidgetItem for the tree.

    Args:
        base: The package BaseFileData.
        locked: True if the package is locked.

    Returns:
        The QTreeWidgetItem.
    """
    name = base.tree_node.name if base.tree_node else base.pname  # tree_node is None in tests
    item = QTreeWidgetItem([name])
    uuid = base.tree_node.uuid if base.tree_node else io_util.uuid_from_path(base.filename)
    item.setData(0, Qt.UserRole, uuid)
    base_icon_path = gui_util.get_icon_path(base.tree_node)
    if locked:
        lock_icon_path = resources_util.get_resource_path(':/resources/icons/dmi_component_lock.svg')
        icon = QIcon(MergedSvgIconEngine(base_icon_path, lock_icon_path))
    else:
        icon = QIcon(gui_util.get_icon_path(base.tree_node))
    item.setIcon(0, icon)
    return item


def _create_dlg_input(dlg_input: DialogInput, data: type[BaseFileData], locked: bool) -> DialogInput:
    """Create and return DialogInput.

    Args:
        dlg_input: Information needed by the dialog.
        data: The data class.
        locked: True if the dialog is to be locked.

    Returns:
        See description.
    """
    return DialogInput(
        data=data,
        locked=locked,
        selected_cells=dlg_input.selected_cells,
        filter_on_selected_cells=dlg_input.filter_on_selected_cells,
        ugrid_uuid=dlg_input.ugrid_uuid,
        query=dlg_input.query,
    )
