"""ImsDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QLabel, QLineEdit, QPushButton, QSpinBox, QTableWidget, QVBoxLayout
)
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.mf6.gui import gui_util, table_gui
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.misc import util


class ImsDialog(PackageDialogBase):
    """The IMS package dialog."""

    # Constants

    # These must correspond EXACTLY with the variables in the ImsData class (we use getattr)

    # Nonlinear
    OUTER_DVCLOSE = 'OUTER_DVCLOSE'
    OUTER_MAXIMUM = 'OUTER_MAXIMUM'
    UNDER_RELAXATION = 'UNDER_RELAXATION'
    UNDER_RELAXATION_THETA = 'UNDER_RELAXATION_THETA'
    UNDER_RELAXATION_KAPPA = 'UNDER_RELAXATION_KAPPA'
    UNDER_RELAXATION_GAMMA = 'UNDER_RELAXATION_GAMMA'
    UNDER_RELAXATION_MOMENTUM = 'UNDER_RELAXATION_MOMENTUM'
    BACKTRACKING_NUMBER = 'BACKTRACKING_NUMBER'
    BACKTRACKING_TOLERANCE = 'BACKTRACKING_TOLERANCE'
    BACKTRACKING_REDUCTION_FACTOR = 'BACKTRACKING_REDUCTION_FACTOR'
    BACKTRACKING_RESIDUAL_LIMIT = 'BACKTRACKING_RESIDUAL_LIMIT'
    LINEAR_SOLVER = 'LINEAR_SOLVER'  # 'SAMG'

    # Linear
    INNER_MAXIMUM = 'INNER_MAXIMUM'
    INNER_DVCLOSE = 'INNER_DVCLOSE'
    INNER_RCLOSE = 'INNER_RCLOSE'
    RCLOSE_OPTION = 'RCLOSE_OPTION'
    LINEAR_ACCELERATION = 'LINEAR_ACCELERATION'
    RELAXATION_FACTOR = 'RELAXATION_FACTOR'
    PRECONDITIONER_LEVELS = 'PRECONDITIONER_LEVELS'
    PRECONDITIONER_DROP_TOLERANCE = 'PRECONDITIONER_DROP_TOLERANCE'
    NUMBER_ORTHOGONALIZATIONS = 'NUMBER_ORTHOGONALIZATIONS'
    SCALING_METHOD = 'SCALING_METHOD'
    REORDERING_METHOD = 'REORDERING_METHOD'

    # Columns
    COLUMN_COUNT = 3
    CHECKBOX_COLUMN = 0  # 'Use' column
    LABEL_COLUMN = 1
    VALUE_COLUMN = 2

    def __init__(self, dlg_input, parent=None):
        """Initializes the dialog.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.setup_ui()

    @override
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'NONLINEAR', 'LINEAR']
        self.default_sections = ['NONLINEAR', 'LINEAR']

    @override
    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'NONLINEAR':
            self.setup_nonlinear_section()
        elif section_name == 'LINEAR':
            self.setup_linear_section()
        else:
            super().setup_section(section_name)

    @override
    def setup_options(self, vlayout: QVBoxLayout) -> None:
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout: The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)
        self._add_button_set_parameters_to_complexity_defaults(vlayout)

    def _add_button_set_parameters_to_complexity_defaults(self, vlayout):
        """Adds a button so user can manually reset all parameters to match the defaults for the complexity.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        btn = QPushButton('Set NONLINEAR and LINEAR Parameters to Complexity Defaults')
        pos = vlayout.indexOf(self.options_gui.uix['widget_csv_outer_output fileout'])
        vlayout.insertWidget(pos, btn)
        btn.clicked.connect(self.on_update_parameters_to_match_complexity)

        # Disable/enable button with the complexity checkbox
        chk = self.options_gui.uix['chk_complexity']
        enabling_lambda = lambda state, arg1=chk, arg2=btn: arg2.setEnabled(arg1.isChecked())  # noqa E731
        chk.stateChanged.connect(enabling_lambda)
        enabling_lambda(chk.isChecked())

    @override
    def do_enabling(self):
        """Enables/disables the table widgets."""
        super().do_enabling()
        tw = self.uix.get('tw_nonlinear')
        if tw:
            self.enable_widget(self.OUTER_DVCLOSE, tw)
            self.enable_widget(self.OUTER_MAXIMUM, tw)
            self.enable_widget(self.UNDER_RELAXATION, tw)
            self.enable_widget(self.UNDER_RELAXATION_THETA, tw)
            self.enable_widget(self.UNDER_RELAXATION_KAPPA, tw)
            self.enable_widget(self.UNDER_RELAXATION_MOMENTUM, tw)
            self.enable_widget(self.UNDER_RELAXATION_GAMMA, tw)
            self.enable_widget(self.BACKTRACKING_NUMBER, tw)
            self.enable_widget(self.BACKTRACKING_TOLERANCE, tw)
            self.enable_widget(self.BACKTRACKING_REDUCTION_FACTOR, tw)
            self.enable_widget(self.BACKTRACKING_RESIDUAL_LIMIT, tw)
            self.enable_widget(self.LINEAR_SOLVER, tw)

        tw = self.uix.get('tw_linear')
        if tw:
            self.enable_widget(self.INNER_MAXIMUM, tw)
            self.enable_widget(self.INNER_DVCLOSE, tw)
            self.enable_widget(self.INNER_RCLOSE, tw)
            self.enable_widget(self.RCLOSE_OPTION, tw, self.get_use(self.INNER_RCLOSE, tw))
            self.enable_widget(self.LINEAR_ACCELERATION, tw)
            self.enable_widget(self.RELAXATION_FACTOR, tw)
            self.enable_widget(self.PRECONDITIONER_LEVELS, tw)
            self.enable_widget(self.PRECONDITIONER_DROP_TOLERANCE, tw)
            self.enable_widget(self.NUMBER_ORTHOGONALIZATIONS, tw)
            self.enable_widget(self.SCALING_METHOD, tw)
            self.enable_widget(self.REORDERING_METHOD, tw)

    def setup_nonlinear_section(self):
        """Sets up the NONLINEAR section."""
        section = 'NONLINEAR'
        self.add_group_box_to_scroll_area(section)

        tw = self.uix['tw_nonlinear'] = QTableWidget()
        tw.setColumnCount(ImsDialog.COLUMN_COUNT)
        tw.setHorizontalHeaderLabels(['Use', 'Option', 'Value'])
        tw.verticalHeader().hide()
        self.uix[section]['layout'].addWidget(tw)
        self.uix[section]['layout'].addWidget(
            QLabel('(LINEAR_SOLVER SAMG requires "SAMG-Modflow" or "SAMG-MODFLOW+'
                   ' Add-on" license)')
        )

        # Nonlinear
        self.add_double_edit_row(self.OUTER_DVCLOSE, 0.01, tw)
        self.add_int_spin_row(self.OUTER_MAXIMUM, 1, tw)
        self.add_combo_box_row(self.UNDER_RELAXATION, ['NONE', 'SIMPLE', 'COOLEY', 'DBD'], 0, tw)
        self.add_double_edit_row(self.UNDER_RELAXATION_THETA, 0.7, tw)
        self.add_double_edit_row(self.UNDER_RELAXATION_KAPPA, 0.1, tw)
        self.add_double_edit_row(self.UNDER_RELAXATION_GAMMA, 0.2, tw)
        self.add_double_edit_row(self.UNDER_RELAXATION_MOMENTUM, 0.001, tw)
        self.add_int_spin_row(self.BACKTRACKING_NUMBER, 10, tw)
        self.add_double_edit_row(self.BACKTRACKING_TOLERANCE, 1e4, tw)
        self.add_double_edit_row(self.BACKTRACKING_REDUCTION_FACTOR, 0.2, tw)
        self.add_double_edit_row(self.BACKTRACKING_RESIDUAL_LIMIT, 100.0, tw)
        table_gui.add_text_edit_row(
            self.LINEAR_SOLVER, 'SAMG', tw, self.dlg_input.locked, checkbox_slot=self.on_checkbox
        )

        self.data_to_table_nonlinear()

        tw.horizontalHeader().setStretchLastSection(True)
        tw.resizeColumnToContents(ImsDialog.CHECKBOX_COLUMN)
        tw.resizeColumnToContents(ImsDialog.LABEL_COLUMN)
        tw.resizeRowsToContents()
        table_gui.disable_column(tw, ImsDialog.LABEL_COLUMN)

    def setup_linear_section(self):
        """Sets up the LINEAR section."""
        section = 'LINEAR'
        self.add_group_box_to_scroll_area(section)

        tw = self.uix['tw_linear'] = QTableWidget()
        tw.setColumnCount(ImsDialog.COLUMN_COUNT)
        tw.setHorizontalHeaderLabels(['Use', 'Option', 'Value'])
        tw.verticalHeader().hide()
        self.uix[section]['layout'].addWidget(tw)

        # Linear
        self.add_int_spin_row(self.INNER_MAXIMUM, 100, tw)
        self.add_double_edit_row(self.INNER_DVCLOSE, 0.001, tw)
        self.add_double_edit_row(self.INNER_RCLOSE, 1.0e-1, tw)
        self.add_combo_box_row(self.RCLOSE_OPTION, ['STRICT', 'L2NORM_RCLOSE', 'RELATIVE_RCLOSE'], 0, tw)
        self.add_combo_box_row(self.LINEAR_ACCELERATION, ['CG', 'BICGSTAB'], 0, tw)
        self.add_double_edit_row(self.RELAXATION_FACTOR, 0.0, tw)
        self.add_int_spin_row(self.PRECONDITIONER_LEVELS, 7, tw)
        self.add_double_edit_row(self.PRECONDITIONER_DROP_TOLERANCE, 1.0e-4, tw)
        self.add_int_spin_row(self.NUMBER_ORTHOGONALIZATIONS, 0, tw)
        self.add_combo_box_row(self.SCALING_METHOD, ['NONE', 'DIAGONAL', 'L2NORM'], 0, tw)
        self.add_combo_box_row(self.REORDERING_METHOD, ['NONE', 'RCM', 'MD'], 0, tw)

        self.data_to_table_linear()

        tw.horizontalHeader().setStretchLastSection(True)
        tw.resizeColumnToContents(ImsDialog.CHECKBOX_COLUMN)
        tw.resizeColumnToContents(ImsDialog.LABEL_COLUMN)
        tw.resizeRowsToContents()
        table_gui.disable_column(tw, ImsDialog.LABEL_COLUMN)

    def get_widget(self, option, table_widget):
        """Returns the widget associated with the option string.

        Args:
            option (str): The string associated with the option.
            table_widget (QTableWidget): The table.

        Returns:
            widget (Something derived from QWidget): The widget.
        """
        items = table_widget.findItems(option, Qt.MatchExactly)
        if items:
            widget = table_widget.cellWidget(items[0].row(), ImsDialog.VALUE_COLUMN)
            if widget:
                return widget
        return None

    def set_value(self, option, value, table_widget):
        """Sets the value for the widget associated with the option string.

        Args:
            option (str): The string associated with the option.
            value (?): The value (could be multiple types).
            table_widget (QTableWidget): The table.
        """
        widget = self.get_widget(option, table_widget)
        if type(widget) is QComboBox:
            widget.setCurrentText(value)
        elif type(widget) is QLineEdit:
            widget.setText(str(value))
        elif type(widget) is QSpinBox:
            widget.setValue(value)
        elif type(widget) is QCheckBox:
            widget.setChecked(value)
        else:
            raise AssertionError()

        # Set the "Use" column
        checkbox_widget = self.get_checkbox_widget(option, table_widget)
        use = getattr(self.dlg_input.data, f'{option.lower()}_use')
        with gui_util.SignalBlocker(checkbox_widget):  # Don't signal during setup. do_enabling is called later.
            checkbox_widget.setCheckState(Qt.Checked if use else Qt.Unchecked)
        return widget

    def get_checkbox_widget(self, option, table_widget):
        """Returns the widget item in the Use column.

        Args:
            option (str): The string associated with the option.
            table_widget (QTableWidget): The table.

        Returns:
            (QCheckBox): The checkbox.
        """
        check_box = None
        items = table_widget.findItems(option, Qt.MatchExactly)
        if items:
            widget = table_widget.cellWidget(items[0].row(), ImsDialog.CHECKBOX_COLUMN)
            check_box = widget.layout().itemAt(0).widget()
        return check_box

    def get_use(self, option, table_widget):
        """Returns True or False for whether the "Use" column is checked.

        Args:
            option (str): The string associated with the option.
            table_widget (QTableWidget): The table.

        Returns:
            (bool): True if checkbox is checked, else False.
        """
        widget_item = self.get_checkbox_widget(option, table_widget)
        return widget_item.checkState() == Qt.Checked

    def get_value(self, option, table_widget):
        """Returns the value for the widget associated with the option string.

        Args:
            option (str): The string associated with the option.
            table_widget (QTableWidget): The table.

        Returns:
            value (?): The value (could be multiple types).
        """
        widget = self.get_widget(option, table_widget)
        if type(widget) is QComboBox:
            return widget.currentText()
        elif type(widget) is QLineEdit:
            return widget.text()
        elif type(widget) is QSpinBox:
            return str(widget.value())
        elif type(widget) is QCheckBox:
            return widget.isChecked()
        else:
            raise AssertionError()

    def data_to_table_nonlinear(self):
        """Puts the data values in self.dlg_input.data into the widgets."""
        # Nonlinear
        tw = self.uix['tw_nonlinear']
        self.set_value(self.OUTER_DVCLOSE, self.dlg_input.data.outer_dvclose, tw)
        self.set_value(self.OUTER_MAXIMUM, self.dlg_input.data.outer_maximum, tw)
        self.set_value(self.UNDER_RELAXATION, self.dlg_input.data.under_relaxation, tw)
        self.set_value(self.UNDER_RELAXATION_THETA, self.dlg_input.data.under_relaxation_theta, tw)
        self.set_value(self.UNDER_RELAXATION_KAPPA, self.dlg_input.data.under_relaxation_kappa, tw)
        self.set_value(self.UNDER_RELAXATION_GAMMA, self.dlg_input.data.under_relaxation_gamma, tw)
        self.set_value(self.UNDER_RELAXATION_MOMENTUM, self.dlg_input.data.under_relaxation_momentum, tw)
        self.set_value(self.BACKTRACKING_NUMBER, self.dlg_input.data.backtracking_number, tw)
        self.set_value(self.BACKTRACKING_TOLERANCE, self.dlg_input.data.backtracking_tolerance, tw)
        self.set_value(self.BACKTRACKING_REDUCTION_FACTOR, self.dlg_input.data.backtracking_reduction_factor, tw)
        self.set_value(self.BACKTRACKING_RESIDUAL_LIMIT, self.dlg_input.data.backtracking_residual_limit, tw)
        self.set_value(self.LINEAR_SOLVER, self.dlg_input.data.linear_solver, tw)

    def data_to_table_linear(self):
        """Puts the data values in self.dlg_input.data into the widgets."""
        # Linear
        tw = self.uix['tw_linear']
        self.set_value(self.INNER_MAXIMUM, self.dlg_input.data.inner_maximum, tw)
        self.set_value(self.INNER_DVCLOSE, self.dlg_input.data.inner_dvclose, tw)
        self.set_value(self.INNER_RCLOSE, self.dlg_input.data.inner_rclose, tw)
        self.set_value(self.RCLOSE_OPTION, self.dlg_input.data.rclose_option, tw)
        self.set_value(self.LINEAR_ACCELERATION, self.dlg_input.data.linear_acceleration, tw)
        self.set_value(self.RELAXATION_FACTOR, self.dlg_input.data.relaxation_factor, tw)
        self.set_value(self.PRECONDITIONER_LEVELS, self.dlg_input.data.preconditioner_levels, tw)
        self.set_value(self.PRECONDITIONER_DROP_TOLERANCE, self.dlg_input.data.preconditioner_drop_tolerance, tw)
        self.set_value(self.NUMBER_ORTHOGONALIZATIONS, self.dlg_input.data.number_orthogonalizations, tw)
        self.set_value(self.SCALING_METHOD, self.dlg_input.data.scaling_method, tw)
        self.set_value(self.REORDERING_METHOD, self.dlg_input.data.reordering_method, tw)

    def table_to_data_nonlinear(self):
        """Copies the data values in the widgets into self.dlg_input.data."""
        # Nonlinear
        tw = self.uix['tw_nonlinear']
        self.dlg_input.data.outer_dvclose = util.to_float(self.get_value(self.OUTER_DVCLOSE, tw))
        self.dlg_input.data.outer_dvclose_use = self.get_use(self.OUTER_DVCLOSE, tw)
        self.dlg_input.data.outer_maximum = util.to_int(self.get_value(self.OUTER_MAXIMUM, tw))
        self.dlg_input.data.outer_maximum_use = self.get_use(self.OUTER_MAXIMUM, tw)
        self.dlg_input.data.under_relaxation = self.get_value(self.UNDER_RELAXATION, tw)
        self.dlg_input.data.under_relaxation_use = self.get_use(self.UNDER_RELAXATION, tw)
        self.dlg_input.data.under_relaxation_theta = util.to_float(self.get_value(self.UNDER_RELAXATION_THETA, tw))
        self.dlg_input.data.under_relaxation_theta_use = self.get_use(self.UNDER_RELAXATION_THETA, tw)
        self.dlg_input.data.under_relaxation_kappa = util.to_float(self.get_value(self.UNDER_RELAXATION_KAPPA, tw))
        self.dlg_input.data.under_relaxation_kappa_use = self.get_use(self.UNDER_RELAXATION_KAPPA, tw)
        self.dlg_input.data.under_relaxation_gamma = util.to_float(self.get_value(self.UNDER_RELAXATION_GAMMA, tw))
        self.dlg_input.data.under_relaxation_gamma_use = self.get_use(self.UNDER_RELAXATION_GAMMA, tw)
        self.dlg_input.data.under_relaxation_momentum = util.to_float(
            self.get_value(self.UNDER_RELAXATION_MOMENTUM, tw)
        )
        self.dlg_input.data.under_relaxation_momentum_use = self.get_use(self.UNDER_RELAXATION_MOMENTUM, tw)
        self.dlg_input.data.backtracking_number = util.to_int(self.get_value(self.BACKTRACKING_NUMBER, tw))
        self.dlg_input.data.backtracking_number_use = self.get_use(self.BACKTRACKING_NUMBER, tw)
        self.dlg_input.data.backtracking_tolerance = util.to_float(self.get_value(self.BACKTRACKING_TOLERANCE, tw))
        self.dlg_input.data.backtracking_tolerance_use = self.get_use(self.BACKTRACKING_TOLERANCE, tw)
        self.dlg_input.data.backtracking_reduction_factor = util.to_float(
            self.get_value(self.BACKTRACKING_REDUCTION_FACTOR, tw)
        )
        self.dlg_input.data.backtracking_reduction_factor_use = self.get_use(self.BACKTRACKING_REDUCTION_FACTOR, tw)
        self.dlg_input.data.backtracking_residual_limit = util.to_float(
            self.get_value(self.BACKTRACKING_RESIDUAL_LIMIT, tw)
        )
        self.dlg_input.data.backtracking_residual_limit_use = self.get_use(self.BACKTRACKING_RESIDUAL_LIMIT, tw)
        self.dlg_input.data.linear_solver = self.get_value(self.LINEAR_SOLVER, tw)
        self.dlg_input.data.linear_solver_use = self.get_use(self.LINEAR_SOLVER, tw)

    def table_to_data_linear(self):
        """Copies the data values in the widgets into self.dlg_input.data."""
        # Linear
        tw = self.uix['tw_linear']
        self.dlg_input.data.inner_maximum = util.to_int(self.get_value(self.INNER_MAXIMUM, tw))
        self.dlg_input.data.inner_maximum_use = self.get_use(self.INNER_MAXIMUM, tw)
        self.dlg_input.data.inner_dvclose = util.to_float(self.get_value(self.INNER_DVCLOSE, tw))
        self.dlg_input.data.inner_dvclose_use = self.get_use(self.INNER_DVCLOSE, tw)
        self.dlg_input.data.inner_rclose = util.to_float(self.get_value(self.INNER_RCLOSE, tw))
        self.dlg_input.data.inner_rclose_use = self.get_use(self.INNER_RCLOSE, tw)
        self.dlg_input.data.rclose_option = self.get_value(self.RCLOSE_OPTION, tw)
        self.dlg_input.data.rclose_option_use = self.get_use(self.RCLOSE_OPTION, tw)
        self.dlg_input.data.linear_acceleration = self.get_value(self.LINEAR_ACCELERATION, tw)
        self.dlg_input.data.linear_acceleration_use = self.get_use(self.LINEAR_ACCELERATION, tw)
        self.dlg_input.data.relaxation_factor = util.to_float(self.get_value(self.RELAXATION_FACTOR, tw))
        self.dlg_input.data.relaxation_factor_use = self.get_use(self.RELAXATION_FACTOR, tw)
        self.dlg_input.data.preconditioner_levels = util.to_int(self.get_value(self.PRECONDITIONER_LEVELS, tw))
        self.dlg_input.data.preconditioner_levels_use = self.get_use(self.PRECONDITIONER_LEVELS, tw)
        self.dlg_input.data.preconditioner_drop_tolerance = util.to_float(
            self.get_value(self.PRECONDITIONER_DROP_TOLERANCE, tw)
        )
        self.dlg_input.data.preconditioner_drop_tolerance_use = self.get_use(self.PRECONDITIONER_DROP_TOLERANCE, tw)
        self.dlg_input.data.number_orthogonalizations = util.to_int(self.get_value(self.NUMBER_ORTHOGONALIZATIONS, tw))
        self.dlg_input.data.number_orthogonalizations_use = self.get_use(self.NUMBER_ORTHOGONALIZATIONS, tw)
        self.dlg_input.data.scaling_method = self.get_value(self.SCALING_METHOD, tw)
        self.dlg_input.data.scaling_method_use = self.get_use(self.SCALING_METHOD, tw)
        self.dlg_input.data.reordering_method = self.get_value(self.REORDERING_METHOD, tw)
        self.dlg_input.data.reordering_method_use = self.get_use(self.REORDERING_METHOD, tw)

    def enable_widget(self, option, table_widget, condition=True):
        """Enables or disables the widget based on the condition, use checkbox, and dialog locked state.

        Also looks at "Use" column

        Args:
            option (str): The string associated with the option.
            table_widget (QTableWidget): The table.
            condition (bool): Some condition.
        """
        use = self.get_use(option, table_widget)
        self.get_widget(option, table_widget).setEnabled(condition and use and not self.dlg_input.locked)

    def on_checkbox(self):
        """Called when a checkbox state changes."""
        self.do_enabling()

    def on_update_parameters_to_match_complexity(self):
        """Called with the button is clicked."""
        complexity = self.options_gui.uix['cbx_complexity'].currentText()
        message = (
            f'This will set all NONLINEAR and LINEAR parameters to their default values corresponding'
            f' to the current complexity ({complexity}). Continue?'
        )
        rv = message_box.message_with_ok_cancel(parent=self, message=message, icon='Warning')
        if not rv:
            return
        self.on_complexity(self.options_gui.uix['cbx_complexity'].currentIndex())

    def on_complexity(self, index):
        """Called when the complexity combo box is changed.

        Args:
            index (int): The new current index of the combo box.
        """
        if not self.loaded:
            return

        self.dlg_input.data.set_complexity(index)
        self.loaded = False  # prevent infinite loop from setting the combo box value in data_to_table
        self.data_to_table_nonlinear()
        self.data_to_table_linear()
        self.loaded = True
        self.do_enabling()

    def add_double_edit_row(self, label, default, table_widget):
        """Adds a row with a double spin control.

        Args:
            label: The label.
            default: The default value.
            table_widget: The table widget.
        """
        table_gui.add_double_edit_row(
            label, default, table_widget, self.dlg_input.locked, checkbox_slot=self.on_checkbox
        )

    def add_int_spin_row(self, label, default, table_widget):
        """Adds a row with an integer spin control.

        Args:
            label: The label.
            default: The default value.
            table_widget: The table widget.
        """
        table_gui.add_int_spin_row(
            label=label,
            default=default,
            minimum=-100000,
            maximum=100000,
            table_widget=table_widget,
            locked=self.dlg_input.locked,
            checkbox_slot=self.on_checkbox
        )

    def add_combo_box_row(self, label, options_list, default, table_widget):
        """Adds a row to the table containing a combo box.

        Args:
            label (str): The label string, or label next to the widget.
            options_list (list of str): Strings to put in the combo box.
            default (int): Index of the default choice to show.
            table_widget (QTableWidget): The table.
        """
        table_gui.add_combo_box_row(
            label=label,
            options_list=options_list,
            default=default,
            table_widget=table_widget,
            locked=self.dlg_input.locked,
            checkbox_slot=self.on_checkbox
        )

    def get_data(self):
        """Returns the self.dlg_input.data object.

        Returns:
            See description.
        """
        return self.dlg_input.data

    @override
    def widgets_to_data(self) -> None:
        """Get info from widgets and store it in dlg_input.data."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.table_to_data_nonlinear()
            self.table_to_data_linear()
