"""OptionsGui class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util, options_util
from xms.mf6.gui.options_defs import (
    Checkbox, Checkbox3Fields, CheckboxButton, CheckboxCheckbox, CheckboxComboBox, CheckboxField, CheckboxFieldButton,
    CheckboxPrintFormat, LineEdit
)


def enable_widgets(chk, widgets):
    """Enables/disables widgets based on the checked state of the checkbox.

    This is a free function because of the second answer on https://stackoverflow.com/questions/35819538/

    Args:
        chk (QCheckBox): A checkbox.
        widgets (list or tuple): The widgets.
    """
    for widget in widgets:
        widget.setEnabled(chk.isChecked())


class OptionsGui:
    """Class to handle the GUI of the OPTIONS block."""
    def __init__(self, parent):
        """Initializes the class.

        Args:
            parent: The parent dialog.
        """
        self.parent = parent
        self.options_block = parent.dlg_input.data.options_block if parent else None
        self.uix = {}  # Dict that stores the options widgets

    def setup(self, vlayout):
        """Adds all the widgets for the options to the layout.

        For info on the lambdas, see
         https://stackoverflow.com/questions/35819538/using-lambda-expression-to-connect-slots-in-pyqt

        Args:
            vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        """
        for def_ in self.options_block.definitions:  # 'def_' because 'definition' is long and 'def' is a keyword
            chk = None
            def_type = type(def_)
            enabling_lambda = None

            if def_type == Checkbox:
                chk = options_util.checkbox_or_label(
                    key=def_.option, brief=def_.brief, widget_dict=self.uix, vlayout=vlayout
                )
                self._connect_check_box_method(chk, def_.check_box_method)
            elif def_type == CheckboxButton:
                _, chk, btn = options_util.checkbox_button(
                    key=def_.option,
                    brief=def_.brief,
                    button_text=def_.button_text,
                    widget_dict=self.uix,
                    vlayout=vlayout
                )
                # Create a lambda to enable/disable the button
                enabling_lambda = lambda state, arg1=chk, arg2=btn: arg2.setEnabled(arg1.isChecked())  # noqa E731
                chk.stateChanged.connect(enabling_lambda)

                self._connect_check_box_method(chk, def_.check_box_method)
                self._connect_button_method(btn, def_.button_method)
            elif def_type == CheckboxComboBox:
                _, chk, cbx = options_util.checkbox_combobox(
                    key=def_.option, brief=def_.brief, widget_dict=self.uix, items=def_.items, vlayout=vlayout
                )
                # Create a lambda to enable/disable the combobox
                enabling_lambda = lambda state, arg1=chk, arg2=cbx: arg2.setEnabled(arg1.isChecked())  # noqa E731
                chk.stateChanged.connect(enabling_lambda)

                # Initialize the combobox
                default = def_.value if def_.value else def_.items[0]
                value = self.options_block.get(def_.option, default).upper()
                cbx.setCurrentText(value)

                self._connect_check_box_method(chk, def_.check_box_method)
                self._connect_combo_box_method(cbx, def_.combo_box_method)
            elif def_type == CheckboxCheckbox:
                chk, chk2 = options_util.checkbox_checkbox(
                    key=def_.option,
                    brief=def_.brief,
                    key2=def_.option2,
                    brief2=def_.brief2,
                    widget_dict=self.uix,
                    vlayout=vlayout
                )
                # Create a lambda to enable/disable the combobox
                enabling_lambda = lambda state, arg1=chk, arg2=chk2: arg2.setEnabled(arg1.isChecked())  # noqa E731
                chk.stateChanged.connect(enabling_lambda)

                # Initialize the second checkbox
                value = self.options_block.get(def_.option)
                chk2.setChecked(bool(value))
            elif def_type == CheckboxField:
                value = self.options_block.get(def_.option, def_.value)
                _, chk, field = options_util.checkbox_field(
                    key=def_.option,
                    brief=def_.brief,
                    widget_dict=self.uix,
                    type_=def_.type_,
                    value=value,
                    vlayout=vlayout,
                    add_stretch=True,
                    read_only=def_.read_only,
                    minimum=def_.minimum,
                    maximum=def_.maximum
                )
                # Create a lambda to enable/disable the field
                enabling_lambda = lambda state, arg1=chk, arg2=field: arg2.setEnabled(arg1.isChecked())  # noqa E731
                chk.stateChanged.connect(enabling_lambda)
            elif def_type == LineEdit:
                value = self.options_block.get(def_.option, def_.value)
                _, label, field = options_util.line_edit(
                    key=def_.option,
                    brief=def_.brief,
                    widget_dict=self.uix,
                    type_=def_.type_,
                    value=value,
                    label=False,
                    vlayout=vlayout,
                    add_stretch=True,
                    read_only=def_.read_only
                )
            elif def_type == Checkbox3Fields:
                value = self.options_block.get(def_.option, '')
                if value:
                    # It's all one string. Split it up and extract the values for each field
                    words = value.split()
                    field_values = [words[1], words[3], words[5]]
                else:
                    field_values = [def_.value1, def_.value2, def_.value3]
                chk, w1, w2, w3 = options_util.checkbox_3_fields(
                    checkbox_key=def_.option,
                    key1=def_.option1,
                    type1=def_.type1,
                    value1=field_values[0],
                    key2=def_.option2,
                    type2=def_.type2,
                    value2=field_values[1],
                    key3=def_.option3,
                    type3=def_.type3,
                    value3=field_values[2],
                    widget_dict=self.uix,
                    vlayout=vlayout
                )
                # Create lambda to enable/disable the fields
                enabling_lambda = lambda state, a1=chk, wlist=(w1, w2, w3): enable_widgets(a1, wlist)  # noqa E731
                chk.stateChanged.connect(enabling_lambda)
            elif def_type == CheckboxFieldButton:
                value = self.options_block.get(def_.option, def_.value)
                _, chk, field, btn = options_util.setup_checkbox_line_edit_button(
                    key=def_.option,
                    brief=def_.brief,
                    type_=def_.type_,
                    value=value,
                    button_text=def_.button_text,
                    widget_dict=self.uix,
                    vlayout=vlayout
                )
                # Create a lambda to enable/disable the field
                enabling_lambda = lambda state, arg1=chk, wlist=(field, btn): enable_widgets(arg1, wlist)  # noqa E731
                chk.stateChanged.connect(enabling_lambda)

                # Connect signals/slots
                self._connect_check_box_method(chk, def_.check_box_method)
                self._connect_field_method(field, def_.field_method)
                self._connect_button_method(btn, def_.button_method)
            elif def_type == CheckboxPrintFormat:
                first_word = def_.option.split()[0]
                # value = self.options_block.get(f'{first_word} PRINT_FORMAT')
                chk, cbx, w1, w2, w3 = options_util.setup_print_format(
                    first_word=first_word, widget_dict=self.uix, vlayout=vlayout
                )
                options_util.load_print_format_options(first_word, self.options_block, widget_dict=self.uix)

                # Create lambda to enable/disable the fields
                enabling_lambda = lambda state, a1=chk, wlist=(cbx, w1, w2, w3): enable_widgets(a1, wlist)  # noqa E731
                chk.stateChanged.connect(enabling_lambda)

            # Init checked state but don't emit signal because options are done first and nothing else exists yet
            with gui_util.SignalBlocker(chk):  # Don't signal during setup. do_enabling is called later.
                chk.setChecked(def_.option in self.options_block.options)
            if enabling_lambda:  # Call the enabling lambda now so things are enabled/disabled appropriately
                enabling_lambda(chk.isChecked())

    def _get_method(self, method_name):
        """Returns the method of the dialog corresponding to method_name, or None.

        Args:
            method_name (str): Name of the method.

        Returns:
            The method or None.
        """
        method = None
        if method_name:
            if hasattr(self.parent, method_name) and callable(getattr(self.parent, method_name)):
                method = getattr(self.parent, method_name)
            else:
                raise TypeError(f'Error: {self.parent} does not define method {method_name}.')
        return method

    def _connect_check_box_method(self, check_box, method_name):
        """Connects the method to the check box stateChanged signal.

        Args:
            check_box (QCheckBox): The check box.
            method_name (str): Name of the method of the dialog to call.
        """
        if method_name:
            method = self._get_method(method_name)
            check_box.stateChanged.connect(method)

    def _connect_combo_box_method(self, combo_box, method_name):
        """Connects the method to the combo box currentTextChanged signal.

        Args:
            combo_box (QComboBox): The combo box.
            method_name (str): Name of the method of the dialog to call.
        """
        if method_name:
            method = self._get_method(method_name)
            combo_box.currentTextChanged.connect(method)

    def _connect_field_method(self, field, method_name):
        """Connects the method to the field editingFinished signal.

        Args:
            field (QLineEdit or QSpinBox): The field
            method_name (str): Name of the method of the dialog to call.
        """
        if method_name:
            method = self._get_method(method_name)
            field.editingFinished.connect(method)

    def _connect_button_method(self, btn, method_name):
        """Connects the method to the button clicked signal.

        Args:
            btn (QPushButton): The button
            method_name (str): Name of the method of the dialog to call.
        """
        if method_name:
            method = self._get_method(method_name)
            btn.clicked.connect(method)

    def save(self):
        """Saves the widget states to the options_block."""
        options_block = OptionsBlock(None)
        for def_ in self.options_block.definitions:  # 'def_' because 'definition' is long and 'def' is a keyword
            def_type = type(def_)
            chk_name = options_util.make_widget_name('chk', def_.option)  # Name of widget in self.uix dict
            checked = self.uix[chk_name].isChecked()

            if def_type == Checkbox:
                options_block.set(def_.option, checked, None)
            elif def_type == CheckboxButton:
                if checked:
                    if def_.option == 'AUXILIARY' or any(word in def_.option for word in {'FILEIN', 'FILEOUT'}):
                        options_block.set(def_.option, checked, self.options_block.get(def_.option, []))
            elif def_type == CheckboxComboBox:
                cbx_name = options_util.make_widget_name('cbx', def_.option)  # Name of widget in self.uix dict
                options_block.set(def_.option, checked, self.uix[cbx_name].currentText())
            elif def_type == CheckboxCheckbox:
                chk2_name = options_util.make_widget_name('chk', def_.option2)  # Name of widget in self.uix dict
                value2 = def_.option2 if self.uix[chk2_name].isChecked() else ''
                options_block.set(def_.option, checked, value2)
            elif def_type == CheckboxField:
                value = self._get_field_value(def_.option, def_.type_)
                options_block.set(def_.option, checked, value)
            elif def_type == Checkbox3Fields:
                words = []
                self._get_field(def_.option1, def_.type1, words)
                self._get_field(def_.option2, def_.type2, words)
                self._get_field(def_.option3, def_.type3, words)
                value = ' '.join(words)
                options_block.set(def_.option, checked, value)
            elif def_type == CheckboxFieldButton:
                value = self._get_field_value(def_.option, def_.type_)
                options_block.set(def_.option, checked, value)
            elif def_type == CheckboxPrintFormat:
                first_word = def_.option.split()[0]
                options_util.save_print_format_options(first_word, options_block, self.uix)

        self.parent.dlg_input.data.options_block.options = copy.deepcopy(options_block.options)

    def _get_field_value(self, option, type_):
        if type_ == 'int':
            name = options_util.make_widget_name('spn', option)  # Name of widget in self.uix dict
            return self.uix[name].value()
        else:
            name = options_util.make_widget_name('edt', option)  # Name of widget in self.uix dict
            return self.uix[name].text()

    def _get_field(self, option, type_, words):
        value = self._get_field_value(option, type_)
        words.append(f'{option} {str(value)}')
