"""ListBlockTableWidget class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtCore import QItemSelection, QItemSelectionModel
from PySide2.QtWidgets import (QAbstractItemView, QApplication, QLabel, QPushButton, QToolBar, QVBoxLayout, QWidget)

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.gui import get_int_dialog, options_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.resources.svgs import COPY_SVG, PASTE_SVG, ROW_ADD_SVG, ROW_DELETE_SVG, ROW_INSERT_SVG


class ListBlockTableWidget(QWidget):
    """A widget showing data in a table, with right-click menus, and an 'Add Rows' button at the bottom.

    Uses QxTableView actually. May be part of a DataTabsWidget.

    Contains a lot of code that previously was repeated in several places. Doesn't replace all hand-coded tables,
    but replaces a lot.
    """
    def __init__(self, parent, dlg_input, label):
        """Initializes the class, sets up the ui.

        Args:
            parent (Something derived from QWidget): The parent window.
            dlg_input (DialogInput): Information needed by the dialog.
            label (str): Optional label that appears right above the table.
        """
        super().__init__(parent)

        self.dlg_input = dlg_input
        self.label = label
        self.models = {}  # Dictionary of classes derived from QAbstractTreeModel
        self.model = None
        self.loaded = False  # Flag to indicate if a package has been loaded
        self.current_model_name = ''
        self.data_changed = False

        self.txt_label = None
        self.table = None  # Becomes QxTableView
        self.hlay_widget_tools = None
        self.btn_add_rows = None
        self._toolbar = None
        self._actions = None

        self.setup_ui()
        self.setup_signals()

    def setup_ui(self):
        """Sets up the UI."""
        self.setUpdatesEnabled(False)  # To avoid signals (doesn't seem to work).

        vlayout = QVBoxLayout()
        self.setLayout(vlayout)

        # Table title
        if self.label:
            self.txt_label = QLabel(f'{self.label}:')
            vlayout.addWidget(self.txt_label)

        # Table
        self.table = gui_util.new_table_view()
        self.table.use_header_checkboxes = True
        self.table.allow_drag_fill = True
        self.table.setMinimumHeight(150)  # I just think this looks better
        vlayout.addWidget(self.table)

        self._add_tools_layout(vlayout)

        # Context menu
        if not self.dlg_input.locked:
            gui_util.set_vertical_header_menu_method(self.table, self.on_index_column_click)
            gui_util.set_table_menu_method(self.table, self.on_right_click)

        self.table.horizontalHeader().setStretchLastSection(True)

        self.loaded = True
        self.setUpdatesEnabled(True)  # To enable signals.
        self.do_enabling()

    def _add_tools_layout(self, vlayout: QVBoxLayout) -> None:
        """Create the layout below the table containing the Add Rows button and toolbar."""
        self.hlay_widget_tools = options_util.create_hlayout_widget('', {})

        # Add Rows button
        self.btn_add_rows = QPushButton('Add Rows...')
        self.hlay_widget_tools.layout().addWidget(self.btn_add_rows)
        self.hlay_widget_tools.layout().addStretch()
        vlayout.addWidget(self.hlay_widget_tools)
        self.btn_add_rows.setEnabled(not self.dlg_input.locked)

        # Toolbar
        self._create_toolbar(self.hlay_widget_tools)

    def _create_toolbar(self, layout_widget: QWidget):
        """Creates the toolbar."""
        self._toolbar = QToolBar()
        button_list = [
            [ROW_INSERT_SVG, 'Insert Row', self.on_insert_rows],
            [ROW_ADD_SVG, 'Add Row', self.on_add_rows],
            [ROW_DELETE_SVG, 'Delete Row', self.on_delete_rows],
        ]
        self._actions = widget_builder.setup_toolbar(self._toolbar, button_list)
        layout_widget.layout().addWidget(self._toolbar)

    def setup_signals(self):
        """Sets up any needed signals."""
        self.btn_add_rows.clicked.connect(self.on_btn_add_rows)

    def on_right_click(self, point):
        """Slot called when user right-clicks in the table.

        Args:
            point(QPoint): The point clicked.
        """
        menu_list = [
            [COPY_SVG, 'Copy', self.table.on_copy],
            [PASTE_SVG, 'Paste', self.table.on_paste],
            [ROW_INSERT_SVG, 'Insert row above', self.on_insert_rows],
            [ROW_ADD_SVG, 'Insert row below', self.on_add_rows],
            [ROW_DELETE_SVG, 'Delete row', self.on_delete_rows],
        ]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.table.viewport().mapToGlobal(point))

    def on_index_column_click(self, point):
        """Called on a right-click event in the index column (vertical header).

        Args:
            point (QPoint): The point clicked
        """
        # Select the row if it's not already selected
        row = self.table.verticalHeader().logicalIndexAt(point)
        selected_list = self.table.selectedIndexes()
        unique_rows = gui_util.get_unique_selected_rows(selected_list)
        if row not in unique_rows:
            self.table.selectRow(row)

        menu_list = [
            [COPY_SVG, 'Copy', self.table.on_copy],
            [PASTE_SVG, 'Paste', self.table.on_paste],
            [ROW_INSERT_SVG, 'Insert row above', self.on_insert_rows],
            [ROW_ADD_SVG, 'Insert row below', self.on_add_rows],
            [ROW_DELETE_SVG, 'Delete row', self.on_delete_rows],
        ]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.table.viewport().mapToGlobal(point))

    def on_selection_changed(self):
        """Called when the user clicks in the table and changes what cells are selected."""
        self.do_enabling()

    def on_btn_add_rows(self):
        """Adds rows to the table."""
        main_file = ''
        if self.dlg_input.data and self.dlg_input.data.filename:
            main_file = self.dlg_input.data.filename
        rv, value = get_int_dialog.run(
            'Rows To Add', 'Number of rows to add at bottom:', None, 1, 100000, self, main_file=main_file
        )
        if rv:
            self.add_rows(value)

    def add_rows(self, count):
        """Adds rows via the 'Add Rows...' button.

        Args:
            count (int): Number of rows to add.
        """
        model = self.models[self.current_model_name]
        model.insertRows(model.rowCount(), count)

        # Update selection
        idx = model.createIndex(model.rowCount() - 1, 0)
        self.table.selectionModel().select(idx, QItemSelectionModel.ClearAndSelect)
        QApplication.processEvents()
        self.table.scrollTo(idx, QAbstractItemView.EnsureVisible)

    def on_insert_rows(self):
        """Inserts rows above the selection, by clicking on the toolbar button."""
        rv, minrow, maxrow = gui_util.get_selected_rows_or_warn(self.table)
        if not rv:
            return

        model = self.models[self.current_model_name]
        model.insertRows(minrow, maxrow - minrow + 1)

        # Update selection
        self.table.selectionModel().clear()
        selection = QItemSelection(model.index(minrow, 0), model.index(maxrow, model.columnCount() - 1))
        self.table.selectionModel().select(selection, QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows)

    def on_add_rows(self):
        """Adds rows below the selection, by clicking on the toolbar button."""
        model = self.models[self.current_model_name]
        empty_table = (model.rowCount() == 0)
        if not empty_table:
            rv, minrow, maxrow = gui_util.get_selected_rows_or_warn(self.table)
            if not rv:
                model.insertRows(model.rowCount(), 1)
            else:
                model.insertRows(maxrow + 1, maxrow - minrow + 1)
        else:
            model.insertRows(model.rowCount(), 1)

            # Select the first cell if the table was previously empty
            idx = model.createIndex(0, 0)
            self.table.selectionModel().select(idx, QItemSelectionModel.ClearAndSelect)
            QApplication.processEvents()
            self.table.scrollTo(idx, QAbstractItemView.EnsureVisible)

    def on_delete_rows(self):
        """Deletes rows from the spreadsheet."""
        rv, minrow, maxrow = gui_util.get_selected_rows_or_warn(self.table)
        if not rv:
            return

        model = self.models[self.current_model_name]
        model.removeRows(minrow, maxrow - minrow + 1)

        # Select a new row
        row = minrow if model.rowCount() > minrow else minrow - 1
        self.table.selectRow(row)
        self.on_selection_changed()

    def do_enabling(self):
        """Enables and disables the widgets appropriately."""
        selection_list = self.table.selectedIndexes()
        selections_exist = len(selection_list) > 0
        not_locked = not self.dlg_input.locked

        self.btn_add_rows.setEnabled(not_locked)
        self._actions[ROW_INSERT_SVG].setEnabled(not_locked and selections_exist)
        self._actions[ROW_ADD_SVG].setEnabled(not_locked)
        self._actions[ROW_DELETE_SVG].setEnabled(not_locked and selections_exist)

    def on_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index.
        """
        del top_left_index, bottom_right_index  # Unused parameters
        self.data_changed = True

    def load_list_block(self, block):
        """Loads the list block into the table.

        Args:
            block (str): Name of the block.
        """
        self.current_model_name = block
        filename = self.dlg_input.data.list_blocks[block]
        column_names, column_types, _ = self.dlg_input.data.get_column_info(block)
        df = self.dlg_input.data.read_csv_file_into_dataframe(block, filename, column_names, column_types)

        # Filter on selected cells
        if hasattr(self.dlg_input.data, 'block_with_cellids'):
            if block == self.dlg_input.data.block_with_cellids and self.dlg_input.filter_on_selected_cells:
                df = gui_util.apply_filter_on_selected_cells(self.dlg_input, self.dlg_input.data.grid_info(), df)

        self.setup_model(df, block)
        self.add_delegates(block)
        self.table.horizontalHeader().setVisible(True)  # Not sure why we have to do this but we do

    def add_delegates(self, block):
        """Adds combo box delegates to the table.

        Args:
            block (str): Name of the block.
        """
        delegate_info = self.dlg_input.data.get_column_delegate_info(block)
        if delegate_info:
            delegate = QxCbxDelegate(self)
            for column in delegate_info:
                delegate.set_strings(column[1])
                self.table.setItemDelegateForColumn(column[0], delegate)

    def _set_model_defaults_and_read_only_columns(self, model, defaults):
        """Does what it says.

        Args:
            model: The model.
            defaults (dict[str, value]): Column names -> default values.
        """
        model.set_default_values(defaults)
        if self.dlg_input.locked:
            read_only_columns = set(range(model.columnCount()))
            model.set_read_only_columns(read_only_columns)

    def setup_model(self, df, array_name):
        """Creates a model and connects it to the table view and some signals.

        Args:
            df (pandas.DataFrame): The DataFrame.
            array_name (str): The name of the array ('RECHARGE' etc)

        Returns:
            (QxPandasTableModel): The model.
        """
        self.models[array_name] = QxPandasTableModel(df)
        names, types, defaults = self.dlg_input.data.get_column_info(array_name)
        self._set_model_defaults_and_read_only_columns(self.models[array_name], defaults)
        self.models[array_name].dataChanged.connect(self.on_data_changed)
        self.update_tool_tips(array_name)

        self.table.setModel(self.models[array_name])
        self.table.selectionModel().selectionChanged.connect(self.on_selection_changed)
        self.on_selection_changed()

        widget_builder.resize_columns_to_contents(self.table)
        return self.models[array_name]

    def setup_models(self):
        """Sets up self.models."""
        for block in self.dlg_input.data.list_blocks.keys():
            filename = self.dlg_input.data.list_blocks[block]
            names, types, defaults = self.dlg_input.data.get_column_info(block)
            data_frame = self.dlg_input.data.read_csv_file_into_dataframe(
                block=block, filename=filename, column_names=names, column_types=types
            )
            self.models[block] = QxPandasTableModel(data_frame)
            self.models[block].set_default_values(defaults)
            if self.dlg_input.locked:
                self.models[block].set_read_only_columns(set(range(self.models[block].columnCount())))

    def save_list_blocks_to_temp(self):
        """If changes have been made, saves changes to temporary files."""
        self.dlg_input.data.list_blocks.clear()
        for filename in self.models.keys():
            temp_filename = self.dlg_input.data.dataframe_to_temp_file(filename, self.models[filename].data_frame)
            if io_util.is_temporary_file(filename):
                os.remove(filename)
            self.dlg_input.data.list_blocks[filename] = temp_filename

    def load_block(self, block):
        """Loads the list block into the table.

        Args:
            block (str): Name of the block.
        """
        if not self.loaded:
            return

        if not block:
            self.do_enabling()
            return

        self.current_model_name = block
        self.table.setModel(self.models[block])
        self.table.horizontalHeader().setVisible(True)
        self.table.selectionModel().selectionChanged.connect(self.on_selection_changed)
        self.on_selection_changed()

        widget_builder.resize_columns_to_contents(self.table)

    def add_default_dataframe(self, new_name):
        """Creates a default dataframe with one row and adds it with a new model.

        Args:
            new_name (str): Model name.
        """
        names, types, defaults = self.dlg_input.data.get_column_info('')
        df = gui_util.empty_dataframe(names, list(types.values()))
        self.models[new_name] = QxPandasTableModel(df)
        self.models[new_name].set_default_values(defaults)
        self.models[new_name].insertRows(0, 1)

    def set_model(self, model, name='model'):
        """Sets the self.models dict at name to model.

        Args:
            model (QxPandasTableModel): The model.
            name (str): The name.
        """
        self.current_model_name = name
        self.models[name] = model
        self.table.setModel(self.models[name])
        widget_builder.resize_columns_to_contents(self.table)
        self.table.selectionModel().selectionChanged.connect(self.on_selection_changed)
        self.on_selection_changed()

        self.update_tool_tips()
        self.data_changed = True

    def update_tool_tips(self, model_name=''):
        """Updates the column header tool tips."""
        if not model_name:
            model_name = self.current_model_name
        if hasattr(self.dlg_input.data, 'get_column_tool_tips'):
            self.models[model_name].set_horizontal_header_tooltips(
                self.dlg_input.data.get_column_tool_tips(block=model_name)
            )

    def get_model(self, name='model'):
        """Returns the model at the given name.

        Args:
            name (str): The name.

        Returns:
            (QxPandasTableModel): The model.
        """
        return self.models[name]

    def delete_model(self, key):
        """Deletes a model.

        Args:
            key (str): Key into the self.models dict.
        """
        del self.models[key]

    def _columns_to_add_or_drop(self, columns, df):
        """Returns the set of columns we need to add and to drop.

        Args:
            columns (list[str]): Column names.
            df (DataFrame): The dataframe.

        Returns:
            (tuple(set[str], set[str]): See description.
        """
        columns_set = set(columns)
        df_columns_set = set(df.columns)
        to_drop = df_columns_set - columns_set
        to_add = columns_set - df_columns_set
        return to_add, to_drop

    def _change_columns(self, df, columns, to_add, to_drop, defaults):
        """Changes the dataframe to have the given columns.

        Args:
            df (DataFrame): The dataframe.
            columns (list[str]): Column names.
            to_add (set[str]): Columns we need to add.
            to_drop (set[str]): Columns we need to drop.
            defaults (dict[str, value]): Column names -> default values.

        Returns:
            The new dataframe.
        """
        # Add, remove, and reorder columns
        if to_drop:
            df.drop(columns=to_drop, inplace=True)
        if to_add:
            for aux in to_add:
                values = [defaults[aux]] * df.shape[0]
                df[aux] = values

        # Reorder
        df = df[columns]
        return df

    def change_columns(self, block, use_aux):
        """Updates AUX in the table if changes were made.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.
        """
        model = self.table.model()
        df = model.data_frame
        columns, _, defaults = self.dlg_input.data.get_column_info(block, use_aux)
        to_add, to_drop = self._columns_to_add_or_drop(columns, df)
        if not to_add and not to_drop:
            return
        model.data_frame = self._change_columns(df, columns, to_add, to_drop, defaults)
        self._set_model_defaults_and_read_only_columns(model, defaults)
        self.table.setModel(model)
        self.table.selectionModel().selectionChanged.connect(self.on_selection_changed)
        self.on_selection_changed()
        self.data_changed = True
        widget_builder.resize_columns_to_contents(self.table)

    def check_aux_change(self, block: str, new_aux: list[str]) -> str:
        """Return an error message if the new aux list is not valid, or '' if it is.

        PyCharm says this method could be a function, but it's also defined in PeriodArrayWidget so that wouldn't make
        sense.

        Args:
            block: Name of the list block.
            new_aux: New aux variable list.

        Returns:
            See description.
        """
        columns, _, _ = self.dlg_input.data.get_column_info(block, use_aux=False)
        msg = ''
        if any(aux in columns for aux in new_aux):
            msg = 'Auxiliary variables cannot have the same name as a standard column.'
        return msg

    def change_aux_variables(self, block, use_aux):
        """Updates AUX in the table if changes were made.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.
        """
        self.change_columns(block, use_aux)

    def accept(self):
        """Saves the current changes."""
        if not self.dlg_input.locked:
            self.save_list_blocks_to_temp()

    def reject(self):
        """Called when the user clicks Cancel."""
        pass
