"""Module for ImportDatasetsThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ImportDatasetsThread']

# 1. Standard Python modules
from datetime import datetime
from pathlib import Path
from typing import Optional
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.adcirc.file_io.solution_importer import ADCIRCSolutionReader
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.dmi.xms_data import XmsData
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter
from xms.gmi.data.generic_model import Section
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.ptmio.file_reader import ParseError
from xms.ptmio.fort14.dataset_reader import read_dataset

# 4. Local modules
from xms.ptm.model.model import HydroDefinition, SedimentDefinition, simulation_model


class ImportDatasetsThread(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Reading dataset files...',
            'working_prompt': 'Reading dataset files. Please wait...',
        }
        self._data = data
        self._control: Section = simulation_model()

        self._elevation_dataset: Optional[DatasetReader] = None
        self._flow_dataset: Optional[DatasetReader] = None
        self._d35_dataset: Optional[DatasetReader] = None
        self._d50_dataset: Optional[DatasetReader] = None
        self._d90_dataset: Optional[DatasetReader] = None
        self._ok = True

    def _run(self):
        """Run the thread."""
        self._get_control()
        self._read_flow()
        self._read_sediments()

        if not self._ok:
            return

        imported_hydro = imported_sediments = False
        if self._elevation_dataset and self._flow_dataset:
            self._data.add_dataset(self._elevation_dataset)
            self._data.add_dataset(self._flow_dataset)
            imported_hydro = True
        if self._d35_dataset and self._d50_dataset and self._d90_dataset:
            self._data.add_dataset(self._d35_dataset)
            self._data.add_dataset(self._d50_dataset)
            self._data.add_dataset(self._d90_dataset)
            imported_sediments = True
        if not imported_hydro and not imported_sediments:
            self._log.warning('No datasets were imported.')

    def _read_flow(self):
        """Read the flow datasets in whatever format they happen to be in."""
        flow_format = self._control.group('mesh').parameter('hydro_type').value
        if flow_format == HydroDefinition.adcirc_ascii:
            self._read_flow_ascii_adcirc()
        elif flow_format == HydroDefinition.adcirc_xmdf:
            self._read_flow_single_xmdf()
        elif flow_format == HydroDefinition.linked:
            self._log.info('Hydrodynamics defined as a dataset on a linked grid, skipping import.')
        else:
            raise AssertionError('Unsupported flow format.')

    def _read_flow_ascii_adcirc(self):
        """Read the flow datasets in ADCIRC ASCII format."""
        elevation_file: str = self._control.group('mesh').parameter('adcirc_ascii_elevation_file').value
        flow_file: str = self._control.group('mesh').parameter('adcirc_ascii_flow_file').value

        if not elevation_file:
            self._log.error('ADCIRC ASCII elevation file not set in model control.')
            self._ok = False
        elif not Path(elevation_file).exists():
            self._log.error('ADCIRC ASCII elevation file set in the model control does not exist.')
            self._ok = False

        if not flow_file:
            self._log.error('ADCIRC ASCII flow file not set in model control.')
            self._ok = False
        elif not Path(flow_file).exists():
            self._log.error('ADCIRC ASCII flow file set in the model control does not exist.')
            self._ok = False

        if not self._ok:
            return

        reftime = datetime(year=1, month=1, day=1)
        temp_dir = XmEnv.xms_environ_process_temp_directory()

        reader = ADCIRCSolutionReader([elevation_file], reftime, temp_dir)
        reader.read()
        if not reader.datasets:
            # The ADCIRC reader already logged an error for us.
            self._ok = False
            return
        self._elevation_dataset = reader.datasets[0]

        reader = ADCIRCSolutionReader([flow_file], reftime, temp_dir)
        reader.read()
        if not reader.datasets:
            # The ADCIRC reader fails silently in this case.
            self._ok = False
            self._log.error(f'Error reading solution file: {flow_file}')
            return
        self._flow_dataset = reader.datasets[0]

    def _read_flow_single_xmdf(self):
        """Read the flow datasets in single-XMDF format."""
        hydro_file: str = self._control.group('mesh').parameter('xmdf_hydro_file').value
        elevation_path: str = self._control.group('mesh').parameter('xmdf_elevation_path').value
        flow_path: str = self._control.group('mesh').parameter('xmdf_flow_path').value

        if not hydro_file:
            self._log.error('XMDF hydro file not set in model control.')
            self._ok = False
        elif not Path(hydro_file).exists():
            self._log.error('XMDF hydro file set in the model control does not exist.')
            self._ok = False

        if not elevation_path:
            self._log.error('Elevation dataset path not set in model control.')
            self._ok = False
        if not flow_path:
            self._log.error('Flow dataset path not set in model control.')
            self._ok = False

        if not self._ok:
            return

        try:
            self._elevation_dataset = DatasetReader(str(hydro_file), group_path=elevation_path)
        except KeyError:
            self._ok = False
            self._log.error('Unable to read elevation dataset.')
        try:
            self._flow_dataset = DatasetReader(str(hydro_file), group_path=flow_path)
        except KeyError:
            self._ok = False
            self._log.error('Unable to read flow dataset.')

    def _read_sediments(self):
        """Read the sediment datasets."""
        sediment_format = self._control.group('mesh').parameter('sediment_type').value
        if sediment_format == SedimentDefinition.ascii:
            self._read_sediments_ascii()
        elif sediment_format == SedimentDefinition.xmdf:
            self._read_sediments_xmdf()
        elif sediment_format == SedimentDefinition.linked:
            self._log.info('Sediments defined as a dataset on a linked grid, skipping import.')
        else:
            raise AssertionError('Unsupported sediment format.')

    def _read_sediments_ascii(self):
        """Read ASCII sediment file."""
        file_path = self._control.group('mesh').parameter('ascii_sediment_file').value

        if not file_path:
            self._log.error('ASCII sediment file not set in model control.')
            self._ok = False
        if not Path(file_path).exists():
            self._log.error('ASCII sediment file set in the model control does not exist.')
            self._ok = False
        if not self._data.linked_grid:
            self._log.error('Importing ASCII sediment datasets requires a grid linked to the simulation.')
            self._ok = False

        if not self._ok:
            return

        try:
            datasets = read_dataset(file_path, num_datasets=3)
        except ParseError as e:
            self._ok = False
            self._log.error('ASCII sediment file set in the model control was invalid.')
            self._log.error(e.message)
            return

        self._d35_dataset = self._make_ascii_dataset('d35', datasets[0])
        self._d50_dataset = self._make_ascii_dataset('d50', datasets[1])
        self._d90_dataset = self._make_ascii_dataset('d90', datasets[2])

    def _make_ascii_dataset(self, name, values):
        """Make a dataset for the ASCII reader."""
        temp_dir = Path(XmEnv.xms_environ_temp_directory())

        dataset_uuid = str(uuid.uuid4())
        path = str(temp_dir / f'{dataset_uuid}.h5')

        writer = DatasetWriter(
            h5_filename=path, name=name, dset_uuid=dataset_uuid, geom_uuid=self._data.linked_grid.uuid
        )

        np_values = np.array([values])
        writer.write_xmdf_dataset([0.0], np_values)
        return writer

    def _read_sediments_xmdf(self):
        """Read XMDF sediments."""
        file_path = self._control.group('mesh').parameter('xmdf_sediment_file').value

        if not file_path:
            self._log.error('XMDF sediment file not set in model control.')
            self._ok = False
        if not Path(file_path).exists():
            self._log.error('XMDF sediment file set in the model control does not exist.')
            self._ok = False
        d35_path = self._control.group('mesh').parameter('d35_dataset_path').value
        if not d35_path:
            self._log.error('d35 dataset path not set in model control.')
            self._ok = False
        d50_path = self._control.group('mesh').parameter('d50_dataset_path').value
        if not d50_path:
            self._log.error('d50 dataset path not set in model control.')
            self._ok = False
        d90_path = self._control.group('mesh').parameter('d90_dataset_path').value
        if not d90_path:
            self._log.error('d90 dataset path not set in model control.')
            self._ok = False

        if not self._ok:
            return

        try:
            self._d35_dataset = DatasetReader(str(file_path), group_path=d35_path)
        except KeyError:
            self._ok = False
            self._log.error('Unable to read d35 dataset.')
        try:
            self._d50_dataset = DatasetReader(str(file_path), group_path=d50_path)
        except KeyError:
            self._ok = False
            self._log.error('Unable to read d50 dataset.')
        try:
            self._d90_dataset = DatasetReader(str(file_path), group_path=d90_path)
        except KeyError:
            self._ok = False
            self._log.error('Unable to read d90 dataset.')

    def _get_control(self):
        """Get the model control."""
        self._log.info('Retrieving model control...')
        sim_data = self._data.simulation_data
        self._control.restore_values(sim_data.global_values)
