"""Module for PtmLoadSolutionThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['PtmLoadSolutionThread']

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.dmi.xms_data import XmsData
from xms.guipy.dialogs.feedback_thread import ExpectedError, FeedbackThread

# 4. Local modules
from xms.ptm.model.model import simulation_model


class PtmLoadSolutionThread(FeedbackThread):
    """Thread for loading the solution."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Loading PTM solution',
            'working_prompt': 'Loading PTM solution. Please wait...',
        }
        self._data = data

    def _run(self):
        """Run the thread."""
        self._log.info('Loading solution...')

        solution_file = self.find_solution_file()
        self.load_solution_file(solution_file)

        self._log.info('Done')

    def find_solution_file(self) -> Path:
        """Find the solution file."""
        self._log.info('Finding solution file...')

        base_path = self._data.simulation_folder
        self._log.info(f'Looking for solution in: {base_path}')

        sim_data = self._data.simulation_data
        model = simulation_model()
        model.restore_values(sim_data.global_values)
        prefix = model.group('output').parameter('output_prefix').value
        name = f'{prefix}_particles.h5'
        self._log.info(f'Expected file name: {name}')

        solution_file = base_path / name
        if not solution_file.exists():
            raise ExpectedError(f'Solution file "{solution_file}" does not exist.')

        return solution_file

    def load_solution_file(self, file: Path):
        """Send the solution file to XMS."""
        self._log.info(f'Loading solution file: {file}')
        self._data.add_particle_set(file)
