"""Module for the trap writer."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['write_traps']

# 1. Standard Python modules
import logging
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.coverage.polygons.polygon_orienteer import get_poly_points, make_poly_counter_clockwise
from xms.data_objects.parameters import Arc as DoArc, Polygon as DoPolygon
from xms.data_objects.parameters import Coverage
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.guipy.data.target_type import TargetType
from xms.ptmio.trap.trap_writer import write_traps as pio_write
from xms.ptmio.trap.traps import Direction, LineTrap, PolygonTrap, Traps as PioTraps

# 4. Local modules
from xms.ptm.model.model import trap_model

logger = logging.getLogger('xms.ptm')


def write_traps(coverage: Coverage, data: CoverageBaseData, where: str | Path | TextIO):
    """
    Write the traps to a file.

    Args:
        coverage: Coverage containing geometry to write.
        data: Data manager with the attributes for the geometry.
        where: Where to write them.
    """
    pio_traps = PioTraps()

    if coverage is not None:
        process_arcs(coverage, data, pio_traps)
        process_polygons(coverage, data, pio_traps)

    pio_write(pio_traps, where)


def process_arcs(coverage: Coverage, data: CoverageBaseData, traps: PioTraps):
    """Process the arcs in the coverage."""
    for arc in coverage.arcs:
        process_arc(arc, data, traps)


def process_arc(arc: DoArc, data: CoverageBaseData, traps: PioTraps):
    """Process an arc."""
    if arc.id not in data.component_id_map[TargetType.arc]:
        return

    mapping = {
        'Decreasing x-coordinate': Direction.decreasing_x,
        'Increasing x-coordinate': Direction.increasing_x,
        'Either Direction': Direction.any
    }

    attributes = trap_model().arc_parameters
    values = data.feature_values(TargetType.arc, feature_id=arc.id)
    attributes.restore_values(values)
    attributes = attributes.group('horizontal_line')

    check_arc(arc)

    trap = LineTrap()
    trap.trap_id = attributes.parameter('id').value
    trap.label = attributes.parameter('name').value
    trap.direction = mapping[attributes.parameter('direction').value]
    trap.bottom = attributes.parameter('bottom').value
    trap.top = attributes.parameter('top').value
    trap.is_open = attributes.parameter('is_open').value
    trap.is_single = attributes.parameter('is_single').value
    trap.start = (arc.start_node.x, arc.start_node.y)
    trap.end = (arc.end_node.x, arc.end_node.y)

    traps.line_traps.append(trap)


def check_arc(arc: DoArc):
    """Check that an arc is valid and warn if necessary."""
    if arc.vertices:
        logger.warning(
            f'Arc {arc.id} has a vertex. The PTM interface does not support vertices. The vertex was discarded.'
        )


def process_polygons(coverage: Coverage, data: CoverageBaseData, traps: PioTraps):
    """Process the polygons in the coverage."""
    for polygon in coverage.polygons:
        process_polygon(polygon, data, traps)


def process_polygon(polygon: DoPolygon, data: CoverageBaseData, traps: PioTraps):
    """Process a polygon in the coverage."""
    if polygon.id not in data.component_id_map[TargetType.polygon]:
        return

    attributes = trap_model().polygon_parameters
    values = data.feature_values(TargetType.polygon, feature_id=polygon.id)
    attributes.restore_values(values)
    attributes = attributes.group('polygon')

    check_polygon(polygon)

    trap = PolygonTrap()
    trap.trap_id = attributes.parameter('id').value
    trap.label = attributes.parameter('name').value
    trap.bottom = attributes.parameter('bottom').value
    trap.top = attributes.parameter('top').value
    trap.is_open = attributes.parameter('is_open').value
    trap.is_single = attributes.parameter('is_single').value

    locations = get_poly_points(polygon.arcs, polygon.arc_directions, inner=False)
    make_poly_counter_clockwise(locations)
    locations.pop()  # Discard the duplicated point
    locations = [(x, y) for (x, y, _z) in locations]
    trap.points = locations

    traps.polygon_traps.append(trap)


def check_polygon(polygon: DoPolygon):
    """Check that a polygon is valid and log a warning if necessary."""
    if polygon.interior_arcs:
        logger.warning(
            f'Polygon {polygon.id} has a hole. The PTM interface does not support polygons with holes. '
            'The hole was discarded.'
        )
