"""Data class for RSM mesh property."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import asdict, dataclass, field

# 2. Third party modules
from xms.gmi.data.generic_model import Group, Section

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import mesh_data_def as mdd
from xms.rsm.data.hpm_data import HpmData


@dataclass
class MeshPropertyData:
    """Manages data file for the hidden simulation component."""
    @dataclass
    class ConstValue:
        """Data class for constant value."""
        value: float = 0.0
        mult: float = 1.0

    @dataclass
    class Dataset:
        """Data class for dataset."""
        uuid: str = ''
        mult: float = 1.0
        specify_layer: bool = False
        layer: int = 0

    @dataclass
    class Transmissivity:
        """Data class for transmissivity."""
        unconfined: bool = False
        lookup_below: float = 0.0
        lookup_above: float = 0.0
        lookup_table: list = field(default_factory=list)

    @dataclass
    class RainEt:
        """Data class for rain and evapotranspiration."""
        curve: tuple = field(default_factory=tuple)
        dss_file: str = ''
        dss_path: str = ''
        dss_mult: float = 1.0
        dss_units: str = ''
        dss_type: str = ''
        grid_file: str = ''
        grid_file_x: float = 0.0
        grid_file_y: float = 0.0
        grid_file_mult: float = 1.0
        grid_file_dbintl: int = 0
        netcdf_file: str = ''
        netcdf_variable: str = ''
        netcdf_mult: float = 1.0
        netcdf_dbintl: int = 0
        netcdf_units: str = ''

    @dataclass
    class Conveyance:
        """Data class for conveyance."""
        specify_man_a: bool = False
        specify_man_b: bool = False
        specify_man_detent: bool = False
        manning_a: float = 0.0
        manning_b: float = 0.0
        manning_detent: float = 0.0
        specify_kad_k: bool = False
        specify_kad_alpha: bool = False
        specify_kad_beta: bool = False
        specify_kad_detent: bool = False
        kadlec_k: float = 0.0
        kadlec_alpha: float = 0.0
        kadlec_beta: float = 0.0
        kadlec_detent: float = 0.0
        lookup_below: float = 0.0
        lookup_above: float = 0.0
        lookup_base: float = 0.0
        lookup_exponent: float = 0.0
        lookup_table: list = field(default_factory=list)

    @dataclass
    class Svconverter:
        """Data class for svconverter."""
        lookup_sc: float = 0.0
        lookup_below: float = 0.0
        lookup_above: float = 0.0
        lookup_table: list = field(default_factory=list)

    label: str = ''
    option: str = mdd.OPT_NOT_SPECIFIED
    specify_index: bool = False
    index_value: int = -999
    const: ConstValue = field(default_factory=ConstValue)
    dataset: Dataset = field(default_factory=Dataset)
    transmissivity: Transmissivity = field(default_factory=Transmissivity)
    rain_et: RainEt = field(default_factory=RainEt)
    conveyance: Conveyance = field(default_factory=Conveyance)
    svconverter: Svconverter = field(default_factory=Svconverter)
    hpm: HpmData = field(default_factory=HpmData)

    def to_string(self):
        """Initializes the data class."""
        dd = asdict(self)
        return str(dd)


class _MeshPropertyDataBase:
    """Manages data file for the hidden simulation component."""
    def __init__(self, section: Section):
        """Initializes the data class.

        Args:
            section (Section): Generic model section.
        """
        self._section = section
        self._grp = None
        self._check_grp_active = False
        self._mesh_prop = {
            mdd.PROP_SURFACE: self._get_prop_const_or_dataset,
            mdd.PROP_BOTTOM: self._get_prop_const_or_dataset,
            mdd.PROP_SHEAD: self._get_prop_const_or_dataset,
            mdd.PROP_VERT_CONDUCTIVITY: self._get_prop_const_or_dataset,
            mdd.PROP_TRANSMISSIVITY: self._transmissivity,
            mdd.PROP_RAIN: self._rain_et,
            mdd.PROP_REFET: self._rain_et,
            mdd.PROP_CONVEYANCE: self._conveyance,
            mdd.PROP_SVCONVERTER: self._svconverter,
            mdd.PROP_HPM: self._hpm,
        }

    def mesh_property_data(self, prop: str) -> MeshPropertyData:
        """Get the mesh property data for the given mesh name.

        Args:
            prop (str): The name of the property.

        Returns:
            The mesh property data for the given mesh name.
        """
        if prop not in mdd.PROP_LIST:
            raise ValueError(f"Invalid mesh property name: {prop}")
        self._grp = self._section.group(prop)
        if self._check_grp_active and not self._grp.is_active:
            return MeshPropertyData()
        return self._mesh_prop[prop]()

    def _get_base_field_values(self) -> MeshPropertyData:
        """Get the base field values for the mesh property data."""
        mp = MeshPropertyData()
        mp.label = self._grp.parameter('label').value
        mp.option = self._grp.parameter('input_type').value
        if 'specify_index' in self._grp.parameter_names:
            mp.specify_index = self._grp.parameter('specify_index').value
            mp.index_value = self._grp.parameter('index_value').value
        return mp

    def _get_prop_const_or_dataset(self) -> MeshPropertyData:
        """Get the mesh property data for the surface, bottom, shead, vk."""
        mp = self._get_base_field_values()
        mp.const.value = self._grp.parameter('constant_value').value
        mp.const.mult = self._grp.parameter('constant_value_mult').value
        if 'dataset' in self._grp.parameter_names:
            mp.dataset.uuid = self._grp.parameter('dataset').value
            mp.dataset.mult = self._grp.parameter('dataset_mult').value
        if 'specify_layer' in self._grp.parameter_names:
            mp.dataset.specify_layer = self._grp.parameter('specify_layer').value
            mp.dataset.layer = self._grp.parameter('layer').value
        return mp

    def _transmissivity(self) -> MeshPropertyData:
        """Get the mesh property data for the transmissivity."""
        mp = self._get_prop_const_or_dataset()
        mp.transmissivity.unconfined = self._grp.parameter('unconfined').value
        mp.transmissivity.lookup_below = self._grp.parameter('lookup_below').value
        mp.transmissivity.lookup_above = self._grp.parameter('lookup_above').value
        mp.transmissivity.lookup_table = self._grp.parameter('lookup_table').value
        return mp

    def _rain_et(self) -> MeshPropertyData:
        """Get the mesh property data for the rain, et."""
        mp = self._get_prop_const_or_dataset()
        mp.rain_et.curve = self._grp.parameter('csv').value
        if mp.rain_et.curve == self._grp.parameter('csv').default:
            mp.rain_et.curve = ()
        mp.rain_et.dss_file = self._grp.parameter('dss_file').value
        mp.rain_et.dss_path = self._grp.parameter('dss_path').value
        mp.rain_et.dss_mult = self._grp.parameter('dss_mult').value
        mp.rain_et.dss_units = self._grp.parameter('dss_units').value
        mp.rain_et.dss_type = self._grp.parameter('dss_type').value
        mp.rain_et.grid_file = self._grp.parameter('grid_file').value
        mp.rain_et.grid_file_x = self._grp.parameter('grid_file_x').value
        mp.rain_et.grid_file_y = self._grp.parameter('grid_file_y').value
        mp.rain_et.grid_file_mult = self._grp.parameter('grid_file_mult').value
        mp.rain_et.grid_file_dbintl = self._grp.parameter('grid_file_dbintl').value
        if 'netcdf_file' in self._grp.parameter_names:
            mp.rain_et.netcdf_file = self._grp.parameter('netcdf_file').value
            mp.rain_et.netcdf_variable = self._grp.parameter('netcdf_variable').value
            mp.rain_et.netcdf_mult = self._grp.parameter('netcdf_mult').value
            mp.rain_et.netcdf_dbintl = self._grp.parameter('netcdf_dbintl').value
            mp.rain_et.netcdf_units = self._grp.parameter('netcdf_units').value
        return mp

    def _conveyance(self) -> MeshPropertyData:
        """Get the mesh property data for the conveyance."""
        mp = self._get_base_field_values()
        if mp.option == mdd.OPT_CONVEY_MANNING:
            mp.conveyance.specify_man_a = self._grp.parameter('manning_a').value
            mp.conveyance.specify_man_b = self._grp.parameter('manning_b').value
            mp.conveyance.specify_man_detent = self._grp.parameter('manning_detent').value
            mp.conveyance.manning_a = self._grp.parameter('manning_a_value').value
            mp.conveyance.manning_b = self._grp.parameter('manning_b_value').value
            mp.conveyance.manning_detent = self._grp.parameter('manning_detent_value').value
        elif mp.option == mdd.OPT_CONVEY_KADLEC:
            mp.conveyance.specify_kad_k = self._grp.parameter('kadlec_K').value
            mp.conveyance.specify_kad_alpha = self._grp.parameter('kadlec_alpha').value
            mp.conveyance.specify_kad_beta = self._grp.parameter('kadlec_beta').value
            mp.conveyance.specify_kad_detent = self._grp.parameter('kadlec_detent').value
            mp.conveyance.kadlec_k = self._grp.parameter('kadlec_K_value').value
            mp.conveyance.kadlec_alpha = self._grp.parameter('kadlec_alpha_value').value
            mp.conveyance.kadlec_beta = self._grp.parameter('kadlec_beta_value').value
            mp.conveyance.kadlec_detent = self._grp.parameter('kadlec_detent_value').value
        elif mp.option == mdd.OPT_CONVEY_LOOKUP:
            mp.conveyance.lookup_below = self._grp.parameter('lookup_below').value
            mp.conveyance.lookup_above = self._grp.parameter('lookup_above').value
            mp.conveyance.lookup_base = self._grp.parameter('lookup_base').value
            mp.conveyance.lookup_exponent = self._grp.parameter('lookup_exponent').value
            mp.conveyance.lookup_table = self._grp.parameter('lookup_table').value
        return mp

    def _svconverter(self) -> MeshPropertyData:
        """Get the mesh property data for the svconverter."""
        mp = self._get_prop_const_or_dataset()
        mp.svconverter.lookup_sc = self._grp.parameter('lookup_sc').value
        mp.svconverter.lookup_below = self._grp.parameter('lookup_below').value
        mp.svconverter.lookup_above = self._grp.parameter('lookup_above').value
        mp.svconverter.lookup_table = self._grp.parameter('lookup_table').value
        if mp.option == mdd.OPT_SV_CONSTANT:
            mp.option = mdd.OPT_CONSTANT
        elif mp.option == mdd.OPT_SV_DATASET:
            mp.option = mdd.OPT_DATASET
        return mp

    def _hpm(self) -> MeshPropertyData:
        """Get the mesh property data for the hpm."""
        mp = MeshPropertyData()
        mp.label = self._grp.parameter('label').value
        mp.specify_index = self._grp.parameter('specify_index').value
        mp.index_value = self._grp.parameter('index_value').value
        mp.option = self._grp.parameter('hpm_type').value
        from xms.rsm.data.hpm_data_def import load_data_from_group
        load_data_from_group(group=self._grp, hpm_data=mp.hpm)
        return mp


class SimulationMeshPropertyData(_MeshPropertyDataBase):
    """Manages data file for the hidden simulation component."""
    def __init__(self, section: Section):
        """Initializes the data class.

        Args:
            section (Section): The section of the generic model.
        """
        super().__init__(section)


class CoverageMeshPropertyData(_MeshPropertyDataBase):
    """Manages data file for the hidden simulation component."""
    def __init__(self, section: Section):
        """Initializes the data class.

        Args:
            section (Section): The section of the generic model.
        """
        super().__init__(section)
        self._check_grp_active = True


def mesh_property_data_from_hpm_group(grp: Group) -> MeshPropertyData:
    """Get the mesh property data from the given hpm group.

    Args:
        grp (Group): The hpm group.

    Returns:
        The mesh property data from the given hpm group.
    """
    mp = MeshPropertyData()
    mp.label = grp.parameter('label').value
    # if 'specify_index' in grp.parameter_names:
    #     mp.specify_index = grp.parameter('specify_index').value
    #     mp.index_value = grp.parameter('index_value').value
    mp.option = grp.parameter('hpm_type').value
    from xms.rsm.data.hpm_data_def import load_data_from_group
    load_data_from_group(group=grp, hpm_data=mp.hpm)
    return mp
