"""Data class for SCHISM simulation component."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.schism.data.base_data import BaseData


class SimData(BaseData):
    """Manages data file for the hidden simulation component."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the data class.

        Args:
            main_file: The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        super().__init__(main_file)
        self.info.attrs['SCHISM_FILE_TYPE'] = 'SCHISM_SIMULATION'
        self.tracer_modules = [
            'T', 'S', 'GEN', 'AGE', 'SED3D', 'EcoSim', 'ICM', 'CoSINE', 'Feco', 'TIMOR', 'FABM', 'DVD'
        ]

        self.commit()

    @property
    def _main_file_name(self) -> str:
        """What to name the component's main file."""
        return 'schism_sim.nc'

    @property
    def global_values(self) -> str:
        """The model's global values."""
        if 'GLOBAL_VALUES' in self.info.attrs:
            return self.info.attrs['GLOBAL_VALUES']
        return GenericModel().global_parameters.extract_values()

    @global_values.setter
    def global_values(self, value: str):
        """The model's global values."""
        self.info.attrs['GLOBAL_VALUES'] = value
