"""Module for XmsData."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property
from pathlib import Path
from typing import Optional, Sequence

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.constraint import read_grid_from_file, UGrid2d
from xms.data_objects.parameters import Component, Projection
from xms.data_objects.parameters import Coverage
from xms.gmi.components.coverage_component import ARC_JSON, POLY_JSON

# 4. Local modules
from xms.schism.components.coverage_component import CoverageComponent


class _DefaultDict(dict):
    """A dictionary similar to collections.defaultdict, which passes the key to its factory instead."""
    def __init__(self, factory):
        super().__init__()
        self.factory = factory

    def __missing__(self, key):
        self[key] = self.factory(key)
        return self[key]


class XmsData:
    """Object for sending data to and retrieving data from XMS."""
    def __init__(self, query: Optional[Query]):
        """
        Initialize the class.

        Args:
            query: Interprocess communication object.
        """
        self.query = query or Query()
        self.datasets: dict[str, Optional[Sequence[float]]] = _DefaultDict(self._get_dataset)

    @cached_property
    def display_projection(self) -> Projection:
        """The current display projection."""
        return self.query.display_projection

    @cached_property
    def project_tree(self) -> tree_util.TreeNode:
        """The project tree."""
        return self.query.copy_project_tree()

    @cached_property
    def sim_tree_node(self) -> tree_util.TreeNode:
        """The tree node for the simulation."""
        whole_tree = self.project_tree

        # We should be at either a tree item (sim, coverage, ugrid) or a hidden component. Hidden components aren't in
        # the tree, but they should be children of tree items.
        item_uuid = self.query.current_item_uuid()
        node = tree_util.find_tree_node_by_uuid(whole_tree, item_uuid)
        if not node:  # We're at a hidden component
            item_uuid = self.query.parent_item_uuid()
            node = tree_util.find_tree_node_by_uuid(whole_tree, item_uuid)

        # Now we have the current tree item, but it might not be the simulation. It might be a mapped coverage if
        # we're running from a component.
        while node.item_typename != 'TI_DYN_SIM' and node.parent is not None:
            node = node.parent

        return node

    @cached_property
    def sim_main_file(self) -> Path:
        """The simulation's main-file."""
        sim_uuid = self.sim_tree_node.uuid
        do_sim = self.query.item_with_uuid(sim_uuid, model_name='SCHISM', unique_name='SimComponent')
        return Path(do_sim.main_file)

    @cached_property
    def mapped_boundary_coverage_file(self) -> Optional[Path]:
        """The main-file for the mapped boundary condition coverage, if one is mapped."""
        sim_tree_node = self.sim_tree_node
        mapped_coverage_node = tree_util.descendants_of_type(
            sim_tree_node, unique_name='MappedBcComponent', only_first=True
        )
        if mapped_coverage_node is None:
            return None
        mapped_coverage_uuid = mapped_coverage_node.uuid
        main_file = self.query.item_with_uuid(mapped_coverage_uuid).main_file
        return main_file

    @cached_property
    def mapped_tides_main_file(self) -> Optional[Path]:
        """The main-file for the mapped tides, if one is mapped."""
        sim_node = self.sim_tree_node
        tides_node = tree_util.descendants_of_type(sim_node, unique_name='MappedTidalComponent', only_first=True)
        if tides_node is None:
            return None

        tides_uuid = tides_node.uuid
        main_file = self.query.item_with_uuid(tides_uuid).main_file
        return main_file

    @cached_property
    def mapped_solver_main_file(self) -> Optional[Path]:
        """The main-file for the mapped upwind solver coverage, if one is mapped."""
        sim_node = self.sim_tree_node
        solver_node = tree_util.descendants_of_type(
            sim_node, unique_name='MappedUpwindSolverCoverageComponent', only_first=True
        )
        if solver_node is None:
            return None

        solver_uuid = solver_node.uuid
        main_file = self.query.item_with_uuid(solver_uuid).main_file
        return main_file

    @cached_property
    def ugrid_tree_node(self) -> tree_util.TreeNode:
        """The tree node of the UGrid."""
        ugrid_node = tree_util.descendants_of_type(
            self.sim_tree_node, allow_pointers=True, xms_types=['TI_MESH2D_PTR'], only_first=True
        )
        return ugrid_node

    @cached_property
    def ugrid_file(self) -> Optional[str]:
        """The file where the UGrid associated with the simulation was dumped to."""
        ugrid_node = self.ugrid_tree_node
        if ugrid_node is None:
            return None
        ugrid_uuid = ugrid_node.uuid
        do_ugrid = self.query.item_with_uuid(ugrid_uuid)
        return do_ugrid.cogrid_file

    @cached_property
    def ugrid(self) -> Optional[UGrid2d]:
        """
        The UGrid associated with the simulation.
        """
        if self.ugrid_file is None:
            return None
        ugrid = read_grid_from_file(self.ugrid_file)
        return ugrid

    @cached_property
    def current_item_name(self) -> str:
        """The name of the current item where the query is at."""
        whole_tree = self.project_tree

        # We should be at either a tree item (sim, coverage, ugrid) or a hidden component. Hidden components aren't in
        # the tree, but they should be children of tree items.
        item_uuid = self.query.current_item_uuid()
        node = tree_util.find_tree_node_by_uuid(whole_tree, item_uuid)
        if not node:  # We're at a hidden component
            item_uuid = self.query.parent_item_uuid()
            node = tree_util.find_tree_node_by_uuid(whole_tree, item_uuid)

        return node.name

    def _get_dataset(self, uuid: str) -> Optional[Sequence[float]]:
        """Get a dataset if it exists."""
        data = self.query.item_with_uuid(uuid)
        if not data:
            return None

        return data.values[0]

    def item_with_uuid(self, item_uuid, generic_coverage=False, model_name=None, unique_name=None):
        """
        Get the item with a particular UUID.

        Args:
            item_uuid: UUID of the item to get.
            generic_coverage: True if the item is one of the dumpable generic coverages
            model_name: The XML model name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify unique_name if model_name provided.
            unique_name: The XML unique name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify model_name if unique_name provided.
        """
        return self.query.item_with_uuid(item_uuid, generic_coverage, model_name, unique_name)

    def add_coverage(self, coverage: Coverage, component: CoverageComponent, upwind_solver: bool = False):
        """
        Add a coverage to XMS.

        Args:
            coverage: The coverage to add.
            component: The coverage's hidden component.
            upwind_solver: Whether this is an upwind solver coverage, as opposed to a boundary condition one.
        """
        do_component = Component(
            comp_uuid=component.uuid,
            main_file=str(component.main_file),
            class_name=component.class_name,
            module_name=component.module_name,
        )
        coverage_type = 'Upwind solver coverage' if upwind_solver else 'Boundary Conditions'
        file_name = POLY_JSON if upwind_solver else ARC_JSON

        component.data.coverage_uuid = coverage.uuid
        component.data.commit()

        component.cov_uuid = coverage.uuid

        file = str(Path(component.main_file).parent / file_name)

        message = list(XmsDisplayMessage(file=file, edit_uuid=component.cov_uuid))

        # Using keywords makes it possible for the import script to do all the initialization (writing .json and .ids
        # files) itself, and also avoids the need to write the .attids files. This way of doing it is very ugly since
        # it's a hack to make things work. Long term it would be nice to make the visible coverage component do this.
        keywords = [{
            'component_coverage_ids': [component.uuid, component.update_ids],
            'display_options': [message],
        }]

        self.query.add_coverage(
            coverage,
            model_name='SCHISM',
            coverage_type=coverage_type,
            components=[do_component],
            component_keywords=keywords
        )
