"""Module for UnmapCoverageRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.schism.components.coverage_component import CoverageComponent
from xms.schism.components.coverage_component_builder import CoverageComponentBuilder
from xms.schism.data.coverage_data import CoverageData
from xms.schism.data.mapped_bc_data import MappedBcData
from xms.schism.dmi.xms_data import XmsData
from xms.schism.external.crc import compare_crc


class UnmapCoverageRunner(FeedbackThread):
    """Feedback thread for unmapping a boundary condition coverage."""
    def __init__(self, data: MappedBcData, query: Query):
        """Constructor."""
        super().__init__(query)
        self.data = data
        self.xms_data = XmsData(self._query)

        self.display_text = {
            'title': 'Unmapping coverage',
            'working_prompt': 'Unmapping SCHISM coverage. Please wait...',
            'warning_prompt': 'Warning(s) encountered. Review log output for more details.',
            'error_prompt': 'Error(s) encountered. Review log output for more details.',
            'success_prompt': 'Completed successfully',
            'note': '',
            'auto_load': 'Close this dialog automatically when importing is finished.'
        }

    def _run(self):
        self._log.info('Checking domain...')
        ugrid_file = self.xms_data.ugrid_file
        if not compare_crc(ugrid_file, self.data.domain_hash):
            self._log.warning('The domain was changed since the coverage was mapped. Unmapping results may be wrong.')

        self._log.info('Retrieving domain...')
        ugrid = self.xms_data.ugrid

        data = CoverageData()  # This is ugly. It's just to coax the data manager into creating a main file.
        data.close()
        component = CoverageComponent(data.main_file)
        data = component.data

        name = f'{self.xms_data.current_item_name} (unmapped)'

        builder = CoverageComponentBuilder(data.main_file, name, ugrid)

        open_arcs = self.data.open_arcs

        for component_id, (nodes, values) in enumerate(open_arcs, start=1):
            nodes = [node - 1 for node in nodes]
            builder.add_node_string(nodes, component_id, feature_id=component_id)
            data.add_feature(TargetType.arc, values, 'open')
        coverage = builder.complete()

        component.cov_uuid = coverage.uuid
        data.commit()
        for component_id in range(1, len(open_arcs) + 1):
            component.update_component_id(TargetType.arc, component_id, component_id)
        component.update_id_files()

        self.xms_data.add_coverage(coverage, component, upwind_solver=False)

        self._log.info('The coverage was unmapped')
