"""ModelControl class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param
import pkg_resources

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules
from xms.srh.data.model_control_advanced import ModelControlAdvanced
from xms.srh.data.model_control_hydro import ModelControlHydro
from xms.srh.data.model_control_output import ModelControlOutput
from xms.srh.data.model_control_sediment import ModelControlSediment


class ModelControl(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    file_type = param.String(
        default='SRH_SIM_DATA',
        doc='file type identifier for file_io',
        precedence=-2,
    )
    file_version = param.String(
        default=pkg_resources.get_distribution('xmssrh').version,
        doc='file version string for file_io',
        precedence=-1,
    )
    mesh_uuid = param.String(
        default='',
        doc='UUID of mesh linked to simulation',
        precedence=-1,
    )
    hydro = param.ClassSelector(
        label='',
        class_=ModelControlHydro,
        doc='General model control options',
        precedence=1,
    )
    output = param.ClassSelector(
        label='',
        class_=ModelControlOutput,
        doc='Output model control options',
        precedence=2,
    )
    advanced = param.ClassSelector(
        label='',
        class_=ModelControlAdvanced,
        doc='Advanced model control options',
        precedence=3,
    )
    enable_sediment = param.Boolean(
        default=False,
        doc='enable sediment transport in the simulation',
        precedence=4,
    )
    sediment = param.ClassSelector(
        label='',
        class_=ModelControlSediment,
        doc='Sediment transport model control options',
        precedence=5,
    )

    def __init__(self):
        """Initializes the data class."""
        # call super first because we need the par magic to happen
        super().__init__()
        self.advanced = ModelControlAdvanced()
        self.hydro = ModelControlHydro()
        self.output = ModelControlOutput()
        self.sediment = ModelControlSediment()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['enable_sediment'],
            depends={'enable_sediment': ['sediment']},
        )

    @param.depends('enable_sediment', watch=True)
    def _update_enable_sediment(self):
        self.enabler.do_enabling(force_disable=False)
