"""BcDataCulvertHy8 class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules


class BcDataCulvertHy8(param.Parameterized):
    """Bc type culvert-hy8 data."""
    units = param.ObjectSelector(
        label='Units for HY-8 display',
        default='English',
        objects=['English', 'Metric'],
        doc='',
        precedence=2,
    )
    total_head = param.Boolean(
        label='Use total head (velocity and water surface)',
        default=False,
        doc='',
        precedence=3,
    )
    simulate_as_link = param.Boolean(
        label='Checked uses 2d terrain for overtopping.\n'
        'Unchecked disables 2d domain in culvert region and uses hy8 for overtopping.',
        default=False,
        doc='',
        precedence=4,
    )
    hy8_crossing_guid = param.String(
        default='',
        doc='The guid of the HY-8 crossing',
        precedence=-1,
    )
    crossing_selector = param.ObjectSelector(
        label='HY-8 crossing',
        default='None selected',
        objects=['None selected'],
        doc='The list is dynamically updated by the data in the HY-8 file.',
        precedence=5,
    )
    launch_hy8 = param.Action(
        default=None,
        label='Launch HY-8...',
        precedence=6,
    )
    tolerance = param.Number(
        default=1.0,
        bounds=(1.0, None),
        doc='elevation tolerance',
        label='Tolerance for checking consistency between SRH mesh elevation and HY8 culvert invert elevation',
        precedence=7
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.hy8_exe = ''
        self.hy8_input_file = ''
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=[],
            depends={},
        )

    def name_guid_dict_from_hy8_file(self, include_crest_length=False):
        """Get a dictionary of crossing name and guid from an HY-8 file.

        Args:
            include_crest_length (bool): Include the crest length in the dictionary.
        """
        hy8_file_dict = hy8_file_dict_from_hy8_file(self.hy8_input_file)
        name_dict = {}

        for crossing, details in hy8_file_dict['crossings'].items():
            if 'GUID' in details:
                name_dict[crossing] = details['GUID']
            if include_crest_length and 'ROADWAYPOINT' in details:
                cl_dict = name_dict.get('##crest_length##', {})
                cl_dict[name_dict[crossing]] = details['ROADWAYPOINT']
                name_dict['##crest_length##'] = cl_dict
        return name_dict


def hy8_file_dict_from_hy8_file(hy8_input_file: str | Path) -> dict:
    """Get a dictionary of crossing name and guid from an HY-8 file."""
    hy8_file_dict = {'crossings': {}}
    if not hy8_input_file or not os.path.isfile(hy8_input_file):
        return hy8_file_dict

    with open(hy8_input_file, 'r') as f:
        lines = f.readlines()
        name = ''
        culvert_dict = {}
        barrel_dict = {}
        for line in lines:
            items = line.split()
            if len(items) < 1:
                pass
            elif 'UNITS' == items[0]:
                hy8_file_dict['UNITS'] = items[1]  # 0 = U.S. Metric, 1 = Imperial
            elif 'STARTCROSSING' == items[0]:
                name = line.replace('STARTCROSSING', '').strip().strip('"')
                hy8_file_dict['crossings'][name] = {'culverts': {}}
            elif 'STARTCULVERT' == items[0]:
                culvert_name = line.replace('STARTCULVERT', '').strip().strip('"')
                hy8_file_dict['crossings'][name]['culverts'][culvert_name] = {}
            elif 'CULVERTSHAPE' == items[0]:
                culvert_dict['CULVERTSHAPE'] = float(items[1])  # 1 = Circular, 2 = Box
            elif 'BARRELDATA' == items[0]:
                barrel_dict['SPAN'] = float(items[1])
                barrel_dict['RISE'] = float(items[2])
                barrel_dict['NTop'] = float(items[3])
                barrel_dict['NBot'] = float(items[4])
                culvert_dict['BARRELDATA'] = barrel_dict  # There are 4 data points (x, y, ?, ?)
                barrel_dict = {}
            elif 'EMBEDDEPTH' == items[0]:
                culvert_dict['EMBEDDEPTH'] = float(items[1])  # Embed
            elif 'INVERTDATA' == items[0]:
                culvert_dict['upstream_invert'] = float(items[2])  # Upstream elevation
                culvert_dict['downstream_invert'] = float(items[4])  # Downstream elevation
            elif 'CROSSGUID' == items[0]:
                hy8_file_dict['crossings'][name]['GUID'] = items[1]
            elif 'ROADWAYPOINT' == items[0]:
                hy8_file_dict['crossings'][name]['ROADWAYPOINT'] = float(items[1])
            elif 'ENDCULVERT' == items[0]:
                hy8_file_dict['crossings'][name]['culverts'][culvert_name] = culvert_dict
                culvert_dict = {}
            elif 'ENDCROSSING' == items[0]:
                name = ''
    return hy8_file_dict
