"""Reads SRH-2D boundary condition data from the hydro file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shlex
import sys

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
import xms.core.filesystem.filesystem as xmsfs

# 4. Local modules
from xms.srh.data.par.bc_data_param import BcDataParam
from xms.srh.file_io import io_util


class HydroBcReader:
    """Reads SRH-2D boundary condition data from the hydro file."""
    def __init__(self, hydro_reader):
        """Initializes the class.

        Args:
            hydro_reader (:obj:`HydroReader`): The main hydro file reader
        """
        self.hydro_reader = hydro_reader
        self.logger = hydro_reader.logger
        self.lines = []
        self._structs = ['hy8', 'culvert', 'weir', 'press', 'gate', 'link']
        self._struct_key_words = {
            'hy8params',
            'culvertparams',
            'culvertweirparams',
            'weirparams',
            'pressureparams',
            'pressovertop',
            'pressweirparams',
            'pressweirparams2',
            'gateparams',
            'linkconstantflow',
            'linktimeseriesflow',
            'linkweirflow',
            'linkrcflow',
            'linkflowtype',
            'linkflowtype2',
            'linkspecifiedlagtime',
            'linkcomputedlagtime',
            'linklagtimeparameters',
            'hy8flowdirection',
            'gateflowdirection',
            'pressweirflowdirection',
            'weirflowdirection',
            'linkflowdirection',
        }
        # ewsparams is from old srh (sms 11.x)
        self._bc_key_words = {
            'iqparams',
            'ewsparamsc',
            'ewsparamsts',
            'ewsparamsrc',
            'eqparams',
            'isupcrparams',
            'wallroughness',
            'intconstantflow',
            'inttsflow',
            'intweirflow',
            'intrcflow',
            'intflowtype',
            'ewsparams',
        }
        self._bc_key_words.update(self._struct_key_words)
        self.bc_type = {
            'inlet-q': 'Inlet-Q (subcritical inflow)',
            'exit-h': 'Exit-H (subcritical outflow)',
            'exit-q': 'Exit-Q (known-Q outflow)',
            'inlet-sc': 'Inlet-SC (supercritical inflow)',
            'exit-ex': 'Exit-EX (supercritical outflow)',
            'wall': 'Wall (no-slip boundary)',
            'symmetry': 'Symmetry (slip boundary)',
            'internal': 'Internal sink',
            'bcdata': 'Bc Data',
        }

        self.struct_type = {
            'hy8nodestrings': 'Culvert HY-8',
            'culvertnodestrings': 'Culvert',
            'weirnodestrings': 'Weir',
            'pressurenodestrings': 'Pressure',
            'gatenodestrings': 'Gate',
            'linknodestrings': 'Link',
        }
        self._words = []
        self._culvert_inlet = self._create_culvert_inlet_lookup_stupid()
        for key in self._structs:
            self.hydro_reader.structures[key] = {}

    def add_bc(self, line):
        """Add a boundary condition arc.

        Args:
            line (:obj:`str`): line from file
        """
        words = line.split()
        if len(words) < 3:
            raise RuntimeError(f'Invalid line format: {line}')

        bc_type = words[2].lower()
        nodestring_id = int(words[1])
        if bc_type == 'monitoring':
            self.hydro_reader.monitor_line_ids.add(nodestring_id)
            return

        is_struct = bc_type.startswith(tuple(self._structs))
        if bc_type not in self.bc_type and not is_struct:
            raise RuntimeError(f'Invalid bc type: {line}')

        bc_par = BcDataParam()
        if not is_struct:
            bc_par.bc_type = self.bc_type[bc_type]
        if bc_type == 'bcdata':
            if len(words) > 3:
                bc_par.bc_data.label = words[3].strip('"')
            else:
                bc_par.bc_data.label = f'bc_data_line_{nodestring_id}'
        self.hydro_reader.bcs[nodestring_id] = bc_par

    def add_bc_struct(self, line):
        """Add a boundary condition structure.

        Args:
            line (:obj:`str`): line from file
        """
        words = line.split()
        if len(words) < 4:
            raise RuntimeError(f'Invalid line format: {line}')

        struct_type = words[0].lower()
        structure_id = int(words[1])

        bc_up_id = int(words[2])
        bc_down_id = int(words[3])
        if bc_up_id in self.hydro_reader.bcs:
            bc_par = self.hydro_reader.bcs[bc_up_id]
        else:
            raise RuntimeError(f'Invalid line format: {line}')

        self.hydro_reader.bcs[bc_up_id] = bc_par
        self.hydro_reader.bcs[bc_down_id] = bc_par

        self.hydro_reader.bc_arc_id_to_bc_id[bc_up_id] = bc_up_id
        self.hydro_reader.bc_arc_id_to_bc_id[bc_down_id] = bc_up_id
        self.hydro_reader.bc_id_to_structure[bc_up_id] = {'up': bc_up_id, 'down': bc_down_id}

        bc_par.arcs.arc_id_0 = bc_up_id
        bc_par.arcs.arc_option_0 = 'Upstream'
        bc_par.arcs.arc_id_1 = bc_down_id
        bc_par.arcs.arc_option_1 = 'Downstream'
        bc_par.bc_type = self.struct_type[struct_type]

        # check for BCDATA lines
        if len(words) > 4:
            if words[4].lower() != 'no':
                line_id = int(words[4])
                bc_par.bc_data_lines.specify_upstream_bcdata_line = True
                bc_par.bc_data_lines.upstream_line_label = self.hydro_reader.bcs[line_id].bc_data.label
            if len(words) > 5:
                if words[5].lower() != 'no':
                    line_id = int(words[5])
                    bc_par.bc_data_lines.specify_downstream_bcdata_line = True
                    bc_par.bc_data_lines.downstream_line_label = self.hydro_reader.bcs[line_id].bc_data.label

        for key in self._structs:
            if words[0].lower().startswith(key):
                self.hydro_reader.structures[key][structure_id] = bc_par

    def is_bc_key_word(self, card):
        """Check if a card is a BC keyword.

        Args:
            card (:obj:`str`): first words on input line

        Returns:
            (:obj:`bool`): True if the card is read by this class
        """
        return card.lower() in self._bc_key_words

    def read_bc_data(self):
        """Reads the bc data."""
        for line in self.lines:
            self._words = shlex.split(line, posix="win" not in sys.platform)
            if len(self._words) < 3:
                raise RuntimeError(f'Invalid line format: {line}')

            bc_id = int(self._words[1])
            bc_par = None
            if self._words[0].lower() in self._struct_key_words:
                for key in self._structs:
                    if self._words[0].lower().startswith(key):
                        bc_par = self.hydro_reader.structures[key][bc_id]
            elif bc_id in self.hydro_reader.bcs:
                bc_par = self.hydro_reader.bcs[bc_id]

            if bc_par is None:
                raise RuntimeError(f'BC Id: {bc_id} not found.')

            read_method_name = f'_read_{self._words[0].lower()}'
            if not hasattr(self, read_method_name) and read_method_name.endswith('flowdirection'):
                read_method_name = '_read_flow_direction'
            read_method = getattr(self, read_method_name)
            read_method(bc_par)

    def _dataframe_from_xys(self, xys_filename, columns, swap_x_y=False):
        """Create a pandas.DataFrame from an xys file.

        Args:
            xys_filename (:obj:`str`): Path to the xys file
            columns (:obj:`list[str]`): The X and Y column names
            swap_x_y (:obj:`bool`): swap the x and y values

        Returns:
            (:obj:`pandas.DataFrame`): See description
        """
        filename = os.path.join(os.path.dirname(self.hydro_reader.filename), xys_filename)
        _, x, y = io_util.read_xys(filename)
        if swap_x_y:
            tmp = x
            x = y
            y = tmp
        return pd.DataFrame({columns[0]: x, columns[1]: y})

    def _dataframe_from_xys_rating_curve(self, xys_filename, columns):
        """Create a pandas.DataFrame from an xys rating curve file.

        Args:
            xys_filename (:obj:`str`): Path to the xys file
            columns (:obj:`list[str]`): The X and Y column names

        Returns:
            (:obj:`pandas.DataFrame`): See description
        """
        filename = os.path.join(os.path.dirname(self.hydro_reader.filename), xys_filename)
        rating_curve, x, y = io_util.read_xys(filename)
        return rating_curve, pd.DataFrame({columns[0]: x, columns[1]: y})

    def _create_culvert_inlet_lookup_stupid(self):
        """Creates a dict to look up culvert inlet stuff."""
        from xms.srh.file_io.hydro_bc_writer import HydroBcWriter
        return dict((v, k) for k, v in HydroBcWriter.get_inlet_coefficients().items())

    def _fill_weir_user_type_vals(self, weir, cw, a, b):
        """Populate weir attributes.

        Args:
            weir (:obj:`BcDataWeir`): The data to fill
            cw (:obj:`float`): The cw parameter
            a (:obj:`float`): The a parameter
            b (:obj:`float`): The b parameter
        """
        weir.cw = float(cw)
        weir.a = float(a)
        weir.b = float(b)

    def _read_iqparams(self, bc_par):
        """Read IQ BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        # If sediment is enabled, there is an extra card in the middle of this line.
        unit_idx = 3 if self._words[3].lower() == 'en' or self._words[3].lower() == 'si' else 4
        dist_idx = unit_idx + 1
        if io_util.is_float(self._words[2]):
            bc_par.inlet_q.discharge_option = 'Constant'
            bc_par.inlet_q.constant_q = float(self._words[2])
            unit_str = 'cfs' if self._words[unit_idx] == 'EN' else 'cms'
            bc_par.inlet_q.constant_q_units = unit_str
        else:
            bc_par.inlet_q.discharge_option = 'Time series'
            bc_par.inlet_q.time_series_q = self._dataframe_from_xys(self._words[2].strip('"\''), ['hrs', 'vol_per_sec'])
            unit_str = 'hrs -vs- cfs' if self._words[unit_idx] == 'EN' else 'hrs -vs- cms'
            bc_par.inlet_q.time_series_q_units = unit_str

        if unit_idx == 4:  # Sediment is enabled
            bc_par.sediment_inflow.sediment_discharge_type = (
                'Capacity' if self._words[3].lower() == 'capacity' else 'File'
            )
            if bc_par.sediment_inflow.sediment_discharge_type == 'File':
                sed_file = self._words[3].strip('"\'')
                sed_file_w_path = xmsfs.resolve_relative_path(self.hydro_reader.filename, sed_file)
                if os.path.isfile(sed_file_w_path):
                    bc_par.sediment_inflow.sediment_file = sed_file_w_path
                else:
                    self.logger.error(f'Unable to find sediment load file: {sed_file}')
                    self.logger.error(f'Full path to file: {sed_file_w_path}')
                    bc_par.sediment_inflow.sediment_discharge_type = 'Capacity'

        if len(self._words) > dist_idx and self._words[dist_idx]:
            bc_par.inlet_q.distribution_at_inlet = self._words[dist_idx].title()

    def _read_ewsparams(self, bc_par):
        """Read EWS params, it may be constant, time series or rating curve.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        try:
            float(self._words[2])  # see if this is a constant exit-h
            self._read_ewsparamsc(bc_par)
        except ValueError:
            xys_filename = self._words[2].strip('"\'')
            filename = os.path.join(os.path.dirname(self.hydro_reader.filename), xys_filename)
            rating_curve, _, _ = io_util.read_xys(filename)
            if rating_curve:
                self._read_ewsparamsrc(bc_par)
            else:
                self._read_ewsparamsts(bc_par)

    def _read_ewsparamsc(self, bc_par):
        """Read EWS constant BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.exit_h.water_surface_elevation_option = 'Constant'
        bc_par.exit_h.constant_wse = float(self._words[2])
        unit_str = 'Feet' if self._words[3] == 'EN' else 'Meters'
        bc_par.exit_h.constant_wse_units = unit_str

    def _read_ewsparamsts(self, bc_par):
        """Read EWS time series BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.exit_h.water_surface_elevation_option = 'Time series'
        bc_par.exit_h.time_series_wse = self._dataframe_from_xys(self._words[2].strip('"\''), ['hrs', 'm or ft'])
        unit_str = 'hrs -vs- feet' if self._words[3] == 'EN' else 'hrs -vs- meters'
        bc_par.exit_h.time_series_wse_units = unit_str

    def _read_ewsparamsrc(self, bc_par):
        """Read EWS rating curve BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.exit_h.water_surface_elevation_option = 'Rating curve'
        bc_par.exit_h.rating_curve = self._dataframe_from_xys(self._words[2].strip('"\''), ['vol/sec', 'WSE'])
        unit_str = 'cfs -vs- feet' if self._words[3] == 'EN' else 'cms -vs- meters'
        bc_par.exit_h.rating_curve_units = unit_str

    def _read_eqparams(self, bc_par):
        """Read EQ BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        if io_util.is_float(self._words[3]):
            bc_par.exit_q.discharge_option = 'Constant'
            bc_par.exit_q.constant_q = float(self._words[3])
            unit_str = 'cfs' if self._words[4] == 'EN' else 'cms'
            bc_par.exit_q.constant_q_units = unit_str
        else:
            bc_par.exit_q.discharge_option = 'Time series'
            bc_par.exit_q.time_series_q = self._dataframe_from_xys(self._words[3].strip('"\''), ['hrs', 'vol/sec'])
            unit_str = 'hrs -vs- cfs' if self._words[4] == 'EN' else 'hrs -vs- cms'
            bc_par.exit_q.time_series_q_units = unit_str

    def _read_isupcrparams(self, bc_par):
        """Read super critical BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        # If sediment is enabled, there is an extra card in the middle of this line.
        unit_idx = 4 if self._words[4].lower() == 'en' or self._words[4].lower() == 'si' else 5
        dist_idx = unit_idx + 1

        if io_util.is_float(self._words[2]):
            bc_par.inlet_sc.discharge_q_option = 'Constant'
            bc_par.inlet_sc.constant_q = float(self._words[2])
            bc_par.inlet_sc.constant_wse = float(self._words[3])
        else:
            bc_par.inlet_sc.discharge_q_option = 'Time series'
            bc_par.inlet_sc.time_series_q = self._dataframe_from_xys(
                self._words[2].strip('"\''), ['hrs', 'vol_per_sec']
            )
            rating_curve, df = self._dataframe_from_xys_rating_curve(self._words[3].strip('"\''), ['hrs', 'elev'])
            if not rating_curve:
                bc_par.inlet_sc.water_surface_elevation.water_elevation_option = 'Time series'
                bc_par.inlet_sc.water_surface_elevation.time_series_wse = df
            else:
                bc_par.inlet_sc.water_surface_elevation.water_elevation_option = 'Rating curve'
                bc_par.inlet_sc.water_surface_elevation.rating_curve = df

        if unit_idx == 5:  # Sediment is enabled
            self.logger.error(
                f'{bc_par.bc_type} encountered in simulation with sediment transport.\n'
                f'This feature is not recommended to use and is not supported in SMS.'
            )
            # bc_par.sediment_inflow.sediment_discharge_type = (
            #     'Capacity' if self._words[4].lower() == 'capacity' else 'File'
            # )
            # if bc_par.sediment_inflow.sediment_discharge_type == 'File':
            #     bc_par.sediment_inflow.sediment_file = self._words[4].strip('"\'')

        unit_str = 'English' if self._words[unit_idx] == 'EN' else 'Metric'
        bc_par.inlet_sc.discharge_q_units = unit_str

        if len(self._words) > dist_idx and self._words[dist_idx]:
            bc_par.inlet_sc.distribution_at_inlet = self._words[dist_idx].title()

    def _read_wallroughness(self, bc_par):
        """Read wall roughness BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.wall.extra_wall_roughness = True
        bc_par.wall.roughness = float(self._words[2])

    def _read_intflowtype(self, bc_par):
        """Read internal sink flow type BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        if self._words[2] == 'CONSTANT':
            bc_par.internal_sink.sink_flow_type = 'Constant'
        elif self._words[2] == 'TIMESERIES':
            bc_par.internal_sink.sink_flow_type = 'Time series'
        elif self._words[2] == 'WEIR':
            bc_par.internal_sink.sink_flow_type = 'Weir'
        elif self._words[2] == 'RATING':
            bc_par.internal_sink.sink_flow_type = 'Rating curve'

    def _read_intconstantflow(self, bc_par):
        """Read constant internal sink BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.internal_sink.constant_q = float(self._words[2])
        bc_par.internal_sink.constant_q_units = 'cfs' if self._words[3] == 'EN' else 'cms'

    def _read_inttsflow(self, bc_par):
        """Read internal sink time series BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.internal_sink.time_series_q = self._dataframe_from_xys(
            self._words[2].strip('"\''), ['hrs', 'vol_per_sec']
        )
        bc_par.internal_sink.time_series_q_units = 'hrs -vs- cfs' if self._words[3] == 'EN' else 'hrs -vs- cms'

    def _read_intweirflow(self, bc_par):
        """Read internal sink weir BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.internal_sink.weir_coeff = float(self._words[2])
        bc_par.internal_sink.weir_crest_elevation = float(self._words[3])
        bc_par.internal_sink.weir_length_across = float(self._words[4])
        bc_par.internal_sink.weir_units = 'Feet' if self._words[5] == 'EN' else 'Meters'
        bc_par.internal_sink.weir_use_total_head = True if int(self._words[6]) == 0 else False

    def _read_intrcflow(self, bc_par):
        """Read internal sink rating curve BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.internal_sink.rating_curve = self._dataframe_from_xys(
            self._words[2].strip('"\''), ['vol_per_sec', 'WSE'], swap_x_y=True
        )
        bc_par.internal_sink.rating_curve_units = 'cfs -vs- feet' if self._words[3] == 'EN' else 'cms -vs- meters'

    def _read_hy8params(self, bc_par):
        """Read HY8 BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.hy8_culvert.simulate_as_link = False
        bc_par.hy8_culvert.units = 'English' if self._words[3].lower() == 'en' else 'Metric'
        bc_par.hy8_culvert.hy8_crossing_guid = self._words[5].strip('"')
        if self._words[-1].lower() == 'head':
            bc_par.hy8_culvert.total_head = True

    def _read_culvertparams(self, bc_par):
        """Read culvert BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        # TODO: Not sure if this is right. Tutorial does not have anything on the line after entrance_type. If not
        #       using total head, think SRH ignores the rest. No such dependency in GUI though.
        self.logger.warning(
            'Culvert structure encountered in simulation. This feature is not recommended to use and '
            'should be replaced by Culvert-HY8.'
        )
        bc_par.culvert.invert_elevation = float(self._words[2])
        bc_par.culvert.barrel_height = float(self._words[4])
        bc_par.culvert.barrel_length = float(self._words[5])
        bc_par.culvert.barrel_area = float(self._words[6])
        bc_par.culvert.barrel_hydraulic_radius = float(self._words[7])
        bc_par.culvert.barrel_slope = float(self._words[8])
        bc_par.culvert.units = 'Feet' if self._words[9].lower() == 'en' else 'Meters'
        bc_par.culvert.num_barrels = int(self._words[10])
        bc_par.culvert.entrance_type = 'mitered' if float(self._words[11]) == 0.7 else 'non-mitered'
        if len(self._words) > 15:
            # this is asinine; see the original DMI xml for this list
            inlet_str = f'{self._words[12]} {self._words[13]} {self._words[14]} {self._words[15]}'
            if inlet_str in self._culvert_inlet.keys():
                bc_par.culvert.inlet_coefficients = self._culvert_inlet[inlet_str]
            bc_par.culvert.loss_coefficient = float(self._words[16])
            bc_par.culvert.mannings_n = float(self._words[17])
        # see if the last item in the line is the keyword 'HEAD'
        if self._words[-1].lower() == 'head':
            bc_par.culvert.total_head = True

    def _read_culvertweirparams(self, bc_par):
        """Read culvert weir BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.culvert.crest_elevation = float(self._words[2])
        bc_par.culvert.length_of_weir_over_culvert = float(self._words[3])
        bc_par.culvert.weir.type = self._words[4].title()
        if bc_par.culvert.weir.type == 'User':
            self._fill_weir_user_type_vals(bc_par.culvert.weir, self._words[5], self._words[6], self._words[7])

    def _read_weirparams(self, bc_par):
        """Read weir BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.weir.crest_elevation = float(self._words[2])
        bc_par.weir.length = float(self._words[4])
        bc_par.weir.units = 'Feet' if self._words[5].lower() == 'en' else 'Meters'
        bc_par.weir.type.type = self._words[6].title()
        if bc_par.weir.type.type == 'User':
            self._fill_weir_user_type_vals(bc_par.weir.type, self._words[7], self._words[8], self._words[9])
        if self._words[-1].lower() == 'head':
            bc_par.weir.total_head = True

    def _read_pressureparams(self, bc_par):
        """Read pressure BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.pressure.ceiling_type = 'Flat' if self._words[2].lower() == 'flat' else 'Parabolic'
        bc_par.pressure.upstream_elevation = float(self._words[3])
        bc_par.pressure.downstream_elevation = float(self._words[4])
        bc_par.pressure.roughness = float(self._words[5])
        bc_par.pressure.units = 'Feet' if self._words[6].lower() == 'en' else 'Meters'

    def _read_pressovertop(self, bc_par):
        """Read pressure overtopping BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.pressure.overtopping = True if self._words[2] == '1' else False

    def _read_pressweirparams(self, bc_par):
        """Read pressure weir BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.pressure.weir_type.type = self._words[5].title()
        bc_par.pressure.weir_crest_elevation = float(self._words[2])
        bc_par.pressure.weir_length = float(self._words[4])
        if bc_par.pressure.weir_type.type == 'User':
            self._fill_weir_user_type_vals(bc_par.pressure.weir_type, self._words[6], self._words[7], self._words[8])
        if self._words[-1].lower() == 'head':
            bc_par.pressure.total_head = True

    def _read_pressweirparams2(self, bc_par):
        """Read pressure weir BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.pressure.weir_type.type = self._words[5].title()
        bc_par.pressure.weir_crest_elevation = float(self._words[2])
        bc_par.pressure.weir_length = float(self._words[3])
        if bc_par.pressure.weir_type.type == 'User':
            self._fill_weir_user_type_vals(bc_par.pressure.weir_type, self._words[6], self._words[7], self._words[8])
        if self._words[-1].lower() == 'head':
            bc_par.pressure.total_head = True

    def _read_gateparams(self, bc_par):
        """Read gate BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.gate.crest_elevation = float(self._words[2])
        bc_par.gate.height = float(self._words[4])
        bc_par.gate.width = float(self._words[5])
        bc_par.gate.units = 'Feet' if self._words[6].lower() == 'en' else 'Meters'
        bc_par.gate.contract_coefficient = float(self._words[7])
        bc_par.gate.type.type = self._words[8].title()
        if bc_par.gate.type.type == 'User':
            self._fill_weir_user_type_vals(bc_par.gate.type, self._words[9], self._words[10], self._words[11])

    def _read_flow_direction(self, bc_par):
        """Read flow direction BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.flow_direction.specify_flow_direction = True
        if self._words[2].lower() != 'def':
            bc_par.flow_direction.x_direction = float(self._words[2])
        if len(self._words) > 3 and self._words[3].lower() != 'def':
            bc_par.flow_direction.y_direction = float(self._words[3])

    def _read_linkconstantflow(self, bc_par):
        """Read constant link BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.link.constant_q = float(self._words[2])
        bc_par.link.constant_q_units = 'cfs' if self._words[3].lower() == 'en' else 'cms'

    def _read_linktimeseriesflow(self, bc_par):
        """Read time series link BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.link.time_series_q = self._dataframe_from_xys(self._words[2].strip('"\''), ['hrs', 'vol_per_sec'])
        bc_par.link.time_series_q_units = 'hrs -vs- cfs' if self._words[3].lower() == 'en' else 'hrs -vs- cms'

    def _read_linkweirflow(self, bc_par):
        """Read link weir BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.link.weir.type.type = 'User'
        bc_par.link.weir.type.cw = float(self._words[2])
        bc_par.link.weir.type.a = 16.4  # default used in SRH_PRE
        bc_par.link.weir.type.b = 0.432  # default used in SRH_PRE
        bc_par.link.weir.crest_elevation = float(self._words[3])
        bc_par.link.weir.length = float(self._words[4])
        bc_par.link.weir.units = 'Feet' if self._words[5].lower() == 'en' else 'Meters'
        bc_par.link.weir.total_head = True if self._words[6] == '1' else False
        # old data structure
        # bc_par.link.weir.coeff = float(self._words[2])
        # bc_par.link.weir.crest_elevation = float(self._words[3])
        # bc_par.link.weir.length = float(self._words[4])
        # bc_par.link.weir.units = 'Feet' if self._words[5].lower() == 'en' else 'Meters'
        # bc_par.link.weir.total_head = True if self._words[6] == '1' else False

    def _read_linkrcflow(self, bc_par):
        """Read rating curve link BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.link.rating_curve = self._dataframe_from_xys(
            self._words[2].strip('"\''), ['vol_per_sec', 'WSE'], swap_x_y=True
        )
        bc_par.link.rating_curve_units = 'cfs -vs- feet' if self._words[3].lower() == 'en' else 'cms -vs- meters'

    def _read_linkflowtype(self, bc_par):
        """Read link flow type BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        look_up = {'constant': 'Constant', 'timeseries': 'Time series', 'weir': 'Weir', 'rating': 'Rating curve'}
        bc_par.link.inflow_type = look_up[self._words[2].lower()]

    def _read_linkflowtype2(self, bc_par):
        """Read link flow type BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        # from SRH_PRE
        # (1) DISCHARGE Qvalue UNIT
        # (2) RATING    Fname  UNIT [M_ID]
        # (3) WEIR ZC LW UNIT TYPE [CW] [a b] [HEAD]
        # (4) HY8  CROSSING_NAME  TABLE_NAME  [HEAD]
        if self._words[2].lower() == 'discharge':
            constant = True  # Qvalue is a number or a timeseries file
            try:
                float(self._words[3])
            except ValueError:
                constant = False
            if constant:
                bc_par.link.inflow_type = 'Constant'
                bc_par.link.constant_q = float(self._words[3])
                bc_par.link.constant_q_units = 'cfs' if self._words[4].lower() == 'en' else 'cms'
            else:
                bc_par.link.inflow_type = 'Time series'
                bc_par.link.time_series_q = self._dataframe_from_xys(
                    self._words[3].strip('"\''), ['hrs', 'vol_per_sec']
                )
                bc_par.link.time_series_q_units = 'hrs -vs- cfs' if self._words[4].lower() == 'en' else 'hrs -vs- cms'
        elif self._words[2].lower() == 'rating':
            bc_par.link.inflow_type = 'Rating curve'
            bc_par.link.rating_curve = self._dataframe_from_xys(
                self._words[3].strip('"\''), ['vol_per_sec', 'WSE'], swap_x_y=True
            )
            bc_par.link.rating_curve_units = 'cfs -vs- feet' if self._words[4].lower() == 'en' else 'cms -vs- meters'
        elif self._words[2].lower() == 'weir':
            bc_par.link.inflow_type = 'Weir'
            bc_par.link.weir.crest_elevation = float(self._words[3])
            bc_par.link.weir.length = float(self._words[4])
            bc_par.link.weir.units = 'Feet' if self._words[5].lower() == 'en' else 'Meters'
            bc_par.link.weir.type.type = self._words[6].title()
            if bc_par.link.weir.type.type == 'User':
                bc_par.link.weir.type.cw = float(self._words[7])
                bc_par.link.weir.type.a = float(self._words[8])
                bc_par.link.weir.type.b = float(self._words[9])
            if self._words[-1].lower() == 'head':
                bc_par.link.weir.total_head = True
        elif self._words[2].lower() == 'hy8':
            bc_par.bc_type = 'Culvert HY-8'
            bc_par.hy8_culvert.simulate_as_link = True
            crossing_name = self._words[3].strip('"')
            crossing_name = crossing_name.replace('-NoHY8Overtopping', '')
            hy8_file = xmsfs.resolve_relative_path(self.hydro_reader.filename, self.hydro_reader.bc_hy8_file)
            bc_par.hy8_culvert.hy8_input_file = hy8_file
            name_dict = bc_par.hy8_culvert.name_guid_dict_from_hy8_file(include_crest_length=True)
            if crossing_name in name_dict:
                bc_par.hy8_culvert.hy8_crossing_guid = name_dict[crossing_name]
            else:
                msg = f'HY-8 crossing : "{crossing_name}" not found in HY-8 file: {hy8_file}. ' \
                      f'Edit the HY-8 structure and select the appropriate crossing.'
                self.logger.error(msg)
            if self._words[-1].lower() == 'head':
                bc_par.hy8_culvert.total_head = True

    def _read_linklagtimeparameters(self, bc_par):
        """Read link flow specified lag time BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        if len(self._words) < 7:  # specified time
            bc_par.link.link_lag_method = 'Specified'
            bc_par.link.specified_lag = float(self._words[2])
        else:  # computed lag time
            bc_par.link.link_lag_method = 'Computed'
            bc_par.link.conduit_length = float(self._words[2])
            bc_par.link.conduit_diameter = float(self._words[3])
            bc_par.link.conduit_slope = float(self._words[4])
            bc_par.link.conduit_mannings = float(self._words[5])
            bc_par.link.conduit_units = 'Feet' if self._words[6].lower() == 'en' else 'Meters'

    def _read_linkspecifiedlagtime(self, bc_par):
        """Read link flow specified lag time BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.link.link_lag_method = 'Specified'
        bc_par.link.specified_lag = float(self._words[2])

    def _read_linkcomputedlagtime(self, bc_par):
        """Read link flow computed lag time BC parameters.

        Args:
            bc_par (:obj:`BcDataParam`): The BC data
        """
        bc_par.link.link_lag_method = 'Computed'
        bc_par.link.conduit_length = float(self._words[2])
        bc_par.link.conduit_diameter = float(self._words[3])
        bc_par.link.conduit_slope = float(self._words[4])
        bc_par.link.conduit_mannings = float(self._words[5])
        bc_par.link.conduit_units = 'Feet' if self._words[6].lower() == 'en' else 'Meters'
