"""Creates an Materials and Sediment Materials coverage hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shlex
import sys
import uuid

# 2. Third party modules
import pandas as pd
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.data_objects.parameters import Component
from xms.guipy.data.polygon_texture import PolygonTexture

# 4. Local modules
from xms.srh.components import material_component as mac
from xms.srh.components.sed_material_component import SedMaterialComponent
from xms.srh.data.material_data import DEFAULT_MATERIAL_COLORS


class MaterialComponentBuilder:
    """Builds hidden component for an SRH-2D Materials or Sediment Materials coverage."""
    def __init__(self, comp_dir, mat_names, poly_assignments, new_comp_uuid=None, display_atts=None):
        """Construct the builder.

        Args:
            comp_dir (:obj:`str`): Path to the directory to put component data. Should be XMS temp component
                directory if not testing.
            mat_names (:obj:`dict`): Material ID to material name
            poly_assignments (:obj:`dict`): Material ID to list of polygon IDs with that material assignment
            new_comp_uuid (:obj:`str`): UUID to use for the new component (useful for testing). Randomized if
            not provided.
            display_atts (:obj:`dict`): Material ID to display option string ('r g b texture'). If not provided,
            will be randomly generated.
        """
        self._comp_dir = comp_dir
        self._comp_uuid = new_comp_uuid if new_comp_uuid else str(uuid.uuid4())
        self._names = mat_names
        self._polys = poly_assignments
        self._display_atts = display_atts
        self._next_mat_color = 0  # For randomizing material polygon display options
        self._display_atts = display_atts if display_atts else {}

    def _get_material_poly_display(self, mat_id):
        """Get a set of randomized polygon fill display options.

        Args:
            mat_id (:obj:`int`): The material to get options for

        Returns:
            (:obj:`str`): The randomized display options as a string. Formatted as 'R G B texture', where R, G, B,
            and texture are integers.
        """
        # Check if this material already has display attributes specified.
        clr_str = self._display_atts.get(mat_id, '')
        if clr_str:
            return clr_str

        if mat_id == 0:  # Always use the same defaults for unassigned.
            return '0 0 0 1'

        if self._next_mat_color >= len(DEFAULT_MATERIAL_COLORS):
            self._next_mat_color = 0
        clr = DEFAULT_MATERIAL_COLORS[self._next_mat_color]
        # We decided that we want all the materials to have a solid pattern by default.
        clr_str = f'{clr[0]} {clr[1]} {clr[2]} {int(PolygonTexture.null_pattern)}'
        self._next_mat_color += 1
        return clr_str

    def _build_material_data(self, manningsn):
        """Create the material coverage's hidden component and data.

        Args:
            manningsn (:obj:`dict`): Material ID to either a constant Manning's N value or a list of x,y series curve
                tuples

        Returns:
            (:obj:`xarray.Dataset`): The material list dataset
        """
        # Build up the material list.
        sorted_mat_ids = list(sorted(self._names.keys()))  # Make unassigned first
        sorted_mat_names = [self._names[mat_id] for mat_id in sorted_mat_ids]
        style = [self._get_material_poly_display(mat_id) for mat_id in sorted_mat_ids]
        const_mannings = [
            manningsn[mat_id] if mat_id in manningsn and type(manningsn[mat_id]) is float else 0.02
            for mat_id in sorted_mat_ids
        ]
        depth_varied_curve = [
            0 if mat_id not in manningsn or type(manningsn[mat_id]) is float else 1 for mat_id in sorted_mat_ids
        ]
        curves = ['' for _ in range(len(self._names))]
        mat_data = {
            'id': xr.DataArray(sorted_mat_ids),
            'Color and Texture': xr.DataArray(style),
            'Name': xr.DataArray(sorted_mat_names),
            "Manning's N": xr.DataArray(const_mannings),
            'Depth Varied Curve': xr.DataArray(depth_varied_curve),
            'Curve': xr.DataArray(curves),
        }
        return xr.Dataset(data_vars=mat_data), sorted_mat_ids, depth_varied_curve

    def _write_component_id_files(self, comp_dir):
        """Write XMS feature id and component id files so we can initialize map component ids in SMS.

        Args:
            comp_dir (:obj:`str`): Path to the component data directory (folder containing the component main file)
        """
        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        att_ids = []
        comp_ids = []
        for mat_id, poly_ids in self._polys.items():
            if mat_id < 0:  # pragma no cover
                continue  # Don't need to write out unassigned polys since it is the default category.
            non_default_poly_ids = [poly_id for poly_id in poly_ids if poly_id > 0]
            att_ids.extend(non_default_poly_ids)
            comp_ids.extend([mat_id for _ in range(len(non_default_poly_ids))])
        id_file = os.path.join(comp_dir, mac.MAT_INITIAL_ATT_ID_FILE)
        write_display_option_ids(id_file, att_ids)
        id_file = os.path.join(comp_dir, mac.MAT_INITIAL_COMP_ID_FILE)
        write_display_option_ids(id_file, comp_ids)

    def build_material_component(self, cov_uuid, manningsn):
        """Create an SRH-2D Material coverage's hidden component and data_object.

        Args:
            cov_uuid (:obj:`str`): UUID of the Material coverage to build component for
            manningsn (:obj:`dict`): Material ID to either a constant Manning's N value or a list of x,y
                series curve tuples

        Returns:
            (:obj:`xms.data_objects.parameters.Component`): data_object for the new MaterialComponent to
            send back to SMS.
        """
        # Create a new UUID and folder for the component data
        mat_comp_dir = os.path.join(self._comp_dir, self._comp_uuid)
        os.makedirs(mat_comp_dir, exist_ok=True)
        mat_main_file = os.path.join(mat_comp_dir, 'mat_comp.nc')

        # Create the data_object Component to send back to SMS
        do_comp = Component(
            comp_uuid=self._comp_uuid, main_file=mat_main_file, model_name='SRH-2D', unique_name='Material_Component'
        )

        mat_comp = mac.MaterialComponent(mat_main_file)
        mat_data, sorted_mat_ids, depth_varied_curve = self._build_material_data(manningsn)
        mat_comp.data.set_materials(mat_data)
        mat_comp.data.info.attrs['cov_uuid'] = cov_uuid
        mat_comp.cov_uuid = cov_uuid

        # Build up the depth curve datasets
        for idx, mat_id in enumerate(sorted_mat_ids):
            if depth_varied_curve[idx]:
                curve_data = {
                    'Depth': xr.DataArray([row[0] for row in manningsn[mat_id]]),
                    "Manning's N": xr.DataArray([row[1] for row in manningsn[mat_id]]),
                }
                mat_comp.data.set_depth_curve(mat_id, xr.Dataset(data_vars=curve_data))

        mat_comp.data.commit()
        mat_comp.update_display_id_files([], sorted_mat_ids)
        mat_comp.update_display_options_file()

        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        self._write_component_id_files(mat_comp_dir)

        return do_comp

    def build_sed_material_component(self, cov_uuid, sed_mat_data):
        """Create an SRH-2D Sediment Material coverage's hidden component and data.

        Args:
            cov_uuid (:obj:`str`): UUID of the Sediment Material coverage to build component for
            sed_mat_data (:obj:`list`): List of the sediment material data.

                Each element should either be a string in
                srhhydro format ('SUBSURFACETHICKNESS mat_id layer_id thickness units_card density')

                OR a tuple of following format: (mat_id, layer_id, [(x1, y1), ..., (xN, yN)])
                for time-varying gradation curve layers. NUMSUBSURFACELAYERS cards in string format are ignored.

        Returns:
            (:obj:`xms.data_objects.parameters.Component`): data_object for the new SedMaterialComponent to send
            back to SMS.
        """
        # Create a new UUID and folder for the component data
        mat_comp_dir = os.path.join(self._comp_dir, self._comp_uuid)
        os.makedirs(mat_comp_dir, exist_ok=True)
        mat_main_file = os.path.join(mat_comp_dir, 'sed_mat_comp.nc')

        # Create the data_object Component to send back to SMS
        do_comp = Component(
            comp_uuid=self._comp_uuid,
            main_file=mat_main_file,
            model_name='SRH-2D',
            unique_name='SedMaterial_Component'
        )

        # put material properties into the data class
        mat_comp = SedMaterialComponent(mat_main_file)
        mat_data, sorted_mat_ids, _ = self._build_material_data({})
        mat_comp.data.set_materials(mat_data)
        mat_comp.data.info.attrs['cov_uuid'] = cov_uuid
        mat_comp.cov_uuid = cov_uuid

        # get the sediment properties
        sed_prop = {}
        for line in sed_mat_data:
            if type(line) is str:
                words = shlex.split(line, posix="win" not in sys.platform)
                if words[0].lower() == 'numsubsurfacelayers':
                    continue
                mat_id = int(words[1])
                lay_id = int(words[2])
                thick = float(words[3])
                units = 'English' if words[4].lower() == 'en' else 'SI'
                density = 0.0
                # Change for SRH 3.3.0 - this is not used any more
                # if len(words) > 5:
                #     density = float(words[5])
                record = {
                    'att_id': lay_id,
                    'Thickness': thick,
                    'Units': units,
                    'Density': density,
                    'Gradation Curve': ''
                }
                if mat_id not in sed_prop:
                    sed_prop[mat_id] = []
                sed_prop[mat_id].append(record)
            else:
                mat_id = line[0]
                lay_id = line[1]
                curve = line[2]
                xy_list = list(zip(*curve))
                curve_data = {
                    'Particle diameter (mm)': xr.DataArray(list(xy_list[0])),
                    'Percent finer': xr.DataArray(list(xy_list[1])),
                }
                dset = xr.Dataset(data_vars=curve_data)
                mat_comp.data.set_gradation_curve(mat_id, lay_id, dset)

        for key, val in sed_prop.items():
            mat_comp.data.set_sediment_properties(key, pd.DataFrame(val).to_xarray())

        mat_comp.data.commit()
        mat_comp.update_display_id_files([], sorted_mat_ids)
        mat_comp.update_display_options_file()

        # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
        self._write_component_id_files(mat_comp_dir)

        return do_comp
