"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.component_with_menus_base import MessagesAndRequests
from xms.gmi.components2.gmi_sim_component_base import GmiSimComponentBase
from xms.gmi.data.generic_model import Section
from xms.gmi.gui.section_dialog import SectionDialog

# 4. Local modules
from xms.srh_swmm_coupler.data.model import Model
from xms.srh_swmm_coupler.tools import SrhInternalSinkFlowsFromSwmmInletPipeFlows, SwmmInletFlowsFromSrhMonitorLines


class SimComponent(GmiSimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for the SWMM model simulation."""

    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self._srh_monitor_cov = None
        self._swmm_nodes_cov = None
        _ = self.data
        self._model = None
        self.tree_commands.append(('SWMM Inlet Flows from Monitor Line Flows...',
                                   self._swmm_inlet_flows_from_srh_monitor_lines))
        self.tree_commands.append(('SRH Internal Sink Flows from SWMM Inlet Pipe Flows...',
                                   self._srh_internal_sink_flows_from_swmm_inlet_pipe_flows))
        self.tree_commands.reverse()

    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        return self._model.get_coupled_model().global_parameters

    def _open_model_control(self, dialog_name: str, query: Query, parent: QWidget):
        """
        Run the model control dialog.

        Args:
            dialog_name: A name that can be used as a registry key for loading and saving dialog settings, such as its
                geometry and active selections.
            query: Interprocess communication object.
            parent: Parent widget for the dialog.
        """
        dialog_name = f'{self.module_name}.model_control'
        self._model = Model(query)
        section = self._get_global_parameter_section()
        values = self.data.global_values
        if values:
            section.restore_values(values)
        # Get current table values
        table_param = section.group('general').parameter('srh_swmm_map')
        table = table_param.value
        # Get inlet nodes from SWMM simulation
        inlets = self._model.inlets
        default_sink = self._model.default_sink
        default_monitor = self._model.default_monitor
        # Remove any inlets that no longer exist in the SWMM model
        table = [values for values in table if values[0] in inlets]
        new_table = table.copy()
        # Add any inlets that are not in the existing table
        for inlet in inlets:
            in_table = False
            for values in table:
                if values[0] == inlet:
                    in_table = True
            if not in_table:
                new_table.append([inlet, default_sink, default_monitor])
        table_param.value = new_table
        dlg = SectionDialog(
            parent=parent,
            section=section,
            dlg_name=dialog_name,
            window_title='Model control',
            enable_unchecked_groups=True,
            hide_checkboxes=True
        )
        if dlg.exec():
            values = dlg.section.extract_values()
            self.data.global_values = values
            self.data.commit()

    def _swmm_inlet_flows_from_srh_monitor_lines(self, query: Query, _params: list[dict],
                                                 parent: QWidget) -> MessagesAndRequests:
        """Converts SRH monitor line flows to SWMM inlet flows based on model control parameters for this simulation.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests
        """
        self._model = Model(query)
        section = self._get_global_parameter_section()
        values = self.data.global_values
        if values:
            section.restore_values(values)
        # Get current table values
        table_param = section.group('general').parameter('srh_swmm_map')
        table = table_param.value
        damping_factor = section.group('general').parameter('damping_factor').value
        tool = SwmmInletFlowsFromSrhMonitorLines(query, table, damping_factor)
        tool.run()
        return self.messages, self.requests

    def _srh_internal_sink_flows_from_swmm_inlet_pipe_flows(self, query: Query, _params: list[dict],
                                                            parent: QWidget) -> MessagesAndRequests:
        """Converts SRH monitor line flows to SWMM inlet flows based on model control parameters for this simulation.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests
        """
        self._model = Model(query)
        section = self._get_global_parameter_section()
        values = self.data.global_values
        if values:
            section.restore_values(values)
        # Get current table values
        table_param = section.group('general').parameter('srh_swmm_map')
        table = table_param.value
        damping_factor = section.group('general').parameter('damping_factor').value
        tool = SrhInternalSinkFlowsFromSwmmInletPipeFlows(query, table, damping_factor)
        tool.run()
        return self.messages, self.requests
