"""Determines how to run an STWAVE simulation and read the solution."""

# 1. Standard Python modules
import datetime
import logging
import os
import re

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, Query
from xms.components.bases.run_base import RunBase
from xms.data_objects.parameters import Point, RectilinearGrid
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.stwave.data import stwave_consts as const
from xms.stwave.data.simulation_data import SimulationData
from xms.stwave.feedback.solution_import_runner import read_solution_feedback
from xms.stwave.file_io.dataset_reader import DatasetReaderSTWAVE
from xms.stwave.file_io.eng_reader import EngReader


class SimulationRun(RunBase):
    """A class that tells SMS how to run STWAVE and load its solution."""

    def __init__(self, dummy_mainfile=''):
        """Constructor."""
        super().__init__()
        self.query = None
        self.geom_uuid = 'CCCCCCCC-CCCC-CCCC-CCCC-CCCCCCCCCCCC'
        self.sim_name = 'Sim'
        self.filename = ''
        self.eng_reader = None
        self.arg_list = []
        self.read_data = False
        self.rect_grid = None
        self.half_plane = False
        self.ref_time = None
        self.single_solution_file = None
        self._logger = logging.getLogger('xms.stwave')

    def setup_query(self):
        """Sets timeout, retries, root context, and root index for the query member."""
        self.query = Query()
        sim_item = self.query.current_item()
        self.sim_name = sim_item.name
        if not self.sim_name:
            return  # not called from a simulation
        comp_item = self.query.item_with_uuid(sim_item.uuid, unique_name='Sim_Component', model_name='STWAVE')
        sim_data = SimulationData(comp_item.main_file)
        self.ref_time = datetime.datetime.strptime(sim_data.info.attrs['reftime'], ISO_DATETIME_FORMAT)
        self.half_plane = sim_data.info.attrs['plane'] == const.PLANE_TYPE_HALF

    def read_grid_definition(self):
        """
        Read the grid definition from the .sim input file.

        Don't want to read the whole file, quit once we have what we need.
        """
        # Look for a .sim file in the same directory with the same base name. If it exists, grab the grid definition
        # from it.
        sim_in = f'{self.sim_name}.sim'
        if os.path.isfile(sim_in):
            with open(sim_in, 'r') as sim_file:
                grid_params = {
                    'angle': None,
                    'dx': None,
                    'dy': None,
                    'x0': None,
                    'y0': None,
                    'n_cell_i': None,
                    'n_cell_j': None,
                }

                for line in sim_file:
                    if None not in grid_params.values():
                        break  # have everything we need, stop reading the file

                    # Skip comments and blank lines
                    line = line.strip().replace("\'", '')
                    if not line or line[0] == '#' or line[0] == '&' or line[0] == '/':
                        continue

                    line_data = [x.strip() for x in line.split('=') if x != '']
                    # remove comma at end of the line
                    line_data[len(line_data) - 1] = line_data[len(line_data) - 1].replace(',', '')
                    token = line_data[0].lower()

                    if token == 'x0':
                        grid_params['x0'] = float(line_data[1])
                    elif token == 'y0':
                        grid_params['y0'] = float(line_data[1])
                    elif token == 'azimuth':
                        grid_params['azimuth'] = float(line_data[1])
                    elif token == 'dx':
                        grid_params['dx'] = float(line_data[1])
                    elif token == 'dy':
                        grid_params['dy'] = float(line_data[1])
                    elif token == 'n_cell_i':
                        grid_params['n_cell_i'] = int(line_data[1])
                    elif token == 'n_cell_j':
                        grid_params['n_cell_j'] = int(line_data[1])

                self.rect_grid = RectilinearGrid()
                if grid_params['x0'] is not None and grid_params['y0'] is not None:
                    self.rect_grid.origin = Point(grid_params['x0'], grid_params['y0'])
                if grid_params['azimuth'] is not None:
                    self.rect_grid.angle = grid_params['azimuth']
                if grid_params['dx'] is not None and grid_params['dy'] is not None:
                    num_i = 1  # Really only need dx and dy
                    num_j = 1
                    if grid_params['n_cell_i'] is not None:
                        num_i = grid_params['n_cell_i']
                    if grid_params['n_cell_j'] is not None:
                        num_j = grid_params['n_cell_j']
                    self.rect_grid.set_sizes(num_i, grid_params['dx'], num_j, grid_params['dy'])

    def add_spectral_coverages(self):
        """Reads specral coverage from file_name, and adds it to a spectral dictionary that is sent to query."""
        filename_lower = self.filename.lower()

        if filename_lower.endswith('obse.out'):
            cover_name = f'{self.sim_name} - obse'
        elif filename_lower.endswith('nest.out'):
            cover_name = f'{self.sim_name} - nest'
        elif filename_lower.endswith('eng'):
            if self.sim_name == '':
                # Parse out the project name from the read file.
                cover_name = os.path.basename(self.filename)
            else:
                cover_name = f'{self.sim_name} - ENG'
        else:
            return  # This is not a spectral coverage output file. Keep looking.

        # If we are reading a solution spectral coverage, we need to read the STWAVE grid definition. STWAVE output
        # spectral files specify node locations by i-j coordinates on the input STWAVE grid. We cannot read this
        # solution if we don't have a matching .sim file in the same directory.

        # read the output spectral coverage
        self._logger.info(f'Reading output spectral file: {self.filename}')
        reader = EngReader(self.filename, {}, rect_grid=self.rect_grid, reftime=self.ref_time)
        self.eng_reader = reader
        out_spec_cov, dum1, dum2, dum3 = reader.read()
        if out_spec_cov:
            out_spec_cov.m_cov.name = cover_name
            self.query.add_coverage(out_spec_cov)
            self.read_data = True

    def add_grid_datasets(self):
        """Adds the dataset in file to query. Could be a wave, break, radiation stress, or ends with .tp.out."""
        ts_times = []
        dset_reader = None

        grid_angle = 0.0
        if self.rect_grid:
            grid_angle = self.rect_grid.angle

        # read the wave output dataset. Will have wave.out_max extension if a CSTORM output
        if self.filename.lower().endswith('wave.out') or self.filename.lower().endswith('wave.out_max'):
            self._logger.info(f'Reading output wave file: {self.filename}')
            dset_reader = DatasetReaderSTWAVE('WAVE', self.filename, ts_times, self.geom_uuid, grid_angle,
                                              reftime=self.ref_time)
            dsets = dset_reader.read_wave_file()
            if dsets:
                for dset in dsets:
                    if dset:
                        self.query.add_dataset(dset)
                        self.read_data = True
            return  # We are done if this is the wave dataset. Special case.
        elif self.filename.lower().endswith('.break.out'):
            # read the breaking output dataset
            dset_reader = DatasetReaderSTWAVE('BREAK', self.filename, ts_times, self.geom_uuid, grid_angle,
                                              reftime=self.ref_time)
        elif self.filename.lower().endswith('.rads.out'):
            # read the radiation stress output dataset
            dset_reader = DatasetReaderSTWAVE('RADS', self.filename, ts_times, self.geom_uuid, grid_angle,
                                              reftime=self.ref_time)
        elif self.filename.lower().endswith('.tp.out'):
            dset_reader = DatasetReaderSTWAVE('1 div fma', self.filename, ts_times, self.geom_uuid, grid_angle,
                                              reftime=self.ref_time)

        if dset_reader:
            self._logger.info(f'Reading output grid dataset file: {self.filename}')
            dset = dset_reader.read()
            if dset:
                self.query.add_dataset(dset)
                self.read_data = True

    def read(self):
        """Reads spectral coverage and grid definition and sends them to query."""
        if not os.path.isfile(self.filename):
            return  # Not a real file

        # Parse out the project name from the read file.
        basename = os.path.basename(self.filename)
        m = re.search(r'(.+)\..+\.out', basename)
        if m:
            self.sim_name = m.group(1)
        else:
            self.sim_name = ''

        # STWAVE uses i-j locations of the input grid to specify points in output spectra. This is opposed to STWAVE
        # input spectral files, which specify point locations with x,y coordinates. To read solution spectra, we need
        # the grid definition. Also need the grid angle for reading grid solution datasets.
        self.read_grid_definition()

        # read and add the spectral coverages first
        self.add_spectral_coverages()
        if self.read_data:
            return  # This was an output spectral coverage. No need to go on.

        # add the solution dataset
        self.add_grid_datasets()

    def get_executables(self, sim, query, filelocation):
        """
        Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.


        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.api.dmi.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ExecutableCommand`):
                The executable objects to run and the action requests that go with it.
        """
        # Get the simulation hidden component
        do_comp = query.item_with_uuid(sim.uuid, model_name='STWAVE', unique_name='Sim_Component')

        # Setup the solution load ActionRequest
        load_sol = self._create_solution_load_action(filelocation, do_comp.main_file)

        sim_data = SimulationData(do_comp.main_file)
        # If half plane, N_GRD_PART_I must be 1
        n_grd_part_i = 1 if sim_data.info.attrs['plane'] == const.PLANE_TYPE_HALF else \
            sim_data.info.attrs['processors_i']
        n_grd_part_j = sim_data.info.attrs['processors_j']
        num_processors = n_grd_part_i * n_grd_part_j
        if num_processors > 1:  # Run the parallel executable
            cmd = ExecutableCommand(executable='MPIEXEC', executable_order=0, display_name='STWAVE-Parallel',
                                    run_weight=100, progress_script='xml_entry_points/simulation_progress.py')
            cmd.add_commandline_arg('--np')
            cmd.add_commandline_arg(f'{num_processors}')
            cmd.add_commandline_arg('--localonly')
            cmd.add_commandline_arg(f'{query.named_executable_path("STWAVEParallelExec")}')
        else:
            cmd = ExecutableCommand(executable='STWAVE', model='STWAVE', executable_order=0, display_name='STWAVE',
                                    run_weight=100, progress_script='xml_entry_points/simulation_progress.py')
        cmd.add_commandline_arg(f'{sim.name}.sim')
        cmd.add_solution_file(load_sol)
        commands = [cmd]

        return commands

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`xms.api.dmi.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): The solution load ActionRequests for the simulation
        """
        sim_item = query.item_with_uuid(sim.uuid, model_name='STWAVE', unique_name='Sim_Component')
        return [self._create_solution_load_action(file_location=filelocation, sim_comp_mainfile=sim_item.main_file)]

    def _create_solution_load_action(self, file_location, sim_comp_mainfile):
        """Create an ActionRequest to read a solution.

        Args:
            file_location (str): Path to the project export folder
            sim_comp_mainfile (str): Path to the simulation hidden component's main file

        Returns:
            ActionRequest: The solution load ActionRequest
        """
        return ActionRequest(main_file=os.path.join(file_location, 'fort.15'), modality='MODAL',
                             class_name='SimulationRun', module_name='xms.stwave.simulation_runner.simulation_run',
                             method_name='read_solution', parameters={'sim_mainfile': sim_comp_mainfile})

    def _callback_read_solution(self):
        """Callback to read solution."""
        self._logger.info('Loading STWAVE simulation run solution files...')
        try:
            sim_name = self.query.parent_item().name
            if self.single_solution_file:
                filenames = [self.single_solution_file]
            else:
                filenames = [f'{sim_name}.obse.out', f'{sim_name}.nest.out', f'{sim_name}.wave.out',
                             f'{sim_name}.break.out', f'{sim_name}.rads.out', f'{sim_name}.tp.out']
            read_any_data = False
            for file in filenames:
                self.filename = file
                self.read_data = False
                self.read()
                read_any_data = read_any_data or self.read_data
            if not read_any_data:
                self._logger.error('No solution files found. Ensure the model has been run.')
        except Exception:
            self._logger.exception('Error(s) encountered while loading STWAVE solution files.')

    def read_solution(self, query, params, win_cont):
        """Reads the STWAVE Solution.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): a Query object to communicate with GMS.
            params (:obj:`dict`): Generic map of parameters. Contains the structures for various components that
             are required for adding vertices to the Query Context with Add().
            win_cont (QWidget): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Find out what files we should load as part of this run
        self.query = query
        read_solution_feedback(self, win_cont)
        return [], []
