"""ContoursFromSingleRasterValueTool whitebox tool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.vectors import VectorInput
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.utilities.coverage_conversion import convert_lines_to_coverage

ARG_INPUT_RASTER = 0
ARG_CONTOUR_VALUE = 1
ARG_OUTPUT_COVERAGE = 2


class ContoursFromSingleRasterValueTool(Tool):
    """ContoursFromSingleRasterValueTool whitebox tool class."""

    def __init__(self):
        """Initializes the class."""
        super().__init__('Contours from Single Raster Value')
        self._raster = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.raster_argument(name='input_raster', description='Input raster'),
            self.float_argument(name='contour_value', description='Contour Value', value=0.0),
            self.coverage_argument(name='output_coverage', description='Output coverage',
                                   io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        self._raster = self.get_input_raster(arguments[ARG_INPUT_RASTER].value)
        z_min, z_max = self._raster.get_z_min_max()
        contour_value = arguments[ARG_CONTOUR_VALUE].value
        if not z_min < contour_value < z_max:
            errors['contour_value'] = ('The contour elevation needs to be set to a value within the range of '
                                       f'elevations for the raster. {contour_value} is not within the range of the '
                                       'raster elevations.')
        return errors

    def run(self, arguments):
        """Runs the tool.

        Args:
            arguments (list): A list of the tool's arguments.
        """
        # shp_filename = 'c:/temp/contours.shp'
        shp_filename = '/vsimem/contours.shp'
        value = arguments[ARG_CONTOUR_VALUE].value
        ru.raster_to_shapefile_contours(self._raster, [value], shp_filename)
        vi = VectorInput(shp_filename)
        cov_geometry = vi.get_line_features()
        out_cov = arguments[ARG_OUTPUT_COVERAGE].value
        new_cov = convert_lines_to_coverage(cov_geometry, out_cov, self._raster.wkt, self.default_wkt)
        if new_cov is not None:
            self.set_output_coverage(new_cov, arguments[ARG_OUTPUT_COVERAGE])
