"""DataHandler class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import glob
import os
from pathlib import Path
from typing import Dict, List, Optional, Union

# 2. Third party modules
from geopandas import GeoDataFrame

# 3. Aquaveo modules
from xms.constraint import Grid, read_grid_from_file, read_grid_uuid_from_file
from xms.core.filesystem import filesystem
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter
from xms.gdal.rasters import raster_utils
from xms.gdal.rasters import RasterInput

# 4. Local modules
from .coverage_reader import CoverageReader
from .coverage_writer import CoverageWriter
from .dataset_filters import dataset_included


def _read_coverage(file_path: str) -> GeoDataFrame:
    """Read a coverage.

    Args:
        file_path (str): The file path.

    Returns:
        (GeoDataFrame): The coverage.
    """
    reader = CoverageReader(file_path)
    return reader.read()


def _read_coverage_uuid(file_path: str) -> str:
    """Read a coverage.

    Args:
        file_path (str): The file path.

    Returns:
        (str): The coverage UUID as a string.
    """
    reader = CoverageReader(file_path)
    return reader.read_uuid()


class DataHandler:
    """Class to provide data to the tool such as a list of UGrids or datasets."""
    file_folder: Optional[str]

    def __init__(self, file_folder: Optional[str] = None):
        """Construct a DataHandler object."""
        self.file_folder = file_folder
        self.output_folder = None
        self.check_duplicate_output = False
        self._grid_files = None
        self._dataset_files = None
        self._coverage_files = None
        self._grid_uuid_to_name = {}  # For testing. Map between grid UUID and grid name (filename).
        self._coverage_name_to_uuids = {}
        self._raster_files = None
        self._default_wkt = None
        self._vertical_datum = None
        self._vertical_units = None
        self._uses_file_system = True
        self.check_structured = True

    @property
    def uses_file_system(self) -> bool:
        """Access private _uses_file_system."""
        return self._uses_file_system

    def get_available_grids(self) -> List[str]:
        """Get a list of the available grids.

        Returns:
            (List[str]): A list of the available grids.
        """
        if self._grid_files is None:
            self._load_grid_files()
            if self._grid_files is None:
                return []

        grids = list(self._grid_files.keys())
        grids.sort()
        return grids

    def get_available_datasets(self, filters: Optional[set] = None) -> List[str]:
        """Returns a list of the available datasets.

        Args:
            filters (Optional[list]: The filters to apply to the returned list.

        Returns:
            (List[str]): A list of the available datasets.
        """
        if self._dataset_files is None:
            self._load_dataset_files()

        datasets = []
        for dset_name in self._dataset_files.keys():
            dset = self.get_input_dataset(dset_name)
            included = dataset_included(dset, filters)
            if included:
                datasets.append(dset_name)
        datasets.sort()
        return datasets

    def get_grid_datasets(self, grid: str, filters: Optional[list] = None) -> List[str]:
        """Returns a list of datasets for an input grid.

        Args:
            grid (str): The grid.
            filters (Optional[list]: The filters to apply to the returned list.

        Returns:
            (List[str]): A list of the grid's datasets.
        """
        grid_datasets = []
        if grid:
            datasets = self.get_available_datasets(filters)
            for dset_name in datasets:
                if dset_name.find(grid + '/') == 0:
                    grid_datasets.append(dset_name)
        return grid_datasets

    def get_available_coverages(self, filters: Optional[Dict[str, str]] = None) -> List[str]:
        """Get a list of the available coverages.

        Args:
            filters (Optional[Dict[str, str]]): The filters to apply to the returned list. Unused in this
                implementation. All coverage files are available in tests. In the GUI, we will filter on the tree items.

        Returns:
            (List[str]): A list of the available coverages.
        """
        if self._coverage_files is None:
            self._load_coverage_files()

        if self._coverage_files is not None:
            coverages = list(self._coverage_files.keys())
        else:
            coverages = []
        coverages.sort()
        return coverages

    def coverage_exists(self, coverage_name: str) -> bool:
        """Check if a coverage name already exists in the tree."""
        return coverage_name in self.get_available_coverages()

    def get_available_rasters(self) -> List[str]:
        """Get a list of the available rasters.

        Returns:
            (list[str]): A list of the available rasters.
        """
        if self._raster_files is None:
            self._load_raster_files()

        rasters = list(self._raster_files.keys())
        rasters.sort()
        return rasters

    def get_input_grid(self, grid_name: str) -> Optional[Grid]:
        """Get a grid chosen from the list of available grids (self.available_grids).

        Args:
            grid_name (str): The grid to retrieve.

        Returns:
            (Grid): The grid object.
        """
        if self.file_folder is None:
            if not os.path.isfile(grid_name):
                return None
            return self._read_grid(grid_name)
        if self._grid_files is None:
            self._load_grid_files()
        if grid_name not in self._grid_files:
            return None
        grid_file = self._grid_files[grid_name]
        return self._read_grid(grid_file)

    def get_grid_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input grid name from a UUID.

        Args:
            uuid: The grid UUID.

        Returns:
            The grid name or None.
        """
        if self._grid_files is None:
            self._load_grid_files()

        for grid_name, grid_file in self._grid_files.items():
            grid_uuid = read_grid_uuid_from_file(grid_file)
            if grid_uuid == uuid:
                return grid_name
        return None

    def get_uuid_from_grid_name(self, grid_name: str) -> Optional[str]:
        """Get a UUID from an input grid name.

        Args:
            grid_name: The grid name.

        Returns:
            The grid UUID or None.
        """
        if not self._grid_uuid_to_name:
            self._load_grid_files()

        for cur_uuid, name in self._grid_uuid_to_name.items():
            if name == grid_name:
                return cur_uuid
        return None

    def set_output_grid(self, grid: Grid, grid_name: str, projection: str, force_ugrid: bool = True) -> None:
        """Set output grid for argument.

        Args:
            grid (Grid): The output grid.
            grid_name (str): The name of the new grid. For testing the grid name is the output
                file path in the "files" folder.
            projection (str): The WKT of the grid.
            force_ugrid (bool): Whether to save the output as an UGrid in SMS.
        """
        if self.file_folder is None and self.output_folder is None:
            file_path = grid_name
        else:
            test_path = self._get_output_folder()
            if self.file_folder_structured():
                file_path = os.path.join(test_path, 'grids', grid_name + '.xmc')
                if self._grid_files is not None:
                    self._grid_files[grid_name] = file_path
                self._grid_uuid_to_name[grid.uuid] = grid_name
            else:
                file_path = os.path.join(test_path, grid_name)
                if self._grid_files is not None:
                    self._grid_files[grid_name] = file_path
        Path(file_path).parent.mkdir(parents=True, exist_ok=True)
        grid.write_to_file(file_path, binary_arrays=False)

    def _load_grid_files(self) -> None:
        """Load grid files available for testing.

        Loads self._grid_files dictionary containing user grid name as key and file path as value.
        """
        if self.file_folder is None:
            return
        if self.file_folder_structured():
            grid_files = glob.glob(self.file_folder + '/grids/**/*.xmc', recursive=True)
        else:
            grid_files = glob.glob(self.file_folder + '/**/*.xmc', recursive=True)
        self._grid_files = {}
        for file_path in grid_files:
            if self.file_folder_structured():
                relative_file_path = os.path.relpath(file_path, os.path.join(self.file_folder, 'grids'))
            else:
                relative_file_path = os.path.relpath(file_path, self.file_folder)
            name, _ = os.path.splitext(relative_file_path)
            name.replace('\\', '/')
            self._grid_files[name] = file_path
        if self.file_folder_structured():
            for name in self._grid_files.keys():
                grid = self.get_input_grid(name)
                if grid is not None:
                    self._grid_uuid_to_name[grid.uuid] = name

    def _get_coverage_file(self, coverage_name: str) -> str | None:
        if self.file_folder is None:
            if not os.path.isfile(coverage_name):
                return None
            coverage_file = coverage_name
        else:
            if self._coverage_files is None:
                self._load_coverage_files()
            if coverage_name not in self._coverage_files:
                return None
            coverage_file = self._coverage_files[coverage_name]
        return coverage_file

    def get_input_coverage_uuid(self, coverage_name: str) -> str | None:
        """Get a coverage chosen from the list of available coverages (self.available_coverages).

        Args:
            coverage_name (str): The coverage to retrieve.

        Returns:
            (str): The coverage UUID as a string.
        """
        coverage_file = self._get_coverage_file(coverage_name)
        if coverage_file is not None:
            return _read_coverage_uuid(coverage_file)
        return None

    def get_input_coverage(self, coverage_name: str) -> GeoDataFrame | None:
        """Get a coverage chosen from the list of available coverages (self.available_coverages).

        Args:
            coverage_name (str): The coverage to retrieve.

        Returns:
            (GeoDataFrame): The coverage object as a pandas GeoDataFrame containing its attributes.
        """
        coverage_file = self._get_coverage_file(coverage_name)
        if coverage_file is not None:
            return _read_coverage(coverage_file)
        return None

    def get_input_coverage_file(self, coverage_name: str) -> str:
        """Get a coverage file chosen from the list of available coverages (self.available_coverages).

        Args:
            coverage_name: The coverage to retrieve.

        Returns:
            The coverage file path.
        """
        coverage_file = None
        if self.file_folder is None:
            coverage_file = coverage_name
        else:
            if self._coverage_files is None:
                self._load_coverage_files()
            if coverage_name in self._coverage_files:
                coverage_file = self._coverage_files[coverage_name]
        return coverage_file

    def get_coverage_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input coverage from a UUID.

        Args:
            uuid: The coverage UUID.

        Returns:
            The coverage name or None.
        """
        if self._coverage_files is None:
            self._load_coverage_files()

        for coverage_name in self._coverage_files.keys():
            coverage_uuid = self.get_input_coverage_uuid(coverage_name)
            if coverage_uuid is not None:
                if coverage_uuid == uuid:
                    return coverage_name
        return None

    def get_uuid_from_coverage_name(self, coverage_name: str) -> Optional[str]:
        """Get a UUID from an input coverage.

        Args:
            coverage_name: The coverage name.

        Returns:
            The UUID or None.
        """
        if coverage_name in self._coverage_name_to_uuids:
            return self._coverage_name_to_uuids[coverage_name]
        return self.get_input_coverage_uuid(coverage_name)

    def set_output_coverage(self, coverage: GeoDataFrame, coverage_name: str) -> None:
        """Set output grid for argument.

        Args:
            coverage (GeoDataFrame): The output coverage.
            coverage_name (str): The coverage argument whose value is the coverage name. For testing the coverage name
                is the output file path in the "files" folder.
        """
        test_path = self._get_output_folder()
        file_path = os.path.join(test_path, 'coverages', coverage_name + '.h5')
        writer = CoverageWriter(file_path, self.get_default_wkt())
        writer.write(coverage)

    def _load_coverage_files(self) -> None:
        """Load coverage files available for testing.

        Loads self._coverage_files dictionary containing user grid name as key and file path as value.
        """
        if self.file_folder is None:
            return
        coverage_files = glob.glob(self.file_folder + '/coverages/**/*.h5', recursive=True)
        self._coverage_files = {}
        for file_path in coverage_files:
            relative_file_path = os.path.relpath(file_path, os.path.join(self.file_folder, 'coverages'))
            name, _ = os.path.splitext(relative_file_path)
            name.replace('\\', '/')
            self._coverage_files[name] = file_path
        for name in self._coverage_files.keys():
            cov_uuid = self.get_input_coverage_uuid(name)
            if cov_uuid is not None:
                self._coverage_name_to_uuids[name] = cov_uuid

    def get_input_raster(self, raster_name: str) -> Optional[RasterInput]:
        """Get a raster chosen from the list of available rasters (self.available_rasters).

        Args:
            raster_name (str): The raster to retrieve.

        Returns:
            (RasterInput): The raster object.
        """
        if self.file_folder is None:
            raster_file = raster_name
        else:
            raster_file = self.get_input_raster_file(raster_name)
        try:
            return self.load_raster_file(raster_file)
        except Exception:
            return None

    def get_input_raster_file(self, raster_name: str) -> str:
        """Get a raster file path chosen from the list of available rasters (self.available_rasters).

        Args:
            raster_name (str): The raster to retrieve.

        Returns:
            (str): The raster file path.
        """
        raster_file = None
        if self.file_folder is None:
            raster_file = raster_name
        else:
            if self._raster_files is None:
                self._load_raster_files()
            if raster_name in self._raster_files:
                raster_file = self._raster_files[raster_name]
        return raster_file

    def get_raster_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input name from a UUID.

        Args:
            uuid: The raster UUID.

        Returns:
            The input raster name or None.
        """
        if self._raster_files is None:
            self._load_raster_files()

        for raster_name, raster_file in self._raster_files.items():
            uuid_file = os.path.splitext(raster_file)[0] + '.uuid'
            raster_uuid = None
            if os.path.isfile(uuid_file):
                with open(uuid_file) as f:
                    raster_uuid = f.readline().rstrip()
            if raster_uuid == uuid:
                return raster_name
        return None

    def get_uuid_from_raster_name(self, raster_name: str) -> Optional[str]:
        """Get a UUID from a raster name.

        Args:
            raster_name: The raster name.

        Returns:
            The UUID or None.
        """
        if self._raster_files is None:
            self._load_raster_files()

        raster_uuid = None
        if raster_name in self._raster_files:
            uuid_file = os.path.splitext(self._raster_files[raster_name])[0] + '.uuid'
            if os.path.isfile(uuid_file):
                with open(uuid_file) as f:
                    raster_uuid = f.readline().rstrip()
        return raster_uuid

    def load_raster_file(self, raster_file: str) -> RasterInput:
        """Load a raster file.

        Args:
            raster_file (str): The raster file path.

        Returns:
            (RasterInput): The raster input.
        """
        if raster_file is None:
            return None
        return RasterInput(raster_file)

    def get_output_raster(self, raster_name: str, raster_extension: str = '.tif') -> str:
        """Get path to write output raster.

        Args:
            raster_name (str): The raster argument who's value is the raster name. For testing the raster name is the
                output file path in the files folder.
            raster_extension: The file extension to add to the raster_name.

        Returns:
            (str): The raster file path.
        """
        if self.file_folder is None:
            file_path = raster_name
        else:
            test_path = self._get_output_folder()
            if self.file_folder_structured():
                file_path = os.path.join(test_path, 'rasters', f'{raster_name}{raster_extension}')
            else:
                file_path = os.path.join(test_path, f'{raster_name}{raster_extension}')
        return file_path

    def set_output_raster_file(self, raster_file: str, raster_name: str, raster_format: str = 'GTiff',
                               raster_extension: str = '.tif') -> None:
        """Set output raster for argument.

        Args:
            raster_file (str): The output raster file path.
            raster_name (str): The raster argument whose value is the raster name. For testing the raster name is the
                output file path in the files folder.
            raster_format: The short name of the raster format string.  See https://gdal.org/drivers/raster/index.html
                for a list of possible strings.
            raster_extension: The extension to add to the raster_name.
        """
        if raster_file != raster_name:
            file_path = self.get_output_raster(raster_name, raster_extension)
            if file_path != raster_file:
                raster_utils.copy_raster(raster_file, file_path, raster_format)

    def _load_raster_files(self) -> None:
        """Load raster files available for testing.

        Loads self._raster_files dictionary containing user raster name as key and file path as value.
        """
        self._raster_files = {}
        if self.file_folder is None:
            return
        extensions = ('tif', 'dem', 'tiff', 'txt', 'asc', 'bil', 'hdr', 'grd')
        raster_files = []
        if self.file_folder_structured():
            for extension in extensions:
                raster_files.extend(glob.glob(self.file_folder + f'/rasters/**/*.{extension}', recursive=True))
        else:
            for extension in extensions:
                raster_files.extend(glob.glob(self.file_folder + f'/**/*.{extension}', recursive=True))
        for file_path in raster_files:
            if self.file_folder_structured():
                relative_file_path = os.path.relpath(file_path, os.path.join(self.file_folder, 'rasters'))
            else:
                relative_file_path = os.path.relpath(file_path, self.file_folder)
            name, _ = os.path.splitext(relative_file_path)
            name.replace('\\', '/')
            self._raster_files[name] = file_path

    def get_input_dataset(self, dataset_name: str) -> DatasetReader:
        """Get a dataset chosen from the list of available datasets (self.available_datasets).

        Args:
            dataset_name (str): The dataset to retrieve.

        Returns:
            (DatasetReader): The dataset object.
        """
        if self.file_folder is None:
            if not os.path.isfile(dataset_name):
                return None
            dataset_file = dataset_name
            name_with_extension = os.path.basename(dataset_name)
            name = os.path.splitext(name_with_extension)[0]
        else:
            if self._dataset_files is None:
                self._load_dataset_files()
            if dataset_name not in self._dataset_files:
                return None
            dataset_file = self._dataset_files[dataset_name]
            grid_separator = dataset_name.find('/')
            name = dataset_name[grid_separator + 1:]
        try:
            return self.read_dataset_from_file(dataset_file, name)
        except Exception:
            return None

    def get_input_dataset_grid(self, dataset_name: str) -> Grid:
        """Get the geometry of a dataset chosen from the list of available datasets.

         Available datasets come from DataHandler.available_datasets.

        Args:
            dataset_name (str): The dataset to retrieve.

        Returns:
            (Grid): The dataset's constrained grid.
        """
        dset = self.get_input_dataset(dataset_name)
        if dset is None:
            return None
        if not self._grid_uuid_to_name:
            self._load_grid_files()
        geom_name = self._grid_uuid_to_name.get(dset.geom_uuid, None)
        if not geom_name:
            return None
        return self.get_input_grid(geom_name)

    def get_dataset_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get a dataset from a UUID.

        Args:
            uuid: The dataset UUID.

        Returns:
            The dataset name or None.
        """
        if self.file_folder is not None and self._dataset_files is None:
            self._load_dataset_files()

        for dataset_name in self._dataset_files.keys():
            dataset = self.get_input_dataset(dataset_name)
            if dataset is not None:
                dataset_uuid = dataset.uuid
                if dataset_uuid == uuid:
                    return dataset_name
        return None

    def get_uuid_from_dataset_name(self, dataset_name: str) -> Optional[str]:
        """Get a UUID from a dataset.

        Args:
            dataset_name: The dataset name.

        Returns:
            The dataset UUID or None.
        """
        if self.file_folder is not None and self._dataset_files is None:
            self._load_dataset_files()

        dataset = self.get_input_dataset(dataset_name)
        if dataset is not None:
            return dataset.uuid
        return None

    def set_grid_uuid_for_testing(self, grid_name: str, uuid: str) -> None:
        """Set the UUID for a grid. Used to get grid for a dataset for testing.

        Args:
            grid_name (str): The name of the grid.
            uuid (str): The UUID.
        """
        self._grid_uuid_to_name[uuid] = grid_name

    def read_dataset_from_file(self, dataset_file: str, dataset_name: str) -> DatasetReader:
        """Reads the dataset from disk and returns a xms.datasets.dset_reader.DatasetReader.

        Makes a bunch of assumptions, like that there's only one dataset per file and the dataset name is the same
        as the base file name.

        Args:
            dataset_file (str): Path to dataset file.
            dataset_name (str): Name of the dataset. Will be used to construct the H5 group path to the dataset

        Returns:
            (DatasetReader): The dataset object.
        """
        return DatasetReader(dataset_file, dataset_name)

    def get_output_dataset_writer(
            self,
            name: str = 'Dataset',
            dset_uuid: str = None,
            geom_uuid: str = '',
            num_components: int = 1,
            ref_time: Optional[Union[float, datetime.datetime]] = None,
            null_value: float = None,
            time_units: str = 'Days',
            units: str = '',
            dtype: str = 'f',
            use_activity_as_null: bool = False,
            location: str = 'points'
    ) -> DatasetWriter:
        """Get an output dataset writer for a tool.

        Args:
            name (str): Name of the dataset (will be used to build the dataset's group path)
            dset_uuid (str): UUID of the dataset
            geom_uuid (str): UUID of the dataset's geometry
            num_components (int): The number of data components (1=scalar, 2=vector, 3=3D vector)
            ref_time (float, datetime.datetime): The dataset's reference time. Either a Julian float or a
                Python datetime.datetime
            null_value (float): The null value of the dataset, if there is one
            time_units (str): The dataset's time units. One of: 'Seconds', 'Minutes', 'Hours', 'Days'
            units (str): Units of the dataset values
            dtype (str): Data type of the dataset's values. One of: 'f', 'd', 'i'
            use_activity_as_null (bool): If True, inactive values (0) will be treated as null values when computing
                timestep mins and maxs. Implies that activity array is on same location as data values (i.e. has same
                shape).
            location (str): Location of the dataset values. One of the XMDF_DATA_LOCATIONS keys. Note that this does
                not usually need to be set XMS is going to ignore it in most cases. XMS will try to determine the
                dataset location based on the geometries currently loaded (number of nodes, number of points). Here for
                historical reasons.):

        Returns:
            (str): A output dataset writer.
        """
        if self._uses_file_system and self.file_folder is None:
            # for file path use base name for dataset name
            dataset_path = Path(name)
            dataset_name = dataset_path.stem
        else:
            # otherwise use name as nested folder path within self.file_folder
            dataset_name = name
        return DatasetWriter(
            name=dataset_name,
            dset_uuid=dset_uuid,
            geom_uuid=geom_uuid,
            num_components=num_components,
            ref_time=ref_time,
            null_value=null_value,
            time_units=time_units,
            units=units,
            dtype=dtype,
            use_activity_as_null=use_activity_as_null,
            location=location
        )

    def set_output_dataset(self, dataset_writer: DatasetWriter) -> None:
        """Set output dataset for argument.

        Args:
            dataset_writer (DatasetWriter): The output dataset.
        """
        test_path = self._get_output_folder()
        if self.file_folder_structured():
            if not self._grid_uuid_to_name:
                self._load_grid_files()
            grid_name = self._grid_uuid_to_name.get(dataset_writer.geom_uuid)
            grid_path = os.path.join(test_path, 'grids', grid_name)
            copy_to_path = os.path.join(test_path, grid_path, dataset_writer.name + '.h5')
            Path(os.path.dirname(os.path.abspath(copy_to_path))).mkdir(parents=True, exist_ok=True)
            filesystem.copyfile(dataset_writer.h5_filename, copy_to_path)
        else:
            test_path = self.file_folder
            file_path = os.path.join(test_path, dataset_writer.name)
            filesystem.copyfile(dataset_writer.h5_filename, f'{file_path}.h5')

    def _load_dataset_files(self) -> None:
        """Load dataset files available for testing.

        Loads self._dataset_files dictionary containing user dataset name as key and file path as value.
        """
        if self.file_folder_structured():
            dataset_files = glob.glob(self.file_folder + '/grids/**/*.h5', recursive=True)
        else:
            dataset_files = glob.glob(
                self.file_folder + '/**/*.h5', recursive=True
            )  # Assuming all .h5 files are datasets
        self._dataset_files = {}
        for file_path in dataset_files:
            if self.file_folder_structured():
                relative_file_path = os.path.relpath(file_path, os.path.join(self.file_folder, 'grids'))
            else:
                relative_file_path = os.path.relpath(file_path, self.file_folder)
            name, _ = os.path.splitext(relative_file_path)
            name = name.replace('\\', '/')
            self._dataset_files[name] = os.path.normpath(file_path)

    def get_default_wkt(self) -> str:
        """Get the WKT of the default coordinate system.

        Returns:
            (str): The WKT of the default coordinate system.
        """
        return self._default_wkt

    def set_default_wkt(self, wkt: str) -> None:
        """Set WKT of the default coordinate system.

        Args:
            wkt (str): The WKT of the default coordinate system.
        """
        self._default_wkt = wkt

    def get_vertical_datum(self) -> str:
        """Get the vertical datum of the default coordinate system.

        Returns:
            (str): The vertical datum of the default coordinate system.
        """
        return self._vertical_datum

    def set_vertical_datum(self, datum: str) -> None:
        """Set vertical datum of the default coordinate system.

        Args:
            datum (str): The vertical datum of the default coordinate system.
        """
        self._vertical_datum = datum

    def get_vertical_units(self) -> str:
        """Get the vertical units of the default coordinate system.

        Returns:
            (str): The vertical units of the default coordinate system.
        """
        return self._vertical_units

    def set_vertical_units(self, units: str) -> None:
        """Set vertical units of the default coordinate system.

        Args:
            units (str): The vertical units of the default coordinate system.
        """
        self._vertical_units = units

    def send_output_to_xms(self):
        """Send output to XMS.

        This should be called from XMS only.
        """
        # this gets overriden in the XMS data handler
        raise RuntimeError('This should be called from XMS only.')

    def _read_grid(self, file_path: str) -> Optional[Grid]:
        """Read a constrained grid.

        Args:
            file_path (str): The file path.

        Returns:
            (Optional[Grid]): The constrained grid.
        """
        return read_grid_from_file(file_path)

    def file_folder_structured(self) -> bool:
        """Determine if self.file_folder uses structure (has coverages, grids, or rasters folders).

        Returns:
            (bool): If the self.file_folder path is structured.
        """
        if not self.check_structured or self.file_folder is None:
            return False
        if os.path.isdir(os.path.join(self.file_folder, 'coverages')):
            return True
        if os.path.isdir(os.path.join(self.file_folder, 'grids')):
            return True
        if os.path.isdir(os.path.join(self.file_folder, 'rasters')):
            return True
        return False

    def _get_output_folder(self) -> Optional[str]:
        """Get where to write output.

        Returns:
            Path to write output.
        """
        folder = None
        if self.output_folder is not None:
            folder = self.output_folder
        elif self.file_folder is not None:
            folder = self.file_folder
        return folder
