"""MergeGridsTool class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import filecmp
import os
import shutil
import time

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.tool_core import IoDirection, Tool

# 4. Local modules


ARG_INPUT_INTERVAL = 0
ARG_INPUT_MAX_BACKUPS = 1


class MeshBackupTool(Tool):
    """Turn on/off process to create mesh backups."""

    _do_mesh_backup = True

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Auto-backup')

        self._mesh_name = 'backup'
        self._mesh_backups = []
        self._max_num_backups = 10
        self._interval = 45
        self._max_space = None
        self._query = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (:obj:`list`): A list of the initial tool arguments.
        """
        arguments = [
            self.integer_argument(name='interval', description='Interval (seconds)', value=20, optional=False,
                                  min_value=1),
            self.integer_argument(name='max_backups', description='Maximum number of backups', value=10,
                                  optional=False, min_value=1),
            self.grid_argument(name='output_mesh', description='The mesh', hide=True, optional=True,
                               io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (:obj:`list`): The tool arguments.

        Returns:
            (:obj:`dict`): Dictionary of errors for arguments.
        """
        errors = {}

        return errors

    def _get_module_item_of_type(self, tree, module_type):
        """Called to determine if arguments are valid.

        Args:
            tree (:obj:`TreeNode`): the query
            module_type (:obj:`str`): module type string
        Returns:
            (:obj:`TreeNode`): module item
        """
        for module_item in tree.children:
            if module_item.item_typename == module_type:
                return module_item

        return None

    def _get_active_item_under_parent(self, parent, item_type):
        """Called to determine if arguments are valid.

        Args:
            parent (:obj:`TreeNode`): parent
            item_type (:obj:`str`): item type string
        Returns:
            (:obj:`TreeNode`): active item of type
        """
        for child in parent.children:
            if child.item_typename == item_type:
                if child.is_active_item is True:
                    return child
            else:
                item = self._get_active_item_under_parent(child, item_type)
                if item is not None:
                    return item

        return None

    def _get_active_item_of_type(self, tree, module_type, item_type):
        """Called to determine if arguments are valid.

        Args:
            tree (:obj:`TreeNode`): the project tree
            module_type (:obj:`str`): module type string
            item_type (:obj:`str`): item type string
        Returns:
            (:obj:`dict`): Dictionary of errors for arguments.
        """
        module_item = self._get_module_item_of_type(tree, module_type)
        if module_item is None:
            return None

        return self._get_active_item_under_parent(module_item, item_type)

    def _get_backup_file_info(self, module_type, item_uuid):
        """Called to determine if arguments are valid.

        Args:
            module_type (:obj:`str`): module type string
            item_uuid (:obj:`str`): item uuid
        Returns:
            (:obj:`str`, :obj:`str`, :obj:`int`):

                filename (:obj:`str`): Most recent name of the item/most recent backup file

                filename (:obj:`str`): oldest backup file

                num_files (:obj:`int`): num files


        """
        the_folder = XmEnv.xms_environ_temp_directory()
        if module_type == 'TI_ROOT_2DMESH':
            the_folder = os.path.join(the_folder, 'mesh_backup')
            the_folder = os.path.join(the_folder, item_uuid)

        name_file = os.path.join(the_folder, 'name.txt')
        most_recent_file = ''
        most_recent_ts = None
        oldest_file = ''
        oldest_ts = None
        num_files = 0
        if os.path.isdir(the_folder) is True:
            for file in os.scandir(the_folder):
                if file.path != name_file:
                    num_files = num_files + 1
                    ts = file.stat(follow_symlinks=False).st_ctime
                    if most_recent_ts is None or ts > most_recent_ts:
                        most_recent_ts = ts
                        most_recent_file = file.path
                    if oldest_ts is None or ts < oldest_ts:
                        oldest_ts = ts
                        oldest_file = file.path

        return most_recent_file, oldest_file, num_files

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (:obj:`list`): The tool arguments.
        """
        self._interval = arguments[ARG_INPUT_INTERVAL].value
        self._max_num_backups = arguments[ARG_INPUT_MAX_BACKUPS].value

        self.logger.info('Mesh auto-save turned on.')
        self.logger.info(f'Interval: {self._interval} seconds')
        mesh_folder = os.path.join(XmEnv.xms_environ_temp_directory(), 'mesh_backup')
        if not os.path.isdir(mesh_folder):
            os.mkdir(mesh_folder)
        self.logger.info(f'Temp directory: {mesh_folder}')
        if self._query is None: self._query = Query(blocking=False) # noqa E701
        tree = self._query.project_tree
        while self._do_mesh_backup is True:
            start_time = datetime.datetime.now()
            tree = self._query.refresh_project_tree()
            if tree is None:
                self.logger.info('Waiting for process...')
            else:
                active_mesh_tree_item = self._get_active_item_of_type(tree, 'TI_ROOT_2DMESH', 'TI_MESH2D')
                if active_mesh_tree_item is not None:
                    uuid = active_mesh_tree_item.uuid
                    active_mesh = self._query.item_with_uuid(uuid)
                    if active_mesh is not None:
                        this_mesh_folder = os.path.join(mesh_folder, uuid)
                        name_file = os.path.join(this_mesh_folder, 'name.txt')

                        if not os.path.isdir(this_mesh_folder):
                            os.mkdir(this_mesh_folder)
                        else:
                            os.remove(name_file)

                        # save the current name
                        with open(name_file, 'w') as f:
                            mesh_path = tree_util.build_tree_path(tree, uuid)
                            f.write(mesh_path)

                        file_and_path = active_mesh.cogrid_file
                        filename = os.path.basename(file_and_path)

                        recent_file, oldest_file, num_files = self._get_backup_file_info('TI_ROOT_2DMESH', uuid)

                        if recent_file == '' or filecmp.cmp(recent_file, file_and_path) is False:
                            backup_file = os.path.join(this_mesh_folder, filename)
                            shutil.move(file_and_path, backup_file)
                            self.logger.info(f'Created backup of mesh named "{active_mesh.name}".')
                            self.logger.info(f'Number of backups: {num_files + 1}.')

                            # see if we have exceeded the number of backups
                            if num_files >= self._max_num_backups:
                                self.logger.info(f'Maximum backups exceeded. Removing {oldest_file}.')
                                os.remove(oldest_file)
                        else:
                            os.remove(file_and_path)

            finish_time = datetime.datetime.now()
            elapsed = finish_time - start_time
            if elapsed.seconds < self._interval:
                wait_time = self._interval - elapsed.seconds
                time.sleep(wait_time)
