"""Utilities for file IO operations."""
# 1. Standard python modules
import os
from pathlib import Path
import re
import shutil
import struct

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.core.filesystem import filesystem as xfs

# 4. Local modules


TUFLOWFV_RUN_FOLDERS = {
    'bc_dbase',
    'check',
    'model',
    'results',
    'runs',
}
READ_BUFFER_SIZE = 100 * 1024
BOOL_SIZE_BYTES = 1
UINT_SIZE_BYTES = 4
FLOAT_32_SIZE_BYTES = 4
FLOAT_64_SIZE_BYTES = 8
MAX_STR_SIZE_BYTES = 40


def read_binary_into_bool(file):
    """Reads 4 bytes and puts it into an integer.

    Args:
        file (opened binary file): The file we are reading

    Returns:
        int: the value of the 4 bytes as an integer
    """
    file_byte = file.read(BOOL_SIZE_BYTES)
    [x] = struct.unpack('?', file_byte)
    return x


def read_binary_into_int(file):
    """Reads 4 bytes and puts it into an integer.

    Args:
        file (opened binary file): The file we are reading

    Returns:
        int: the value of the 4 bytes as an integer
    """
    four_byte = file.read(UINT_SIZE_BYTES)
    return int.from_bytes(four_byte, 'little')


def read_binary_into_float(file, num_bytes):
    """Reads bytes and puts it into a float.

    Args:
        file (opened binary file): The file we are reading
        num_bytes (int): the number of bytes to read

    Returns:
        float: the value of the bytes as a float
    """
    float_bytes = file.read(num_bytes)
    [x] = struct.unpack('f' if num_bytes == FLOAT_32_SIZE_BYTES else 'd', float_bytes)
    return x


def read_binary_into_str(file, num_bytes):
    """Reads bytes and puts it into a string.

    Args:
        file (opened binary file): The file we are reading
        num_bytes (int): the number of bytes to read

    Returns:
        str: the bytes decoded as a string
    """
    file_bytes = file.read(num_bytes)
    return file_bytes.decode('ascii')


def read_restart_file_time(filename):
    """Read the restart file time (in seconds) from the first value in the restart file.

    Args:
        filename (str): Absolute path to the restart file

    Returns:
        float: The restart file time in seconds
    """
    try:
        with open(filename, 'rb') as f:
            return read_binary_into_float(f, FLOAT_64_SIZE_BYTES)
    except Exception:
        return 0.0


def create_component_folder(comp_uuid):
    """Create a folder for a component in the XMS temp components folder.

    Args:
        comp_uuid (str): UUID of the new component

    Returns:
        str: Path to the new component folder
    """
    temp_comp_dir = Path(XmEnv.xms_environ_temp_directory()) / 'Components' / comp_uuid
    temp_comp_dir.mkdir(parents=True, exist_ok=True)
    return str(temp_comp_dir)


def move_to_shared_folder(sim_folder):
    """Move from a simulation export folder to the shared TUFLOWFV folder.

    Args:
        sim_folder (str): Path to the simulation folder, will be deleted

    Returns:
        str: Path to the shared TUFLOWFV folder
    """
    sim_folder = os.path.normpath(sim_folder)
    tuflowfv_folder = os.path.dirname(sim_folder)
    os.makedirs(tuflowfv_folder, exist_ok=True)
    os.chdir(tuflowfv_folder)
    sim_name = os.path.basename(sim_folder).strip('\\/')
    # Don't delete the sim folder if the user happened to name it the same as one of the standard TUFLOWFV run folders.
    if sim_name not in TUFLOWFV_RUN_FOLDERS:
        shutil.rmtree(sim_folder, ignore_errors=True)
    return tuflowfv_folder


def build_input_filepaths(fvc_filename, sub_dir, basename):
    """Build an absolute path and a relative path from the .fvc for an input file.

    Args:
        fvc_filename (str): Absolute path to the TUFLOWFV control file to build relative filepath reference for
        sub_dir (str): The subdirectory of the root TUFLOWFV simulation folder containing the file. Should not
            contain leading or trailing separators. Note that
            this method only works for files that are not in the same directory as the .fvc file.
        basename (str): Base filename of the input file

    Returns:
        tuple(str, str): The absolute path to the file, the path to the file relative from the .fvc file
    """
    rel_path = f'../{sub_dir}/{basename}'
    tuflowfv_folder = os.path.dirname(os.path.dirname(fvc_filename))
    abs_path = os.path.normpath(os.path.join(tuflowfv_folder, sub_dir, basename))
    return abs_path, rel_path


def read_multi_sim_file(filename):
    """Read a file with multiple control file references.

    Args:
        filename (str): Path to the multi sim file

    Returns:
        list[str]: The normalized absolute paths to the control files to import
    """
    with open(filename, 'r') as f:
        lines = f.readlines()
    fvc_files = []
    for line in lines:
        line = line.strip()
        skip_line = not line or line.startswith('!') or line.startswith('#')  # skip blank lines and comments
        if skip_line:
            continue
        split_line = re.split('[!#]', line)
        value = split_line[0].replace('"', '').strip()
        value = xfs.resolve_relative_path(os.path.dirname(filename), value)
        fvc_files.append(value)
    return fvc_files


def logging_filename(abs_path):
    """Get the filename for logging (basename if testing).

    Args:
        abs_path (str): Aboslute path to the file to log

    Returns:
        str: abs_path or base_name of abs_path if testing
    """
    if XmEnv.xms_environ_running_tests() == 'TRUE':
        return os.path.basename(abs_path)
    return abs_path


def compute_relative_path_safe(base_path, abs_path):
    """Compute a relative path, returning the absolute path on error."""
    try:
        return xfs.compute_relative_path(base_path, abs_path)
    except Exception:
        return abs_path


def str2float2int(text):
    """Coerce a string first into a float and then into a int (common when reading model native input files).

    Args:
        text (str): The text to convert

    Returns:
        int: See description
    """
    return int(float(text))
