"""Qt delegate that displays a button for bringing up the advanced options dialog in the material table."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.tuflowfv.gui import model_control_consts as const
from xms.tuflowfv.gui.advanced_materials_dialog import AdvancedMaterialsDialog

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


class AdvancedMaterialButtonDelegate(QStyledItemDelegate):
    """Qt delegate that displays a button for bringing up the advanced options dialog in the material table."""

    def __init__(self, parent, display_projection, set_mat):
        """Initializes the class.

        Args:
            parent (QObject): The parent object.
            display_projection (Projection): The current display projection. Used for switching unit labels between SI
                and Imperial.
            set_mat (bool): True if the simulation set_mat table, False for normal materials
        """
        super().__init__(parent)
        self.dlg = None
        self.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        self.parent_dlg = parent
        self.display_projection = display_projection
        self.set_mat = set_mat

    def _setup_ui_for_set_mat(self):
        """If simulation set mat, hide note about parent simulation dependency and label texts."""
        if not self.set_mat:
            return  # Normal material, don't change anything
        self.dlg.ui.lbl_parent_sim_note.setVisible(False)
        mixing_model = self.parent().ui.cbx_mixing_model_horiz.currentIndex()
        if mixing_model in [const.HORIZ_MIXING_OPT_SMAGORINSKY, const.HORIZ_MIXING_OPT_WU]:
            # Replace (coefficient or units) portion of text. Units vary based on display projection.
            self.dlg.ui.lbl_horiz_viscosity.setText('Horizontal eddy viscosity coefficient:')
            self.dlg.ui.lbl_horiz_viscosity_min.setText('Minimum eddy viscosity coefficient:')
            self.dlg.ui.lbl_horiz_viscosity_max.setText('Maximum eddy viscosity coefficient:')
            self.dlg.ui.lbl_vert_viscosity_min.setText('Minimum eddy viscosity coefficient:')
            self.dlg.ui.lbl_vert_viscosity_max.setText('Maximum eddy viscosity coefficient:')
        else:
            self.dlg.ui.grp_horiz_viscosity_limits.setVisible(False)
            self.dlg.ui.grp_vert_viscosity_limits.setVisible(False)
            # Just replace the 'coefficient or ' portion of the text. Keep units set based on display projection
            self.dlg.ui.lbl_horiz_viscosity.setText(
                self.dlg.ui.lbl_horiz_viscosity.text().replace('coefficient or ', '')
            )

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Advanced...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            bool: True if the event is handled
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                self.dlg = AdvancedMaterialsDialog(parent=self.parent_dlg, row_idx=index.row(),
                                                   df=self.parent_dlg.material_model.data_frame,
                                                   display_projection=self.display_projection)
                self._setup_ui_for_set_mat()
                self.dlg.show() if self.testing else self.dlg.exec()
                return True
        return super().editorEvent(event, model, option, index)
