"""Qt dialog for assigning attributes of a Boundary Conditions coverage point or arc."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QDoubleValidator
from PySide2.QtWidgets import QApplication, QLabel
import xarray as xr

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.number_corrector import NumberCorrector

# 4. Local modules
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.gui import assign_bc_consts as const, gui_util
from xms.tuflowfv.gui.assign_bc_dialog_ui import Ui_AssignBcDialog
from xms.tuflowfv.gui.bc_series_editor import BcSeriesEditor, extract_time_widget_data, populate_time_widgets
from xms.tuflowfv.gui.grid_variables_dialog import GridVariablesDialog


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class AssignBcDialog(XmsDlg):
    """A dialog showing the Assign BC dialog."""

    def __init__(self, parent, bc_data, all_data, msg, bc_location, comp_id, time_formats):
        """Initializes the Assign BC dialog.

        Args:
            parent (QWidget): The parent Qt dialog
            bc_data (xarray.Dataset): Dataset containing the boundary conditions parameters
            all_data (BcData): The entire BC data class
            msg (str): Warning/multi-select message, if any
            bc_location (int): Location of the BC, one of BC_LOCATION_* constants
            comp_id (int): Component id of the feature being assigned
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        super().__init__(parent, 'xms.tuflowfv.gui.assign_bc_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Display_Options'
        self.bc_location = bc_location
        self.bc_data = bc_data
        self.all_data = all_data
        self.comp_id = comp_id
        self.msg = msg
        self.time_format_std = time_formats[0]
        self.time_format_qt = time_formats[1]
        self._last_subtype_index = 0
        self.ui = Ui_AssignBcDialog()
        self.ui.setupUi(self)
        # Commonly used edit field validators
        self.number_corrector = NumberCorrector(self)
        self.dbl_validator = QDoubleValidator(self)
        self.dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.dbl_noneg_validator = QDoubleValidator(self)
        self.dbl_noneg_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.dbl_noneg_validator.setBottom(0.00000001)
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Add the message label if not empty
        if self.msg:
            label = QLabel(self.msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.ui.scrollable_area_bc_contents.layout().insertWidget(0, label)
        # Setup stuff used by both the arc and point variants
        self._add_combobox_options()
        self._add_validators()
        # Set up widgets specific for arcs/points/global
        if self.bc_location == bcd.BC_LOCATION_ARC:
            self._setup_ui_for_arcs()
        elif self.bc_location == bcd.BC_LOCATION_GLOBAL:
            self._setup_ui_for_global()
        elif self.bc_location == bcd.BC_LOCATION_GRID:
            self._setup_ui_for_gridded()
        elif self.bc_location == bcd.BC_LOCATION_POINT:
            self._setup_ui_for_points()
        else:  # self.bc_location == bcd.BC_LOCATION_POLY:
            self._setup_ui_for_polygons()
        self._connect_common_signals()  # Connect last so location-specific slots get called first when BC type changes
        # Populate widgets from data
        self._load_data()
        # Hide the vertical distribution section until we support 3D
        self.ui.grp_vertical_distribution.setVisible(False)

    def _add_combobox_options(self):
        """Add items to combobox widgets common to all BC locations or have widgets hidden when disabled."""
        gui_util.add_combobox_options(const.BC_CBX_OPTS, self.ui.cbx_vertical_distribution)
        # Add all subtypes initially so we can load data, will filter later
        gui_util.add_combobox_options(const.BC_CBX_OPTS, self.ui.cbx_bc_subtype)
        # Add the global combobox options
        # If this is a point BC, change the option text to 'fvc' instead of '2dm' for the ASCII option.
        if self.bc_location == bcd.BC_LOCATION_POINT:
            # const.BC_CBX_OPTS['cbx_export_format_bc']: (
            #     ['2dm', 'Shapefile'],  # Combobox option test
            #     ['2dm', 'Shapefile'],  # User data
            # ),
            const.BC_CBX_OPTS['cbx_export_format_bc'][0][0] = 'fvc'
        gui_util.add_combobox_options(const.BC_CBX_OPTS, self.ui.cbx_export_format_bc)

    def _add_validators(self):
        """Add validators to edit fields."""
        self.ui.edt_friction_slope.setValidator(self.dbl_validator)
        self.ui.edt_friction_slope.installEventFilter(self.number_corrector)
        self.ui.edt_default1.setValidator(self.dbl_validator)
        self.ui.edt_default1.installEventFilter(self.number_corrector)
        self.ui.edt_default2.setValidator(self.dbl_validator)
        self.ui.edt_default2.installEventFilter(self.number_corrector)
        self.ui.edt_offset1.setValidator(self.dbl_validator)
        self.ui.edt_offset1.installEventFilter(self.number_corrector)
        self.ui.edt_offset2.setValidator(self.dbl_validator)
        self.ui.edt_offset2.installEventFilter(self.number_corrector)
        self.ui.edt_scale1.setValidator(self.dbl_validator)
        self.ui.edt_scale1.installEventFilter(self.number_corrector)
        self.ui.edt_scale2.setValidator(self.dbl_validator)
        self.ui.edt_scale2.installEventFilter(self.number_corrector)
        self.ui.edt_update_dt.setValidator(self.dbl_noneg_validator)
        self.ui.edt_update_dt.installEventFilter(self.number_corrector)
        self.ui.edt_update_dt.setValidator(self.dbl_noneg_validator)
        self._add_grid_validators()

    def _add_grid_validators(self):
        """Add validators to widgets that are only applicable to gridded BC types."""
        if self.bc_location != bcd.BC_LOCATION_GRID:
            return
        # Up to 8 fields for wave BCs
        self.ui.edt_default3.setValidator(self.dbl_validator)
        self.ui.edt_default3.installEventFilter(self.number_corrector)
        self.ui.edt_default4.setValidator(self.dbl_validator)
        self.ui.edt_default4.installEventFilter(self.number_corrector)
        self.ui.edt_default5.setValidator(self.dbl_validator)
        self.ui.edt_default5.installEventFilter(self.number_corrector)
        self.ui.edt_default6.setValidator(self.dbl_validator)
        self.ui.edt_default6.installEventFilter(self.number_corrector)
        self.ui.edt_default7.setValidator(self.dbl_validator)
        self.ui.edt_default7.installEventFilter(self.number_corrector)
        self.ui.edt_default8.setValidator(self.dbl_validator)
        self.ui.edt_default8.installEventFilter(self.number_corrector)

        self.ui.edt_offset3.setValidator(self.dbl_validator)
        self.ui.edt_offset3.installEventFilter(self.number_corrector)
        self.ui.edt_offset4.setValidator(self.dbl_validator)
        self.ui.edt_offset4.installEventFilter(self.number_corrector)
        self.ui.edt_offset5.setValidator(self.dbl_validator)
        self.ui.edt_offset5.installEventFilter(self.number_corrector)
        self.ui.edt_offset6.setValidator(self.dbl_validator)
        self.ui.edt_offset6.installEventFilter(self.number_corrector)
        self.ui.edt_offset7.setValidator(self.dbl_validator)
        self.ui.edt_offset7.installEventFilter(self.number_corrector)
        self.ui.edt_offset8.setValidator(self.dbl_validator)
        self.ui.edt_offset8.installEventFilter(self.number_corrector)

        self.ui.edt_scale3.setValidator(self.dbl_validator)
        self.ui.edt_scale3.installEventFilter(self.number_corrector)
        self.ui.edt_scale4.setValidator(self.dbl_validator)
        self.ui.edt_scale4.installEventFilter(self.number_corrector)
        self.ui.edt_scale5.setValidator(self.dbl_validator)
        self.ui.edt_scale5.installEventFilter(self.number_corrector)
        self.ui.edt_scale6.setValidator(self.dbl_validator)
        self.ui.edt_scale6.installEventFilter(self.number_corrector)
        self.ui.edt_scale7.setValidator(self.dbl_validator)
        self.ui.edt_scale7.installEventFilter(self.number_corrector)
        self.ui.edt_scale8.setValidator(self.dbl_validator)
        self.ui.edt_scale8.installEventFilter(self.number_corrector)

        self.ui.edt_reference_time.setValidator(self.dbl_validator)
        self.ui.edt_reference_time.installEventFilter(self.number_corrector)

    def _setup_ui_for_arcs(self):
        """Setup widgets specific to arcs."""
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.on_arc_bc_type_changed)
        gui_util.add_combobox_options(const.BC_CBX_OPTS, self.ui.cbx_bc_type)
        self._hide_all_grid_widgets()

    def _setup_ui_for_points(self):
        """Setup widgets specific to points."""
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.on_point_bc_type_changed)
        gui_util.add_combobox_options(const.POINT_BC_CBX_OPTS, self.ui.cbx_bc_type)
        self.ui.lbl_offset2.setVisible(False)
        self.ui.edt_offset2.setVisible(False)
        self.ui.lbl_scale2.setVisible(False)
        self.ui.edt_scale2.setVisible(False)
        self.ui.lbl_default2.setVisible(False)
        self.ui.edt_default2.setVisible(False)
        self.ui.layout_friction_slope.setVisible(False)
        self._hide_all_grid_widgets()

    def _setup_ui_for_polygons(self):
        """Setup widgets specific to polygons."""
        # self.ui.cbx_bc_type.currentIndexChanged.connect(self.on_point_bc_type_changed)
        gui_util.add_combobox_options(const.POLYGON_BC_CBX_OPTS, self.ui.cbx_bc_type)
        self.ui.lbl_offset2.setVisible(False)
        self.ui.edt_offset2.setVisible(False)
        self.ui.lbl_scale2.setVisible(False)
        self.ui.edt_scale2.setVisible(False)
        self.ui.lbl_default2.setVisible(False)
        self.ui.edt_default2.setVisible(False)
        self.ui.layout_friction_slope.setVisible(False)
        self._refresh_subtype_cbx(2)  # 2 sub-types for 'QC_POLY', may not always be the case
        self._hide_all_grid_widgets()

    def _setup_ui_for_global(self):
        """Hide widgets not applicable to global BCs."""
        # We currently only support the QG global BC type. So much of this is currently specific to that single type.
        # If we add support for more global BC types, need to connect a slot to update widget states when the BC type
        # changes. See the slots on_arc_bc_type_changed() and on_point_bc_type_changed().
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.on_global_bc_type_changed)
        gui_util.add_combobox_options(const.GLOBAL_BC_CBX_OPTS, self.ui.cbx_bc_type)
        self.ui.layout_friction_slope.setVisible(False)
        self.ui.lbl_offset2.setVisible(False)
        self.ui.edt_offset2.setVisible(False)
        self.ui.lbl_scale2.setVisible(False)
        self.ui.edt_scale2.setVisible(False)
        self.ui.lbl_default2.setVisible(False)
        self.ui.edt_default2.setVisible(False)
        self._hide_all_grid_widgets()
        self.ui.tabs_bc.removeTab(1)  # Hide the global options tab, this is not applicable for global BC types.
        self._refresh_subtype_cbx(2)  # 2 sub-types for 'QG', may not always be the case
        self.setWindowTitle('Assign Global Boundary Condition')

    def _setup_ui_for_gridded(self):
        """Setup widgets specific to gridded BCs."""
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.on_gridded_bc_type_changed)
        self.ui.tog_use_isodate.stateChanged.connect(self.on_tog_use_isodate)
        gui_util.add_combobox_options(const.GRIDDED_BC_CBX_OPTS, self.ui.cbx_bc_type)
        self.ui.layout_friction_slope.setVisible(False)
        self.ui.btn_define_curve.setVisible(False)
        self.ui.lbl_bc_subtype.setVisible(False)
        self.ui.cbx_bc_subtype.setVisible(False)
        self.ui.tabs_bc.removeTab(1)  # Hide the global options tab, this is not applicable for global BC types.
        self.setWindowTitle('Assign Gridded Boundary Condition')

    def _connect_common_signals(self):
        """Connect Qt widget signal/slots that are common to all BC locations or have widgets hidden when disabled."""
        self.ui.btn_vertical_ditribution.clicked.connect(self.on_btn_vertical_distribution)
        self.ui.btn_define_curve.clicked.connect(self.on_btn_define_curve)
        self.ui.btn_grid_file.clicked.connect(self.on_btn_grid_file)
        self.ui.btn_grid_variables.clicked.connect(self.on_btn_grid_variables)
        # When the BC type changes, the BC location-specific code will execute slots first that hides/show widgets in
        # The collapsible groupboxes that will change the size it should expand/shrink its height by
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.ui.grp_default.toggle_group)
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.ui.grp_offset.toggle_group)
        self.ui.cbx_bc_type.currentIndexChanged.connect(self.ui.grp_scale.toggle_group)
        # When the global export option changes to '2dm', display a warning
        self.ui.cbx_export_format_bc.currentIndexChanged.connect(self.on_export_format_changed)

    def _refresh_subtype_cbx(self, num_subtypes):
        """Update the options in the subtype combobox when the BC type changes.

        Args:
            num_subtypes (int): The number of subtypes for the BC type. If less than 1, will hide the subtype combobox.
        """
        subtype_idx = self.ui.cbx_bc_subtype.currentIndex()
        if subtype_idx >= 0:
            self._last_subtype_index = subtype_idx
        if num_subtypes < 1:
            self.ui.cbx_bc_subtype.setVisible(False)
            self.ui.lbl_bc_subtype.setVisible(False)
            return
        self.ui.cbx_bc_subtype.setVisible(True)
        self.ui.lbl_bc_subtype.setVisible(True)
        self.ui.cbx_bc_subtype.clear()
        for i in range(num_subtypes):
            display_text = const.BC_CBX_OPTS['cbx_bc_subtype'][0][i]
            model_card = const.BC_CBX_OPTS['cbx_bc_subtype'][1][i]
            self.ui.cbx_bc_subtype.addItem(display_text, model_card)
        current_idx = min(self._last_subtype_index, self.ui.cbx_bc_subtype.count() - 1)
        self.ui.cbx_bc_subtype.setCurrentIndex(current_idx)

    def _hide_all_grid_widgets(self):
        """Hides all the gridded BC widgets."""
        self.ui.lbl_grid_file.setVisible(False)
        self.ui.btn_grid_file.setVisible(False)
        self.ui.lbl_grid_file_selection.setVisible(False)
        self.ui.btn_grid_variables.setVisible(False)
        self._show_grid_parameters(bcd.UNINITIALIZED_COMP_ID)
        # These time options are defined in the BC curve editor for arc, point, and global BCs.
        self.ui.grp_reference_time.setVisible(False)
        self.ui.grp_time_units.setVisible(False)

    def _show_grid_parameters(self, index):
        """Enable/disable the grid default, offset, and scale widgets based on grid BC type.

        Args:
            index (int): Combobox index of the grid BC type, pass -1 to disable all but the widgets for the first two
                fields.  Gridded Wave BCs have up to 8 fields, while arc, point, and global BCs have a maximum of two
                fields (time series curves).
        """
        if index == const.BC_TYPE_MSLP:  # 1 field for MSLP
            self.ui.lbl_default2.setVisible(False)
            self.ui.edt_default2.setVisible(False)
            self.ui.lbl_offset2.setVisible(False)
            self.ui.edt_offset2.setVisible(False)
            self.ui.lbl_scale2.setVisible(False)
            self.ui.edt_scale2.setVisible(False)
        elif index > const.BC_TYPE_MSLP:  # Don't mess with second field if -1 (arc, point, or global)
            self.ui.lbl_default2.setVisible(True)
            self.ui.edt_default2.setVisible(True)
            self.ui.lbl_offset2.setVisible(True)
            self.ui.edt_offset2.setVisible(True)
            self.ui.lbl_scale2.setVisible(True)
            self.ui.edt_scale2.setVisible(True)

        enable_wave = index == const.BC_TYPE_WAVE
        self.ui.lbl_default3.setVisible(enable_wave)
        self.ui.edt_default3.setVisible(enable_wave)
        self.ui.lbl_default4.setVisible(enable_wave)
        self.ui.edt_default4.setVisible(enable_wave)
        self.ui.lbl_default5.setVisible(enable_wave)
        self.ui.edt_default5.setVisible(enable_wave)
        self.ui.lbl_default6.setVisible(enable_wave)
        self.ui.edt_default6.setVisible(enable_wave)
        self.ui.lbl_default7.setVisible(enable_wave)
        self.ui.edt_default7.setVisible(enable_wave)
        self.ui.lbl_default8.setVisible(enable_wave)
        self.ui.edt_default8.setVisible(enable_wave)

        self.ui.lbl_offset3.setVisible(enable_wave)
        self.ui.edt_offset3.setVisible(enable_wave)
        self.ui.lbl_offset4.setVisible(enable_wave)
        self.ui.edt_offset4.setVisible(enable_wave)
        self.ui.lbl_offset5.setVisible(enable_wave)
        self.ui.edt_offset5.setVisible(enable_wave)
        self.ui.lbl_offset6.setVisible(enable_wave)
        self.ui.edt_offset6.setVisible(enable_wave)
        self.ui.lbl_offset7.setVisible(enable_wave)
        self.ui.edt_offset7.setVisible(enable_wave)
        self.ui.lbl_offset8.setVisible(enable_wave)
        self.ui.edt_offset8.setVisible(enable_wave)

        self.ui.lbl_scale3.setVisible(enable_wave)
        self.ui.edt_scale3.setVisible(enable_wave)
        self.ui.lbl_scale4.setVisible(enable_wave)
        self.ui.edt_scale4.setVisible(enable_wave)
        self.ui.lbl_scale5.setVisible(enable_wave)
        self.ui.edt_scale5.setVisible(enable_wave)
        self.ui.lbl_scale6.setVisible(enable_wave)
        self.ui.edt_scale6.setVisible(enable_wave)
        self.ui.lbl_scale7.setVisible(enable_wave)
        self.ui.edt_scale7.setVisible(enable_wave)
        self.ui.lbl_scale8.setVisible(enable_wave)
        self.ui.edt_scale8.setVisible(enable_wave)

    def _show_all_feature_widgets(self, visible):
        """Show or hide the groups (hide if unassigned, show if anything else) for arc and point BC types.

        Args:
            visible (bool): True if groups should be visible, False to hide
        """
        # Hide/show widgets
        self.ui.lbl_bc_subtype.setVisible(visible)
        self.ui.cbx_bc_subtype.setVisible(visible)
        self.ui.btn_define_curve.setVisible(visible)
        self.ui.tog_include_mslp.setVisible(visible)
        # Hide/show groups
        self.ui.grp_default.setVisible(visible)
        self.ui.grp_offset.setVisible(visible)
        self.ui.grp_scale.setVisible(visible)
        self.ui.grp_update_dt.setVisible(visible)
        # This is always disabled until we support 3D
        # self.ui.grp_vertical_distribution.setVisible(visible)

    def _update_label_text(self, index):
        """Update text in labels when BC type changes.

        Args:
            index (int): Combobox option index of the new BC type
        """
        if self.bc_location == bcd.BC_LOCATION_ARC and index in [const.BC_TYPE_MONITOR, const.BC_TYPE_ZG]:
            return  # Everything hidden if unassigned arc or ZG arc

        if self.bc_location == bcd.BC_LOCATION_ARC:
            label_text = const.ARC_BC_LABEL_TEXT
        elif self.bc_location == bcd.BC_LOCATION_POINT:
            label_text = const.POINT_BC_LABEL_TEXT
        elif self.bc_location == bcd.BC_LOCATION_POLY:
            label_text = const.POLY_BC_LABEL_TEXT
        elif self.bc_location == bcd.BC_LOCATION_GLOBAL:
            label_text = const.GLOBAL_BC_LABEL_TEXT
        else:  # Gridded BCs
            label_text = const.GRID_BC_LABEL_TEXT
        self.ui.lbl_default1.setText(label_text['lbl_default1'].get(index, ''))
        self.ui.lbl_default2.setText(label_text['lbl_default2'].get(index, ''))
        self.ui.lbl_offset1.setText(label_text['lbl_offset1'].get(index, ''))
        self.ui.lbl_offset2.setText(label_text['lbl_offset2'].get(index, ''))
        self.ui.lbl_scale1.setText(label_text['lbl_scale1'].get(index, ''))
        self.ui.lbl_scale2.setText(label_text['lbl_scale2'].get(index, ''))
        self._update_grid_label_text(index)

    def _update_grid_label_text(self, index):
        """Update text in labels that are only applicable to gridded BCs when BC type changes.

        Args:
            index (int): Combobox option index of the new BC type
        """
        if self.bc_location != bcd.BC_LOCATION_GRID:
            return
        label_text = const.GRID_BC_LABEL_TEXT
        self.ui.lbl_default3.setText(label_text['lbl_default3'].get(index, ''))
        self.ui.lbl_default4.setText(label_text['lbl_default4'].get(index, ''))
        self.ui.lbl_default5.setText(label_text['lbl_default5'].get(index, ''))
        self.ui.lbl_default6.setText(label_text['lbl_default6'].get(index, ''))
        self.ui.lbl_default7.setText(label_text['lbl_default7'].get(index, ''))
        self.ui.lbl_default8.setText(label_text['lbl_default8'].get(index, ''))
        self.ui.lbl_offset3.setText(label_text['lbl_offset3'].get(index, ''))
        self.ui.lbl_offset4.setText(label_text['lbl_offset4'].get(index, ''))
        self.ui.lbl_offset5.setText(label_text['lbl_offset5'].get(index, ''))
        self.ui.lbl_offset6.setText(label_text['lbl_offset6'].get(index, ''))
        self.ui.lbl_offset7.setText(label_text['lbl_offset7'].get(index, ''))
        self.ui.lbl_offset8.setText(label_text['lbl_offset8'].get(index, ''))
        self.ui.lbl_scale3.setText(label_text['lbl_scale3'].get(index, ''))
        self.ui.lbl_scale4.setText(label_text['lbl_scale4'].get(index, ''))
        self.ui.lbl_scale5.setText(label_text['lbl_scale5'].get(index, ''))
        self.ui.lbl_scale6.setText(label_text['lbl_scale6'].get(index, ''))
        self.ui.lbl_scale7.setText(label_text['lbl_scale7'].get(index, ''))
        self.ui.lbl_scale8.setText(label_text['lbl_scale8'].get(index, ''))

    def _update_grid_variables_button_state(self):
        """Enables/disables the grid BC NetCDF variables button based on existence of selected dataset file."""
        grid_file = self.ui.lbl_grid_file_selection.text()
        enable = grid_file and grid_file != gui_util.NULL_SELECTION and self.all_data.does_file_exist(grid_file)
        self.ui.btn_grid_variables.setEnabled(enable)
        if self._get_current_bc_type() in bcd.CURTAIN_BC_TYPES:
            self.ui.btn_define_curve.setEnabled(enable)  # Tie state of variables button to curve button if curtain BC

    def _initialize_state(self):
        """Initialize widget states after populating from data."""
        if self.bc_location == bcd.BC_LOCATION_ARC:
            self.on_arc_bc_type_changed(self.ui.cbx_bc_type.currentIndex())
        elif self.bc_location == bcd.BC_LOCATION_POINT:
            self.on_point_bc_type_changed(self.ui.cbx_bc_type.currentIndex())
        elif self.bc_location == bcd.BC_LOCATION_GLOBAL:
            self.on_global_bc_type_changed(self.ui.cbx_bc_type.currentIndex())
        else:  # Gridded BC types
            self.on_tog_use_isodate(self.ui.tog_use_isodate.checkState())
            self.on_gridded_bc_type_changed(self.ui.cbx_bc_type.currentIndex())
            self._update_grid_variables_button_state()
        # If a feature type boundary condition coverage, initialize the global GIS export warning.
        if self.bc_location in [bcd.BC_LOCATION_POINT, bcd.BC_LOCATION_ARC, bcd.BC_LOCATION_POLY]:
            self.on_export_format_changed(self.ui.cbx_export_format_bc.currentIndex())

    def _get_current_bc_type(self):
        """Get the current BC type data card from the combobox.

        Returns:
            str: The TUFLOWFV card value
        """
        return self.ui.cbx_bc_type.itemData(self.ui.cbx_bc_type.currentIndex())

    def _get_netcdf_data_for_curtain(self):
        """Read the time and distance variables for a curtain boundary and create a lazy load xarray Dataset.

        Returns:
            tuple: pd.DataFrame with a single time column, np.array of the lengths along the arc (X-axis), a lazy load
                xarray Dataset pointing to the NetCDF data file, path to the NetCDF file.
        """
        dset = None
        try:
            abs_path = xfs.resolve_relative_path(self.all_data.info.attrs['proj_dir'],
                                                 self.ui.lbl_grid_file_selection.text())
            dset = xr.load_dataset(abs_path, decode_times=False)
            all_vars = list(dset.keys())
            all_vars.extend(list(dset.coords))
            # Ensure the required time and chainage variables exist (this may be specific to WL_CURT).
            time_var = self.bc_data.variable1.item()
            if time_var not in dset:
                raise Exception(f'Unable to find required time variable in NetCDF file.\nSpecified: {time_var}\n'
                                f'Possible variables: {all_vars}')
            chainage_var = self.bc_data.variable2.item()
            if chainage_var not in dset:
                raise Exception('Unable to find required chainage variable in NetCDF file.\nSpecified: '
                                f'{chainage_var}\nPossible variables: {all_vars}')

            time_df = dset[time_var].to_dataframe()
            time_df.columns = [f'TIME ({time_df.columns[0]})']  # Add default variable name to column name
            dist_df = dset[chainage_var].to_dataframe()
            dist_df.columns = [f'CHAINAGE ({dist_df.columns[0]})']  # Add default variable name to column name
            return time_df, dist_df, dset, abs_path
        except Exception as ex:
            msg = 'Error loading curtain boundary from NetCDF file. Ensure a valid NetCDF file has been selected and ' \
                  f'all required variables have been specified.\n{str(ex)}'
            message_with_ok(parent=self, message=msg, app_name='SMS', icon='Critical', win_icon=self.windowIcon())
            if dset is not None:
                dset.close()  # Close file handle
        return None, None, None, None

    def _load_data(self):
        """Populate widgets from persistent data."""
        gui_util.set_combobox_from_data(self.ui.cbx_bc_type, self.bc_data['type'][0].item())
        self._last_subtype_index = gui_util.set_combobox_from_data(self.ui.cbx_bc_subtype,
                                                                   int(self.bc_data['subtype'][0].item()))
        self.ui.edt_bc_name.setText(str(self.bc_data['name'][0].item()))
        self.ui.edt_friction_slope.setText(str(self.bc_data['friction_slope'][0].item()))
        self.ui.grp_default.setChecked(bool(int(self.bc_data['define_default'][0].item())))
        self.ui.edt_default1.setText(str(self.bc_data['default1'][0].item()))
        self.ui.edt_default2.setText(str(self.bc_data['default2'][0].item()))
        self.ui.grp_offset.setChecked(bool(int(self.bc_data['define_offset'][0].item())))
        self.ui.edt_offset1.setText(str(self.bc_data['offset1'][0].item()))
        self.ui.edt_offset2.setText(str(self.bc_data['offset2'][0].item()))
        self.ui.grp_scale.setChecked(bool(int(self.bc_data['define_scale'][0].item())))
        self.ui.edt_scale1.setText(str(self.bc_data['scale1'][0].item()))
        self.ui.edt_scale2.setText(str(self.bc_data['scale2'][0].item()))
        self.ui.grp_update_dt.setChecked(bool(int(self.bc_data['define_update_dt'][0].item())))
        self.ui.edt_update_dt.setText(str(self.bc_data['update_dt'][0].item()))
        self.ui.tog_include_mslp.setChecked(bool(int(self.bc_data['include_mslp'][0].item())))
        gui_util.set_combobox_from_data(self.ui.cbx_vertical_distribution,
                                        self.bc_data['vertical_distribution_type'][0].item())
        self.ui.grp_vertical_distribution.setChecked(bool(int(self.bc_data['define_vertical_distribution'][0].item())))
        distribution_file = self.bc_data['vertical_distribution_file'][0].item()
        if distribution_file and self.all_data.does_file_exist(distribution_file):
            self.ui.lbl_vertical_distribution_file_selection.setText(distribution_file)

        # Load the global options
        gui_util.set_combobox_from_data(self.ui.cbx_export_format_bc, self.all_data.globals.attrs['export_format'])

        # Gridded dataset file selector used by curtain BCs as well
        dset_file = self.bc_data['grid_dataset_file'][0].item()
        if dset_file and self.all_data.does_file_exist(dset_file):
            self.ui.lbl_grid_file_selection.setText(dset_file)
        self._load_grid_data()
        self._initialize_state()

    def _load_grid_data(self):
        """Populate widgets that are only applicable to gridded BCs from persistent data."""
        if self.bc_location != bcd.BC_LOCATION_GRID:
            return  # These parameters not defined on arc, point, or global BC types
        # Time widgets are defined in the BC curve editor for the other types.
        self.ui.grp_time_units.setChecked(bool(int(self.bc_data['define_time_units'][0].item())))
        populate_time_widgets(self, self.bc_data, self.time_format_qt)
        # Default, offset, and scale not used by other BC location types.
        self.ui.edt_default3.setText(str(self.bc_data['default3'][0].item()))
        self.ui.edt_default4.setText(str(self.bc_data['default4'][0].item()))
        self.ui.edt_default5.setText(str(self.bc_data['default5'][0].item()))
        self.ui.edt_default6.setText(str(self.bc_data['default6'][0].item()))
        self.ui.edt_default7.setText(str(self.bc_data['default7'][0].item()))
        self.ui.edt_default8.setText(str(self.bc_data['default8'][0].item()))
        self.ui.edt_offset3.setText(str(self.bc_data['offset3'][0].item()))
        self.ui.edt_offset4.setText(str(self.bc_data['offset4'][0].item()))
        self.ui.edt_offset5.setText(str(self.bc_data['offset5'][0].item()))
        self.ui.edt_offset6.setText(str(self.bc_data['offset6'][0].item()))
        self.ui.edt_offset7.setText(str(self.bc_data['offset7'][0].item()))
        self.ui.edt_offset8.setText(str(self.bc_data['offset8'][0].item()))
        self.ui.edt_scale3.setText(str(self.bc_data['scale3'][0].item()))
        self.ui.edt_scale4.setText(str(self.bc_data['scale4'][0].item()))
        self.ui.edt_scale5.setText(str(self.bc_data['scale5'][0].item()))
        self.ui.edt_scale6.setText(str(self.bc_data['scale6'][0].item()))
        self.ui.edt_scale7.setText(str(self.bc_data['scale7'][0].item()))
        self.ui.edt_scale8.setText(str(self.bc_data['scale8'][0].item()))

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        bc_type = self._get_current_bc_type()
        self.bc_data['type'][0] = bc_type

        # Be careful with subtype because it can return None if we removed all the options.
        subtype_idx = self.ui.cbx_bc_subtype.currentIndex()
        subtype = self.ui.cbx_bc_subtype.itemData(subtype_idx) if subtype_idx >= 0 else self._last_subtype_index + 1
        self.bc_data['subtype'][0] = subtype
        self.bc_data['name'][0] = self.ui.edt_bc_name.text()
        self.bc_data['friction_slope'][0] = float(self.ui.edt_friction_slope.text())
        self.bc_data['define_default'][0] = 1 if self.ui.grp_default.isChecked() else 0
        self.bc_data['default1'][0] = float(self.ui.edt_default1.text())
        self.bc_data['default2'][0] = float(self.ui.edt_default2.text())
        self.bc_data['define_offset'][0] = 1 if self.ui.grp_offset.isChecked() else 0
        self.bc_data['offset1'][0] = float(self.ui.edt_offset1.text())
        self.bc_data['offset2'][0] = float(self.ui.edt_offset2.text())
        self.bc_data['define_scale'][0] = 1 if self.ui.grp_scale.isChecked() else 0
        self.bc_data['scale1'][0] = float(self.ui.edt_scale1.text())
        self.bc_data['scale2'][0] = float(self.ui.edt_scale2.text())
        self.bc_data['define_update_dt'][0] = 1 if self.ui.grp_update_dt.isChecked() else 0
        self.bc_data['update_dt'][0] = float(self.ui.edt_update_dt.text())
        self.bc_data['include_mslp'][0] = 1 if self.ui.tog_include_mslp.isChecked() else 0
        self.bc_data['define_vertical_distribution'][0] = 1 if self.ui.grp_vertical_distribution.isChecked() else 0
        self.bc_data['vertical_distribution_type'][0] = self.ui.cbx_vertical_distribution.itemData(
            self.ui.cbx_vertical_distribution.currentIndex()
        )
        distribution_file = self.ui.lbl_vertical_distribution_file_selection.text()
        if distribution_file != gui_util.NULL_SELECTION:
            self.bc_data['vertical_distribution_file'][0] = distribution_file

        # Save the global options
        self.all_data.globals.attrs['export_format'] = self.ui.cbx_export_format_bc.itemData(
            self.ui.cbx_export_format_bc.currentIndex()
        )

        # Gridded dataset file selector used by both gridded and curtain BCs.
        dset_file = self.ui.lbl_grid_file_selection.text()
        if dset_file != gui_util.NULL_SELECTION:
            self.bc_data['grid_dataset_file'][0] = dset_file
        self._save_grid_data()

    def _save_grid_data(self):
        """Store widget values specific to gridded BC types in the persistent dataset on 'OK'."""
        if self.bc_location != bcd.BC_LOCATION_GRID:
            return  # The rest of the parameters not defined on arc, point, or global BC types

        # Time widgets are defined in the BC curve editor for the other types.
        self.bc_data['define_time_units'][0] = 1 if self.ui.grp_time_units.isChecked() else 0
        extract_time_widget_data(self, self.bc_data)
        # Default, offset, and scale not used by other BC location types.
        self.bc_data['default3'][0] = float(self.ui.edt_default3.text())
        self.bc_data['default4'][0] = float(self.ui.edt_default4.text())
        self.bc_data['default5'][0] = float(self.ui.edt_default5.text())
        self.bc_data['default6'][0] = float(self.ui.edt_default6.text())
        self.bc_data['default7'][0] = float(self.ui.edt_default7.text())
        self.bc_data['default8'][0] = float(self.ui.edt_default8.text())
        self.bc_data['offset3'][0] = float(self.ui.edt_offset3.text())
        self.bc_data['offset4'][0] = float(self.ui.edt_offset4.text())
        self.bc_data['offset5'][0] = float(self.ui.edt_offset5.text())
        self.bc_data['offset6'][0] = float(self.ui.edt_offset6.text())
        self.bc_data['offset7'][0] = float(self.ui.edt_offset7.text())
        self.bc_data['offset8'][0] = float(self.ui.edt_offset8.text())
        self.bc_data['scale3'][0] = float(self.ui.edt_scale3.text())
        self.bc_data['scale4'][0] = float(self.ui.edt_scale4.text())
        self.bc_data['scale5'][0] = float(self.ui.edt_scale5.text())
        self.bc_data['scale6'][0] = float(self.ui.edt_scale6.text())
        self.bc_data['scale7'][0] = float(self.ui.edt_scale7.text())
        self.bc_data['scale8'][0] = float(self.ui.edt_scale8.text())

    def _update_vector_input_state(self, index):
        """Update the state of widgets that are specific to arc types with more than one standard column vs. X.

        Args:
            index (int): 0-base combobox index of the current arc BC type

        Returns:
            bool: True if the current BC type is a curtain
        """
        is_curtain = index == const.BC_TYPE_WL_CURT  # Only WL_CURT for now
        is_vector = is_curtain or index == const.BC_TYPE_WLS
        self.ui.lbl_offset2.setVisible(is_vector)
        self.ui.edt_offset2.setVisible(is_vector)
        self.ui.lbl_scale2.setVisible(is_vector)
        self.ui.edt_scale2.setVisible(is_vector)
        self.ui.lbl_default2.setVisible(is_vector)
        self.ui.edt_default2.setVisible(is_vector)
        return is_curtain

    def _update_curtain_state(self, is_curtain):
        """Update the state of widgets that are specific to curtain boundaries when the arc BC type changes.

        Args:
            is_curtain (bool): True if the current arc BC type is a curtain boundary
        """
        self.ui.lbl_grid_file.setVisible(is_curtain)
        self.ui.btn_grid_file.setVisible(is_curtain)
        self.ui.lbl_grid_file_selection.setVisible(is_curtain)
        self.ui.btn_grid_variables.setVisible(is_curtain)
        if is_curtain:  # Read-only if a NetCDF file
            self.ui.btn_define_curve.setText('View Curve...')
            # Disable variables and curve buttons if no file selected
            self._update_grid_variables_button_state()
        else:
            self.ui.btn_define_curve.setText('Define Curve...')

    def _update_wave_state(self):
        """Hide/show widgets only applicable to wave boundaries."""
        bc_type = self.ui.cbx_bc_type.itemData(self.ui.cbx_bc_type.currentIndex())
        visible = bc_type in bcd.MSLP_BC_TYPES
        self.ui.tog_include_mslp.setVisible(visible)

    def on_arc_bc_type_changed(self, index):
        """Called when the BC type combobox for an arc BC is changed.

        Args:
            index (int): Combobox option index of the new BC type
        """
        # Hide everything if unassigned or ZG
        is_unassigned = index in [const.BC_TYPE_MONITOR, const.BC_TYPE_ZG]
        if is_unassigned:
            self._show_all_feature_widgets(False)
        else:
            self._show_all_feature_widgets(True)

        # If a sloped WSE type, there are two columns in the curve (plus time). Curtains have more variable number, but
        # for now we only support WL_CURT - TIME, CHAINAGE, ELEVATION, WL where the last two values may be defaulted.
        is_curtain = self._update_vector_input_state(index)
        # Hide the "Define Curve..." button if QN, ZG, or unassigned
        is_qn = index == const.BC_TYPE_QN
        self.ui.btn_define_curve.setVisible(not is_qn and not is_unassigned)
        # Hide/show friction slope
        self.ui.layout_friction_slope.setVisible(is_qn)
        # Hide show curtain specific stuff
        self._update_curtain_state(is_curtain)

        # Enable/disable subtypes
        num_subtypes = -1
        if index == const.BC_TYPE_QN:
            num_subtypes = 2
        elif index in [const.BC_TYPE_Q, const.BC_TYPE_HQ]:
            num_subtypes = 4
        elif index in [const.BC_TYPE_WL, const.BC_TYPE_WLS, const.BC_TYPE_WL_CURT]:
            num_subtypes = 6
        self._refresh_subtype_cbx(num_subtypes)

        # Update text in labels for new type
        self._update_label_text(index)

        # Hide/show widgets only applicable to wave types
        self._update_wave_state()

    def on_point_bc_type_changed(self, index):
        """Called when the BC type combobox for a point BC is changed.

        Args:
            index (int): Combobox option index of the new BC type
        """
        # No unassigned point type
        self._show_all_feature_widgets(True)

        # Enable/disable subtypes
        num_subtypes = -1
        if index == const.BC_TYPE_QC:
            num_subtypes = 2
        self._refresh_subtype_cbx(num_subtypes)

        # Update text in labels for new type
        self._update_label_text(index)

    def on_export_format_changed(self, index):
        """Called when the global export combobox option is changed.

        Args:
            index (int): Combobox option index
        """
        if index == const.GLOBAL_2DM_IDX:
            self.ui.lbl_gis_export_warning.setVisible(True)
        else:  # index == GLOBAL_SHAPEFILE_IDX
            self.ui.lbl_gis_export_warning.setVisible(False)

    def on_global_bc_type_changed(self, index):
        """Called when the BC type combobox for a global BC is changed.

        Notes:
            This currently doesn't do anything, but I hooked up the signal/slot so it follows the pattern of the other
            BC location types. Also will probably need to do more as we support other global BC types.

        Args:
            index (int): Combobox option index of the new BC type
        """
        # Update text in labels for new type
        self._update_label_text(index)

    def on_gridded_bc_type_changed(self, index):
        """Called when the BC type combobox for a gridded BC is changed.

        Args:
            index (int): Combobox option index of the new BC type
        """
        # Update text in labels for new type
        self._update_label_text(index)
        self._show_grid_parameters(index)
        self._update_wave_state()

    def on_btn_vertical_distribution(self):
        """Called when the vertical distribution file selector button is clicked."""
        gui_util.select_file(self, self.ui.lbl_vertical_distribution_file_selection,
                             'Select a vertical distribution file', 'All Files (*.*)',
                             self.all_data.info.attrs['proj_dir'], False)

    def on_btn_grid_file(self):
        """Called when the grid dataset file selector is clicked."""
        gui_util.select_file(self, self.ui.lbl_grid_file_selection, 'Select a gridded dataset file',
                             'NetCDF Files(*.nc);;All Files (*.*)', self.all_data.info.attrs['proj_dir'], False)
        self._update_grid_variables_button_state()

    def on_btn_define_curve(self):
        """Slot called when 'Define Curve...' button is clicked to display the XY series editor."""
        bc_type = self._get_current_bc_type()
        is_curtain = bc_type in bcd.CURTAIN_BC_TYPES
        dist_df = None
        variable_dset = None
        filename = None
        if is_curtain:
            # Build up a single column DataFrame of the times for the table. We will plot elevation and WL vs distance
            # along the arc, updating as the user changes timesteps.
            df, dist_df, variable_dset, filename = self._get_netcdf_data_for_curtain()
            if df is None or dist_df is None or variable_dset is None or filename is None:
                return  # Dataset closed in _get_netcdf_data_for_curtain() if opened and error
        else:
            bc_curve = self.all_data.get_bc_curve(self.comp_id, bc_type, True)
            df = bc_curve.to_dataframe()

        QApplication.setOverrideCursor(Qt.WaitCursor)  # This can take a second to load with a big curve
        defaults = None
        if self.ui.grp_default.isChecked():
            # Think there will be more that just two if using other modules and BC types, but for now the maximum
            # number of values we plot against X is 2.
            defaults = [float(self.ui.edt_default1.text()), float(self.ui.edt_default2.text())]
        x_is_time = df.columns[0] == 'Time'

        dlg = BcSeriesEditor(bc_data=self.bc_data, bc_curve=df, series_name=bc_type, x_is_time=x_is_time,
                             defaults=defaults, time_formats=(self.time_format_std, self.time_format_qt),
                             dist_df=dist_df, variable_dset=variable_dset, netcdf_filename=filename, parent=self)
        QApplication.restoreOverrideCursor()
        if dlg.exec() and not is_curtain:
            bc_curve = dlg.model.data_frame.to_xarray()
            self.all_data.set_bc_curve(self.comp_id, bc_type, bc_curve)
        if variable_dset is not None:
            variable_dset.close()

    def on_tog_use_isodate(self, state):
        """Called when use ISODATE toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        use_isodate = state == Qt.Checked
        self.ui.edt_reference_time.setVisible(not use_isodate)
        self.ui.date_reference_time.setVisible(use_isodate)
        label_text = 'Reference date' if use_isodate else 'Reference time (h)'
        self.ui.lbl_reference_time.setText(label_text)

    def on_btn_grid_variables(self):
        """Slot called when 'Define Variables...' button is clicked (only applicable to gridded and curtain BCs)."""
        abs_path = xfs.resolve_relative_path(self.all_data.info.attrs['proj_dir'],
                                             self.ui.lbl_grid_file_selection.text())
        bc_type = self.ui.cbx_bc_type.itemData(self.ui.cbx_bc_type.currentIndex())
        dlg = GridVariablesDialog(bc_data=self.bc_data, bc_type=bc_type, grid_bc_file=abs_path, parent=self)
        dlg.exec()

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()
