"""QxTableView implementation."""
# 1. Standard python modules
import re

# 2. Third party modules
from PySide2.QtCore import QModelIndex
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QApplication

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon
from xms.guipy.widgets.qx_table_view import QxTableView

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


def is_float(value):
    """Check if a value is a float.

    Args:
        value (object): The object to test

    Returns:
        bool: True if the value is a float, False otherwise
    """
    try:
        float(value)
    except (TypeError, ValueError):
        return False
    return True


class BcSeriesTableView(QxTableView):
    """QxTableView implementation for use in TUFLOWFV packages."""

    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)
        self.x_is_time_column = False
        self.paste_delimiter = ',|\t'  # Overwrite if want to support pasting text that is not tab or comma delimited.

    def _check_time_dtype(self, clipboard_rows):
        """Ensure the dtype of the time column being pasted matches the existing rows.

        Args:
            clipboard_rows (list[str]): Rows being pasted.

        Returns:
            bool: False if data being pasted is incompatible with the current time column dtype
        """
        if not self.x_is_time_column or not clipboard_rows:
            return True  # X column is not time, can only paste rows of floats
        # Check the dtype of the current time column
        currently_float = is_float(self.model().defaults['Time'])
        # Check the dtype of the first row of data to be pasted
        first_row = re.split(self.paste_delimiter, clipboard_rows[0])
        if not first_row:
            return True
        pasting_float = is_float(first_row[0])
        if pasting_float == currently_float:
            return True  # dtype of current time column matches data being pasted
        # dtype of data being pasted does not match data in existing rows' time column
        msg = 'Unable to paste data because data type of the time column does not match existing rows. Ensure ' \
              'the currently selected time units match the data being pasted.'
        message_with_ok(parent=self, message=msg, app_name='SMS', win_icon=QIcon(get_xms_icon()))
        return False

    def on_paste(self):
        """Pastes data from the clipboard into the selected cells."""
        self.pasting = True
        text = QApplication.clipboard().text()
        clipboard_rows = list(filter(None, text.split("\n")))
        init_index = self.selectedIndexes()[0]
        init_row = init_index.row()
        init_col = init_index.column()

        if not self._check_time_dtype(clipboard_rows):
            self.pasting = False
            return

        # Insert rows if necessary
        if self.model().rowCount() < init_row + len(clipboard_rows):
            count = init_row + len(clipboard_rows) - self.model().rowCount()
            self.model().insertRows(self.model().rowCount(), count)

        selected_count = len(self.selectedIndexes())
        row_offset = 0
        if len(clipboard_rows) == 1 and selected_count > 1:
            # Paste one row into multiple selected rows by repeatedly pasting the one row
            last_index = self.selectedIndexes()[-1]
            last_row = last_index.row()
            for row in range(init_row, last_row + 1):
                self.paste_row(clipboard_index=0, clipboard_rows=clipboard_rows, init_row=row, init_col=init_col,
                               row_offset=row_offset)
        else:
            # Paste one or more rows into the table (doesn't matter how many are selected)
            for i in range(len(clipboard_rows)):
                self.paste_row(clipboard_index=i, clipboard_rows=clipboard_rows, init_row=init_row, init_col=init_col,
                               row_offset=row_offset)

        self.pasting = False
        self.model().dataChanged.emit(QModelIndex(), QModelIndex())  # Needed to update the table view
