"""Qt delegate that displays a button for bringing up the Assign BC dialog."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.gui.bc_grid_definition_dialog import BcGridDefinitionDialog


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def run_grid_definition_dialog(sim_data, grid_id, parent):
    """Run the BC grid definition dialog for a global BC.

    Args:
        sim_data (SimData): The simulation Dataset
        grid_id (int): Grid id of the row to edit
        parent (QWidget): The Qt parent dialog
    """
    dset = sim_data.grid_definitions.where(sim_data.grid_definitions.grid_id == grid_id, drop=True)
    dlg = BcGridDefinitionDialog(parent=parent, grid_definition=dset, sim_data=sim_data)
    if dlg.exec():
        sim_data.update_grid_definition(dlg.grid_definition)


class DefineBcGridButtonDelegate(QStyledItemDelegate):
    """Qt delegate that displays a button for bringing up the Assign BC dialog in the model control dialog."""

    def __init__(self, parent, sim_data):
        """Initializes the class.

        Args:
            parent (QObject): The parent object.
            sim_data (SimData): The simulation data
        """
        super().__init__(parent)
        self._parent_dlg = parent
        self._sim_data = sim_data

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Define Grid...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.

        Returns:
            bool: True if the event is handled
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                row = index.row()
                grid_id = self._sim_data.grid_definitions.coords['grid_id'][row].item()
                run_grid_definition_dialog(sim_data=self._sim_data, grid_id=grid_id, parent=self._parent_dlg)
                return True
        return super().editorEvent(event, model, option, index)
