"""Class to help with checking default values for the namelists namelist."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class WW3NamelistsDefaultsUtil:
    """Class to find out whether values in a namelist are non-default, used when writing out namelists namelist."""
    def __init__(self):
        """Constructor."""
        self._namelists_stored = [
            'FLX3',
            'FLX4',
            'SLN1',
            'SIN1',
            'SIN2',
            'SIN3',
            'SIN6',
            'SNL1',
            'SNL2',
            'SNL3',
            'SNL4',
            'SNLS',
            'SDS1',
            'SDS2',
            'SDS3',
            'SDS6',
            'SWL6',
            'SBT1',
            'SDB1',
            'SIC2',
            'SIS2',
            'STR1',
            'REF1',
            'SIG1',
            'PRO1',
            'PRO2',
            'PRO3',
            'UNST',
            'PSMC',
            'MISC',
            'OUTS',
            'FLD1',
            'FLD2',
        ]

    def _get_namelist_defaults(self, namelist_name):
        """Gets the SimData keys, WW3 string names for file I/O, and default values of the namelist passed in.

        Data taken from:  https://github.com/NOAA-EMC/WW3/blob/develop/model/nml/namelists.nml
        Tables, such as ANL2 and ANL3 are not included, and will need to be manually checked.

        Args:
            namelist_name (:obj:`str`): The name of the namelist.

        Returns:
            (:obj:`tuple`):
                Tuple of:

                    (:obj:`list[str]`):  The keys in the SimData that have non-default values.

                    (:obj:`list[str]`):  The WW3 variable names of the same non-default values.

                    (:obj:`list`):  The WW3 default values (as found in the SimData)
        """
        keys = []
        vars = []
        values = []

        if namelist_name == 'FLX3':
            keys = [
                'FLX3CDMAX',
                'FLX3CTYPE',
            ]
            vars = [
                'CDMAX',
                'CTYPE',
            ]
            values = [
                0.0,
                0,
            ]
        elif namelist_name == 'FLX4':
            keys = [
                'FLX4CDFAC',
            ]
            vars = [
                'CDFAC',
            ]
            values = [
                0.0,
            ]
        elif namelist_name == 'SLN1':
            keys = [
                'SLN1CLIN',
                'SLN1RFPM',
                'SLN1RFHF',
            ]
            vars = [
                'CLIN',
                'RFPM',
                'RFHF',
            ]
            values = [
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SIN1':
            keys = [
                'SIN1CINP',
            ]
            vars = [
                'CINP',
            ]
            values = [
                0.0,
            ]
        elif namelist_name == 'SIN2':
            keys = [
                'ZWND',
                'SIN2SWELLF',
                'SIN2STABSH',
                'SIN2STABOF',
                'SIN2CNEG',
                'SIN2CPOS',
                'SIN2FNEG',
            ]
            vars = [
                'ZWND',
                'SWELLF',
                'STABSH',
                'STABOF',
                'CNEG',
                'CPOS',
                'FNEG',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SIN3':
            keys = [
                'ZWND',
                'ALPHA0',
                'Z0MAX',
                'BETAMAX',
                'SINTHP',
                'ZALP',
                'TAUWSHELTER',
                'SWELLFPAR',
                'SWELLF',
                'SWELLF2',
                'SWELLF3',
                'SWELLF4',
                'SWELLF5',
                'Z0RAT',
            ]
            vars = [
                'ZWND',
                'ALPHA0',
                'Z0MAX',
                'BETAMAX',
                'SINTHP',
                'ZALP',
                'TAUWSHELTER',
                'SWELLFPAR',
                'SWELLF',
                'SWELLF2',
                'SWELLF3',
                'SWELLF4',
                'SWELLF5',
                'Z0RAT',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                1.43,
                0.0,
                0.0,
                0.3,
                'TC 1996',
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SIN6':
            keys = [
                'SIN6SINA0',
                'SIN6SINU10',
            ]
            vars = [
                'SINA0',
                'SINWS',
            ]
            values = [
                0.0,
                0.0,
            ]
        elif namelist_name == 'SNL1':
            keys = [
                'SNL1LAMBDA',
                'SNL1NLPROP',
                'SNL1KDCONV',
                'SNL1KDMIN',
                'SNL1SNLCS1',
                'SNL1SNLCS2',
                'SNL1SNLCS3',
            ]
            vars = [
                'LAMBDA',
                'NLPROP',
                'KDCONV',
                'KDMIN',
                'SNLCS1',
                'SNLCS2',
                'SNLCS3',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SNL2':
            # Note:  missing NDEPTH
            # Note:  ANL2 table/namelist not included
            keys = [
                'SNL2IQTYPE',
                'SNL2TAILNL',
            ]
            vars = [
                'IQTYPE',
                'TAILNL',
            ]
            values = [
                0,
                0.0,
            ]
        elif namelist_name == 'SNL3':
            # Note:  missing NQDEF
            # Note:  ANL3 table/namelist not included
            keys = [
                'SNL3MSC',
                'SNL3NSC',
                'SNL3KDFD',
                'SNL3KDFS',
            ]
            vars = [
                'MSC',
                'NSC',
                'KDFD',
                'KDFS',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SNL4':
            keys = [
                'SNL4INDTSA',
                'SNL4ALTLP',
            ]
            vars = [
                'INDTSA',
                'ALTLP',
            ]
            values = [
                0,
                0,
            ]
        elif namelist_name == 'SNLS':
            keys = [
                'SNLSA34',
                'SNLSFHFC',
                'SNLSDMN',
                'SNLSFC13',
            ]
            vars = [
                'A34',
                'FHFC',
                'DMN',
                'FC1-3',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SDS1':
            keys = [
                'SDS1CDIS',
                'SDS1APM',
            ]
            vars = [
                'CDIS',
                'APM',
            ]
            values = [
                0.0,
                0.0,
            ]
        elif namelist_name == 'SDS2':
            keys = [
                'SDS2SDSA0',
                'SDS2SDSA1',
                'SDS2SDSA2',
                'SDS2SDSB0',
                'SDS2SDSB1',
                'SDS2PHIMIN',
            ]
            vars = [
                'SDS2SDSA0',
                'SDS2SDSA1',
                'SDS2SDSA2',
                'SDS2SDSB0',
                'SDS2SDSB1',
                'SDS2PHIMIN',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SDS3':
            keys = [
                'SDS3SDSC1',
                'SDS3MNMEANP',
                'SDS3WNMEANPTAIL',
                'SDS3SDSDELTA1',
                'SDS3SDSDELTA2',
                'SDS3SDSLF',
                'SDS3SDSHF',
                'SDS3SDSC2',
                'SDS3SDSC4',
                'SDS3SDSBR',
                'SDS3SDSP',
                'SDS3SDSBR2',
                'SDS3SDSC5',
                'SDS3SDSC6',
                'SDS3SDSDTH',
            ]
            vars = [
                'SDSC1',
                'MNMEANP',
                'WNMEANPTAIL',
                'SDSDELTA1',
                'SDSDELTA2',
                'SDSLF',
                'SDSHF',
                'SDSC2',
                'SDSC4',
                'SDSBR',
                'SDSP',
                'SDSBR2',
                'SDSC5',
                'SDSC6',
                'SDSDTH',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SDS6':
            keys = [
                'SDS6SDSET',
                'SDS6SDSA1',
                'SDS6SDSA2',
                'SDS6SDSP1',
                'SDS6SDSP2',
            ]
            vars = [
                'SDSET',
                'SDSA1',
                'SDSA2',
                'SDSP1',
                'SDSP2',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SWL6':
            keys = [
                'SWL6SWLB1',
            ]
            vars = [
                'SWLB1',
            ]
            values = [
                0.0,
            ]
        elif namelist_name == 'SBT1':
            keys = [
                'SBT1GAMMA',
            ]
            vars = [
                'GAMMA',
            ]
            values = [
                0.0,
            ]
        elif namelist_name == 'SDB1':
            keys = [
                'BJALFA',
                'BJGAM',
                'BJFLAG',
            ]
            vars = [
                'BJALFA',
                'BJGAM',
                'BJFLAG',
            ]
            values = [
                1.0,
                0.73,
                'Use Hmax/d Ratio Only',
            ]
        elif namelist_name == 'SIC2':
            keys = [
                'SIC2IC2DISPER',
                'SIC2IC2TURB',
                'SIC2IC2ROUGH',
                'SIC2IC2REYNOLDS',
                'SIC2IC2SMOOTH',
                'SIC2IC2VISC',
            ]
            vars = [
                'IC2DISPER',
                'IC2TURB',
                'IC2ROUGH',
                'IC2REYNOLDS',
                'IC2SMOOTH',
                'IC2VISC',
            ]
            values = [
                0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SIS2':
            keys = [
                'SIS2ISC1',
                'SIS2IS2BACKSCAT',
                'SIS2IS2BREAK',
                'SIS2IS2C1',
                'SIS2IS2C2',
                'SIS2IS2C3',
                'SIS2ISBACKSCAT',
                'SIS2IS2DISP',
                'SIS2FRAGILITY',
                'SIS2IS2DMIN',
                'SIS2IS2DAMP',
                'SIS2',
            ]
            vars = [
                'SIS2ISC1',
                'SIS2IS2BACKSCAT',
                'SIS2IS2BREAK',
                'SIS2IS2C1',
                'SIS2IS2C2',
                'SIS2IS2C3',
                'SIS2ISBACKSCAT',
                'SIS2IS2DISP',
                'SIS2FRAGILITY',
                'SIS2IS2DMIN',
                'SIS2IS2DAMP',
                'SIS2',
            ]
            values = [
                1.0,
                1.0,
                0,
                0.0,
                0.0,
                0.0,
                0.0,
                0,
                0.0,
                0.0,
                0.0,
                0,
            ]
        elif namelist_name == 'STR1':
            keys = [
                'STR1PTRIAD1',
                'STR1PTRIAD2',
                'STR1PTRIAD3',
                'STR1PTRIAD4',
                'STR1PTRIAD5',
            ]
            vars = [
                'PTRIAD1',
                'PTRIAD2',
                'PTRIAD3',
                'PTRIAD4',
                'PTRIAD5',
            ]
            values = [
                0.05,
                2.5,
                10.0,
                0.2,
                0.01,
            ]
        elif namelist_name == 'REF1':
            keys = [
                'REF1REFCOAST',
                'REF1REFFREQ',
                'REF1REFMAP',
                'REF1REFRMAX',
                'REF1REFFREQPOW',
                'REF1REFICEBERG',
                'REF1REFSUBGRID',
                'REF1REFCOSP_STRAIGHT',
            ]
            vars = [
                'REFCOAST',
                'REFFREQ',
                'REFMAP',
                'REFRMAX',
                'REFFREQPOW',
                'REFICEBERG',
                'REFSUBGRID',
                'REFCOSP_STRAIGHT',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'SIG1':
            keys = [
                'SIG1IGMETHOD',
                'SIG1IGADDOUTP',
                'SIG1IGSOURCE',
                'SIG1IGSTERMS',
                'SIG1IGMAXFREQ',
                'SIG1IGEMPIRICAL',
                'SIG1IGBCOVERWRITE',
                'SIG1IGSWELLMAX',
            ]
            vars = [
                'IGMETHOD',
                'IGADDOUTP',
                'IGSOURCE',
                'IGSTERMS',
                'IGMAXFREQ',
                'IGEMPIRICAL',
                'IGBCOVERWRITE',
                'IGSWELLMAX',
            ]
            values = [
                0,
                0.0,
                1,
                0.0,
                0.0,
                0.0,
                0,
                0,
            ]
        elif namelist_name == 'PRO1':
            keys = [
                'PROCFLTM',
            ]
            vars = [
                'CFLTM',
            ]
            values = [
                0.0,
            ]
        elif namelist_name == 'PRO2':
            keys = [
                'PROCFLTM',
                'PRO2DTIME',
                'PRO2LATMIN',
            ]
            vars = [
                'CFLTM',
                'DTIME',
                'LATMIN',
            ]
            values = [
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'PRO3':
            keys = [
                'PROCFLTM',
                'PRO3WDTHCG',
                'PRO3WDTHTH',
            ]
            vars = [
                'CFLTM',
                'WDTHCG',
                'WDTHTH',
            ]
            values = [
                0.0,
                0.0,
                0.0,
            ]
        elif namelist_name == 'UNST':
            keys = [
                'UGOBCAUTO',
                'UGOBCDEPTH',
                'UNSTEXPFSN',
                'UNSTEXPFSPSI',
                'UNSTEXPFSFCT',
                'UNSTIMPFSN',
                'UNSTIMPTOTAL',
                'UNSTEXPTOTAL',
                'UNSTIMPREFRACTION',
                'UNSTIMPFREQSHIFT',
                'UNSTIMPSOURCE',
                'JGS_TERMINATE_MAXITER',
                'JGS_TERMINATE_DIFFERENCE',
                'JGS_TERMINATE_NORM',
                'UNSTJGS_USE_JACOBI',
                'UNSTJGS_BLOCK_GAUSS_SEIDEL',
                'JGS_MAXITER',
                'JGS_PMIN',
                'JGS_DIFF_THR',
                'JGS_NORM_THR',
                'UNSTSETUP_APPLY_WLV',
                'UNSTSOLVERTHR_SETUP',
                'UNSTCRIT_DEP_SETUP',
            ]
            vars = [
                'UGOBCAUTO',
                'UGOBCDEPTH',
                'EXPFSN',
                'EXPFSPSI',
                'EXPFSFCT',
                'IMPFSN',
                'IMPTOTAL',
                'EXPTOTAL',
                'IMPREFRACTION',
                'IMPFREQSHIFT',
                'IMPSOURCE',
                'JGS_TERMINATE_MAXITER',
                'JGS_TERMINATE_DIFFERENCE',
                'JGS_TERMINATE_NORM',
                'UNSTJGS_USE_JACOBI',
                'UNSTJGS_BLOCK_GAUSS_SEIDEL',
                'JGS_MAXITER',
                'JGS_PMIN',
                'JGS_DIFF_THR',
                'JGS_NORM_THR',
                'SETUP_APPLY_WLV',
                'SOLVERTHR_SETUP',
                'CRIT_DEP_SETUP',
            ]
            values = [
                'Listed in ww3_grid.nml',
                0.0,
                1,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                0,
                1,
                1,
                0,
                1,
                1,
                100,
                1.0,
                1.0E-8,
                0.0,
                1,
                1.0E-6,
                0.1,
            ]
        elif namelist_name == 'PSMC':
            keys = [
                'PSMCCFLTM',
                'PSMCDTIME',
                'PSMCLATMIN',
                'PSMCRFMAXD',
                'PSMCLvSMC',
                'PSMCISHFT',
                'PSMCJEQT',
                'PSMCNBISMC',
                'PSMCUNO3',
                'PSMCUNO3Add',
            ]
            vars = [
                'CFLTM',
                'DTIME',
                'LATMIN',
                'RFMAXD',
                'LvSMC',
                'ISHFT',
                'JEQT',
                'NBISMC',
                'UNO3',
                'AVERG',
            ]
            values = [
                0.7,
                0.0,
                86.0,
                80.0,
                1,
                0,
                0,
                0,
                0,
                0,
            ]
        elif namelist_name == 'OUTS':
            keys = [
                'E3D',
                'I1E3D',
                'I2E3D',
                'TH1MF',
                'I1TH1MF',
                'I2TH1MF',
                'STH1MF',
                'I1STH1MF',
                'I2STH1MF',
                'TH2MF',
                'I1TH2MF',
                'I2TH2MF',
                'STH2MF',
                'I1STH2MF',
                'I2STH2MF',
                'P2SFOut',
                'I1P2SF',
                'I2P2SF',
            ]
            vars = [
                'E3D',
                'I1E3D',
                'I2E3D',
                'TH1MF',
                'I1TH1MF',
                'I2TH1MF',
                'STH1MF',
                'I1STH1MF',
                'I2STH1MF',
                'TH2MF',
                'I1TH2MF',
                'I2TH2MF',
                'STH2MF',
                'I1STH2MF',
                'I2STH2MF',
                'P2SFOut',
                'I1P2SF',
                'I2P2SF',
            ]
            values = [
                0,
                1,
                1,
                0,
                1,
                1,
                0,
                1,
                1,
                0,
                1,
                1,
                0,
                1,
                1,
                0,
                1,
                1,
            ]
        elif namelist_name == 'MISC':
            keys = [
                'CICE0',
                'CICEN',
                'PMOVE',
                'XSEED',
                'FLAGTR',
                'XP',
                'XR',
                'XFILT',
                'XIHMAX',
                'HSPMIN',
                'WSM',
                'WSC',
                'FLC',
                'NOSW',
                'FMICHE',
                'STDX',
                'STDY',
                'STDT',
                'P2SF',
            ]
            vars = [
                'CICE0',
                'CICEN',
                'PMOVE',
                'XSEED',
                'FLAGTR',
                'XP',
                'XR',
                'XFILT',
                'XIHMAX',
                'HSPMIN',
                'WSM',
                'WSC',
                'FLC',
                'NOSW',
                'FMICHE',
                'STDX',
                'STDY',
                'STDT',
                'P2SF',
            ]
            values = [
                0.0,
                0.0,
                0.0,
                0.0,
                'No subgrid',
                0.0,
                0.0,
                0.0,
                0,
                0.0,
                0.0,
                0.0,
                0.0,
                0,
                1.6,
                0.0,
                0.0,
                0.0,
                0,
            ]
        elif namelist_name == 'FLD1':
            keys = [
                'TAILTYPE',
                'TAILLEV',
                'TAILT1',
                'TAILT2',
            ]
            vars = [
                'TAILTYPE',
                'TAILLEV',
                'TAILT1',
                'TAILT2',
            ]
            values = [
                0,
                0.01,
                1.25,
                3.0,
            ]
        elif namelist_name == 'FLD2':
            keys = [
                'TAILTYPE',
                'TAILLEV',
                'TAILT1',
                'TAILT2',
            ]
            vars = [
                'TAILTYPE',
                'TAILLEV',
                'TAILT1',
                'TAILT2',
            ]
            values = [
                0,
                0.01,
                1.25,
                3.0,
            ]

        return keys, vars, values

    def check_namelist_defaults(self, namelist_name, attrs):
        """For the namelist name passed in, check if the values in the SimData are not default values.

        The function will grab the parts of the SimData associated with the namelist passed in.
        The current SimData values for that namelist will then be compared with the defaults.
        For each value that is non-default, the function will return a list of the key name found in the SimData, and
        the string used to write out that variable in the WW3 namelist.  This will simplify writing out the data for
        the namelist, though some special checks will need to be done for each individual namelist (for example, a
        namelist might only write certain variables is another variable is a certain value).

        The calling function can also decide wheter to write the namelist at all, as it is not necessary if all the
        values in a namelist are defaults.

        Args:
            namelist_name (:obj:`str`):  The name of the namelist to check.
            attrs (:obj:`xarray.Dataset`):  The xarray Dataset containing the data to compare to.

        Returns:
            (:obj:`tuple(list[str],list[str])`):
            Tuple of:

                (:obj:`list[str]`):  The keys in the SimData that have non-default values.

                (:obj:`list[str]`):  The WW3 variable names of the same non-default values.
        """
        diff_keys = []
        diff_vars = []

        if namelist_name in self._namelists_stored:
            keys, vars, default_values = self._get_namelist_defaults(namelist_name)
            for key, var, default_value in zip(keys, vars, default_values):
                if key in attrs:
                    if attrs[key] != default_value:
                        # We found a non default value, store it so we can return it
                        diff_keys.append(key)
                        diff_vars.append(var)
                else:
                    # We didn't find the key
                    raise ValueError(f"Key in namelist not supported: {key}")
        else:
            raise ValueError("Namelist not supported.")
        return diff_keys, diff_vars
