"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import datetime
import os
from pathlib import Path
import re
import sys
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.model.commands.command import BatchCommandItem, CommandManager
from xms.FhwaVariable.core_data.model.commands.tools import ToolItem, ToolManager
from xms.FhwaVariable.core_data.model.files.file_manager import FileManager
from xms.FhwaVariable.core_data.model.project_data.project_data import ProjectsManager
from xms.FhwaVariable.interface_adapters.view_model.main.message_box import MessageBox
from xms.FhwaVariable.interface_adapters.view_model.main.select_file import SelectFile
from xms.FhwaVariable.interface_adapters.view_model.main.tree_model import FolderItem
from xms.FhwaVariable.interface_adapters.view_model.main.window import Orientation
from xms.FhwaVariable.interface_structures.read_manager import ReadManagerBase
from xms.FhwaVariable.interface_structures.write_manager import WriteManagerBase


class ModelBase(CalcData):
    """A class that defines the interactor for a model."""
    def __init__(self, model_name: str, version: str, model_icon: str, project_icon: str, agency: str,
                 developed_by: str, interface_developed_by: str, project_settings, wiki_url: str,
                 about_dict: dict = None, model_file_types: list = None,
                 write_manager: WriteManagerBase = None, read_manager: ReadManagerBase = None,
                 app_folder: str = None, frozen_app_path: str = None, app_data=None):
        """Initializes the Bridge scour CalcData.

        Args:
            model_name (string): name of the model or app
            version (float): version of the model
            model_icon (string): icon for the model
            project_icon (string): icon for the model's project
            agency (string): agency that spnosors the model
            developed_by (string): company that developed the model
            interface_developed_by (string): company that developed the interface
            project_settings (ProjectSettingsCalc): Project settings interactor
            wiki_url (string): URL to the starting page of the help wiki
            about_dict (string): dictionary of about information
            model_file_types (list): list of file types for the model
            write_manager (WriteManager): Write manager
            read_manager (ReadManager): Read manager
            app_folder (string): path to the folder for the app
            frozen_app_path (string): path to the folder for the frozen app
            app_data (AppData): Application data
        """
        current_file_directory = Path(__file__).parent.parent
        two_levels_up_directory = current_file_directory.parent
        icon_folder = two_levels_up_directory / 'resources' / 'icons'

        super().__init__(app_data=app_data)

        # -- Initialize App/About Data --
        # Initialize data to be displayed in the 'About' dialog
        if about_dict is not None:
            self.about_dict = about_dict
        else:
            compile_message, build_date = self.get_build_date()
            self.about_dict = {
                'app_name': model_name,
                'version': version,
                'model_icon': model_icon,
                'project_icon': project_icon,
                'model_website': None,
                'description': None,
                'wiki_url': wiki_url,
                'agency': agency,
                'agency_icon': None,
                'agency_website': None,
                'agency_description': None,
                'developed_by': developed_by,
                'developed_by_icon': None,
                'developed_by_website': None,
                'developed_by_description': '',
                'build_message': compile_message,
                'build_date': build_date
            }

        # Application or model name
        self.model_name = model_name
        self.version = version
        self.model_icon = model_icon
        self.project_icon = project_icon
        # self.model_website = None
        # self.description = description
        self.model_file_types = ["All Files (*.*)"]
        if model_file_types is not None:
            self.model_file_types = model_file_types

        self.wiki_url = wiki_url

        # Agency that owns/sponsors the model
        self.agency = agency
        # self.agency_icon = None
        # self.agency_website = None
        # self.agency_description = None

        # Organization that developed the model
        self.developed_by = developed_by
        # self.developed_by_icon = None
        # self.developed_by_website = None
        # self.developed_by_description = None

        # Organization that developed the interface to the model
        self.interface_developed_by = interface_developed_by
        # self.interface_developed_by_icon = None
        # self.interface_developed_by_website = None
        # self.interface_developed_by_description = ''

        self.app_folder = app_folder
        self.frozen_app_path = frozen_app_path

        if 'build_date' not in self.about_dict:
            self.about_dict['build_message'], self.about_dict['build_date'] = self.get_build_date()

        # -- Plug-ins --
        # # Plug-in the Graphics View
        # This is placed on the App_data

        # Plug-in the View or Delivery mechanism (For example, Pyside6 GUI or Tethys)
        # This is placed on the App_data
        # self.view = view

        # Plug-in the file I/O managers with external modules
        self.write_manager = write_manager
        self.read_manager = read_manager

        # Plug-in the project settings
        self.project_settings = project_settings

        # Initialize the data storage for Application settings
        if app_data is None:
            app_data = AppData(self.name, self.version, self.agency, self.developed_by)

        # -- Functors --
        # declare functors to be access functions available in the presenter (keep this list short)
        self.about_dlg_func = None
        self.exit_func = None

        # -- Model Base data --
        # Initialize the Model Base data (Command Manager, Tool Manager, Calc dict, document list, and file manager)
        if read_manager is not None:
            self.read_or_save_setting_and_profile_data()
        self.command_manager = CommandManager()
        self.initialize_projects_manager()
        self.tool_manager = ToolManager(app_data)
        self.calcdata_dict = {}
        self.calcs_with_no_add_tool = ['Project', 'Folder']
        # This is a dictionary that maps the CalcData class to the CalcData name
        self.calcdata_reverse_dict = None
        # self.profile_list = None
        # self.client_list = []
        self.document_list = []
        self.file_manager = FileManager(app_data)

        # Initialize the tools later
        self.tools_initialized = False

        # Set the icons
        current_file_directory = Path(__file__).parent
        icon_folder = current_file_directory.parent / 'resources' / 'icons'

        self.calcdata_dict['Project'] = FolderItem(
            name='Project data',
            icon=icon_folder / 'folder.png',
            tool_tip='Project data')
        # TODO: Consider if the project should be created as a CalcData item
        # self.key = f'{name} Project'
        # self.calcdata_dict[self.key] = CalcDataItem(
        #     self.key, item_class=self,
        #     icon=os.path.join(icon_folder, 'gvf.ico'),
        #     tool_tip='Gradually-Varied Flow (GVF) CalcData',
        #     menu_group='Channel',
        #     add_toolbar=False,
        #     complexity=0)

        self.calcdata_dict['Folder'] = FolderItem(
            name='Folder',
            icon=icon_folder / 'folder.png',
            tool_tip='Folder')

    def edit_profile(self,):
        """Edit the profile."""
        profile = self.app_data.get_profile()
        # self.profile_icon = icon_folder / 'profile.ico'
        # TODO: Setup logic to create a new profile if '<new>' is selected
        result, new_calc = self.app_data.edit_item_functor(profile, model_name=self.model_name, model_base=self,
                                                           icon=self.app_data.profile_icon)
        if result and not new_calc.read_only and self.check_if_profile_changed(profile, new_calc):
            # TODO: Immediately save profile changes to the file
            new_calc.set_name(new_calc.input['Profile name'].get_val())
            version = new_calc.input['Version'].get_val()
            if version <= profile.input['Version'].get_val():
                version = profile.input['Version'].get_val() + 0.1
            new_calc.set_version(version)
            new_calc.set_modification_date()
            AppData.profile_settings_list[AppData.selected_profile_settings] = new_calc
            self.read_or_save_profiles_from_file(save=True)

    def delete_profile(self,):
        """Delete the profile."""
        return self.app_data.delete_profile()

    def check_if_profile_changed(self, old_profile, new_profile):
        """Check if the profile has changed."""
        return self._check_if_profile_changed_recursive(old_profile.input, new_profile.input)

    def _check_if_profile_changed_recursive(self, old_profile_varlist, new_profile_varlist):
        """Check if the profile has changed."""
        for key in new_profile_varlist:
            if key not in new_profile_varlist:
                return True
            if isinstance(new_profile_varlist[key], dict):
                if isinstance(old_profile_varlist[key], dict):
                    if self._check_if_profile_changed_recursive(old_profile_varlist[key],
                                                                new_profile_varlist[key]):
                        return True
                else:
                    return True
            elif isinstance(old_profile_varlist[key], dict):
                return True
            if hasattr(new_profile_varlist[key], 'value') and hasattr(new_profile_varlist[key].value, 'input'):
                if hasattr(old_profile_varlist[key], 'value') and hasattr(old_profile_varlist[key].value, 'input'):
                    if self._check_if_profile_changed_recursive(old_profile_varlist[key].value.input,
                                                                new_profile_varlist[key].value.input):
                        return True
                else:
                    return True
            elif hasattr(old_profile_varlist[key], 'input'):
                return True
            elif hasattr(old_profile_varlist[key], 'get_val'):
                if hasattr(new_profile_varlist[key], 'get_val'):
                    if new_profile_varlist[key].get_val() != old_profile_varlist[key].get_val():
                        return True
                else:
                    return True
        return False

    def check_stored_paths(self, model_name=None):
        """Check if the stored paths are valid and update them if necessary.

        Args:
            model_name (str): The name of the model
        """
        if not hasattr(self, 'app_folder') or self.app_folder is None:
            self.app_folder = Path(__file__).parent.parent
        if isinstance(self.app_folder, str):
            self.app_folder = Path(self.app_folder)
        if not hasattr(self, 'frozen_app_path') or self.frozen_app_path is None:
            self.frozen_app_path = Path('xms/FhwaVariable/core_data')
        if isinstance(self.frozen_app_path, str):
            self.frozen_app_path = Path(self.frozen_app_path)

        if model_name is None:
            model_name = self.model_name
            if model_name is None:
                model_name = 'FhwaVariable'
        app_name_safe = re.sub(r'[^a-zA-Z0-9]', '_', model_name)  # Sanitize app name for file paths

        return app_name_safe

    def get_build_date(self, model_name=None):
        """
        Retrieve the build date from the build_date.txt file or return today's date if not compiled.

        Returns:
            str: The build date.
        """
        app_name_safe = self.check_stored_paths(model_name)
        filename = f'{app_name_safe}_build_date.txt'
        if getattr(sys, 'frozen', False):  # Check if running in a compiled executable
            try:
                # Read the build date from the embedded file
                base_path = Path(sys._MEIPASS)
                file = base_path / self.frozen_app_path / filename
                if file.exists():
                    with open(file, "r") as f:
                        build_date = f.read().strip()
                        return "Compiled", build_date
            except FileNotFoundError:
                pass
        # If not compiled, return today's date
        try:
            # Read the build date from the embedded file
            file = self.app_folder / filename
            # don't check if file doesn't exist; get error (not exe, so it should just happen to devs)
            with open(file, "r") as f:
                build_date = f.read().strip()
                return "Not Compiled", build_date
        except FileNotFoundError:
            pass
        return "Not Compiled", datetime.datetime.now().strftime('%Y-%m-%d')

    def get_version(self, model_name=None):
        """
        Retrieve the version from the version.txt file or return '0.0.0' if not found.

        Returns:
            str: The version.
        """
        app_name_safe = self.check_stored_paths(model_name)
        filename = f'{app_name_safe}_version.txt'
        if getattr(sys, 'frozen', False):  # Check if running in a compiled executable
            try:
                # Read the build date from the embedded file
                base_path = Path(sys._MEIPASS)
                file = base_path / self.frozen_app_path / filename
                if file.exists():
                    with open(file, "r") as f:
                        version = f.read().strip()
                        return version
            except FileNotFoundError:
                pass
        # If not compiled, return today's date
        try:
            # Read the build date from the embedded file
            file = self.app_folder / filename
            # don't check if file doesn't exist; get error (not exe, so it should just happen to devs)
            with open(file, "r") as f:
                version = f.read().strip()
                return version
        except FileNotFoundError:
            pass
        return '0.0.0'

    def initialize_projects_manager(self):
        """Initialize the projects manager."""
        self.projects_manager = ProjectsManager(model_name=self.model_name,
                                                model_icon=self.model_icon,
                                                project_icon=self.project_icon)

    def reset_project_settings_to_defaults(self):
        """Reset the project settings to the defaults."""
        self.projects_manager.reset_project_settings_to_defaults()

    def read_or_save_setting_and_profile_data(self, save=False):
        """Read the Setting and Profile data from files."""
        result, settings_location = self.get_setting('Settings location')
        if not result:
            # raise ValueError('Settings location not found in settings')
            print('Settings location not found in settings')
            return False
        if not (isinstance(settings_location, Path)):
            settings_location = Path(settings_location)

        settings_file = settings_location / 'settings.set'
        result, settings = self.read_or_save_settings_from_file([AppData.app_settings], settings_file, save)
        if result:
            settings[0].calc_filename = settings_file
            AppData.app_settings = settings[0]

        # profile_file = settings_location / 'profile.set'
        self.read_or_save_profiles_from_file(settings_location, save)

        # Check if there are files that unlock special settings
        AppData.app_settings.input['Preferences'].value.check_file_keys()
        return True

    def read_or_save_profiles_from_file(self, file_path=None, save=False):
        """Read the profiles from a file.

        Args:
            file_path (Path): Path to the file

        Returns:
            result (bool): True if successful
            settings (list): List of settings
        """
        if file_path is None:
            result, file_path = self.get_setting('Settings location')
            if not result:
                # raise ValueError('Settings location not found in settings')
                print('Settings location not found in settings')
                return False
            if not (isinstance(file_path, Path)):
                file_path = Path(file_path)
        profile_var_file = file_path / 'profile.set'
        if not os.path.exists(profile_var_file.parent):
            os.makedirs(profile_var_file.parent)
        # Read profile
        if save is False and profile_var_file.is_file():
            # Using a normal calc not a setting class
            result, calcs, _, _ = self.read_manager.read_from_hdf5_file(profile_var_file, is_setting=False)
            if not result or len(calcs) <= 0:
                return False, None
            calc = calcs[0]
            if 'Profile selection' in calc.input:
                profile_var = calc.input['Profile selection']
                prev_found = True
                index = 1
                while prev_found:
                    # profile_file = file_path / f'profile_{index}.set'
                    prev_found = self.import_profile(index, file_path)
                    index += 1
                index = profile_var.value
            if index is not None and 0 <= index < len(AppData.profile_settings_list):
                AppData.selected_profile_settings = index
        # Write profile
        else:
            AppData.profile_var.value = AppData.selected_profile_settings
            calc = CalcData(app_data=AppData, model_name=self.model_name)
            calc.input['Profile selection'] = AppData.profile_var
            # Check if the file exists, if so if locked, get user feedback on how to proceed
            self.write_manager.write_to_hdf5_file([calc], profile_var_file, True)
            for i in range(1, len(AppData.profile_settings_list)):
                self.export_profile(i, file_path)
        return True

    def export_profile(self, index=None, file_path=None):
        """Export the profile to a file.

        Args:
            index (int): Index of the profile to export
            file_path (Path): Path to the file

        Returns:
            result (bool): True if successful
        """
        if index is None:
            index = AppData.selected_profile_settings
        if not 0 <= index < len(AppData.profile_settings_list):
            return False

        if file_path is None:
            result, file_paths = self.select_file_to_open(["Settings file (*.set)", "All Files (*.*)"])
            if not result:
                return False
            file_path = file_paths[0]

        if not os.path.exists(file_path.parent):
            os.makedirs(file_path.parent)
        profile = AppData.profile_settings_list[index]
        profile_file = file_path / f'profile_{index}.set'
        self.write_manager.write_to_hdf5_file([profile], profile_file, True)
        return True

    def import_profile(self, index, file_path=None):
        """Export the profile to a file.

        Args:
            index (int): Index of the profile to export
            file_path (Path): Path to the file

        Returns:
            result (bool): True if successful
        """
        if file_path is None:
            result, file_paths = self.select_file_to_open(["Settings file (*.set)", "All Files (*.*)"])
            if not result:
                return False
            file_path = file_paths[0]

        if not os.path.exists(file_path.parent):
            os.makedirs(file_path.parent)
        profile_file = file_path / f'profile_{index}.set'
        if profile_file.is_file():
            result, profiles, _, _ = self.read_manager.read_from_hdf5_file(profile_file, is_setting=True)
            if result and len(profiles) > 0:
                profile = profiles[0]
                self.app_data.add_profile(profile=profile)
            else:
                return False
        else:
            return False
        return True

    def read_or_save_settings_from_file(self, settings, file_path, save=False):
        """Read the settings from a file.

        Args:
            settings (list): List of settings to read
            file_path (Path): Path to the file

        Returns:
            result (bool): True if successful
            settings (list): List of settings
        """
        if save is False and os.path.exists(file_path):
            result, calcdatas, _, _ = self.read_manager.read_from_hdf5_file(file_path, is_setting=True)
            if result and len(calcdatas) > 0:
                return True, calcdatas
        else:
            if not os.path.exists(file_path.parent):
                os.makedirs(file_path.parent)
            # Check if the file exists, if so if locked, get user feedback on how to proceed
            self.write_manager.write_to_hdf5_file(settings, file_path, True)
        return False, None

    def initialize_tools(self, about_icon=None):
        """Initialize the tools for the model.

        Args:
            about_icon (string): path to the about icon
        """
        # This is run once when the GUI is initialized
        # This attaches the commands from the interactor/controller to the tools
        if self.tools_initialized:
            return

        tm = self.tool_manager

        # File
        tm.tools['File']['New'].command = self.new
        tm.tools['File']['Add project'].command = self.add_project
        tm.tools['File']['Open'].command = self.open
        if self.read_manager is None:
            tm.tools['File']['Open'].is_enabled = False
        tm.tools['File']['Save all'].command = self.save_all
        tm.tools['File']['Save as'].command = self.save_as_from_interface
        if self.write_manager is None:
            tm.tools['File']['Save all'].is_enabled = False
            tm.tools['File']['Save as'].is_enabled = False
        tm.tools['File']['Exit'].command = AppData.exit_functor

        # Edit
        tm.tools['Edit']['Support window'].command = AppData.create_support_window
        _, show_wiki = self.get_setting('Show wiki panel')
        tm.tools['Edit']['Show wiki panel'].is_checked = show_wiki
        tm.tools['Edit']['Show wiki panel'].command = self.toggle_show_wiki_panel
        tm.tools['Edit']['Undo'].command = self.undo
        tm.tools['Edit']['Redo'].command = self.redo
        tm.tools['Edit']['Copy'].command = self.copy
        tm.tools['Edit']['Cut'].command = self.cut
        tm.tools['Edit']['Paste'].command = self.paste
        tm.tools['Edit']['Select all'].command = self.set_all_items_selected

        # Display
        if 'Display' in tm.tools:
            tm.tools['Display']['Document window'].command = self.create_document_window
            tm.tools['Display']['Close document window'].command = self.close_document_window
            tm.tools['Display']['Refresh'].command = self.refresh
            tm.tools['Display']['Frame'].command = self.frame
            tm.tools['Display']['Plan view'].command = self.plan_view
            tm.tools['Display']['Front view'].command = self.front_view
            tm.tools['Display']['Side view'].command = self.side_view
            tm.tools['Display']['Oblique view'].command = self.oblique_view
            # self.tools['Display']['Previous view']
            # self.tools['Display']['Next view']

        # Calculators
        add_calc = self.add_calcdata_by_name

        for calc in self.calcdata_dict:
            if calc in self.calcs_with_no_add_tool:
                continue
            tm.tools['Calculators'][f'New {calc}'] = ToolItem(f'New {calc}', command=add_calc,
                                                              tool_tip=f'Add a new {calc} to the project',
                                                              icon=self.calcdata_dict[calc].icon,
                                                              menu_group=self.calcdata_dict[calc].menu_group,
                                                              add_toolbar=self.calcdata_dict[calc].add_toolbar,
                                                              complexity=self.calcdata_dict[calc].complexity)
            tm.tools['Calculators'][f'New {calc}'].set_args_and_kwargs(calc)

        tm.tools['Calculators']['Delete Calculator'].command = self.delete_selected
        tm.tools['Calculators']['Duplicate Calculator'].command = self.duplicate_selected

        # Profile
        tm.tools['Profiles']['Edit'].command = self.edit_profile
        tm.tools['Profiles']['Delete'].command = self.delete_profile
        tm.tools['Profiles']['Import profile file'].command = self.import_profile
        tm.tools['Profiles']['Export profile file'].command = self.export_profile

        # Units

        # Map

        # Windows
        if 'Windows' in tm.tools:
            for k in tm.tools['Windows']:
                for key in tm.tools['Windows'][k]:
                    tm.tools['Windows'][k][key].command = self.set_layout
                    tm.tools['Windows'][k][key].set_args_and_kwargs(key)

            # doc_len = len(AppData.doc_windows)
            # if 0 < doc_len <= 6:
            #     for key in tm.tools['Windows'][doc_len]:
            #         tm.tools['Windows'][doc_len][key].command = self.set_layout
            #         tm.tools['Windows'][doc_len][key].set_args_and_kwargs(key)

        # Help
        tm.tools['Help']['About'].command = AppData.about_functor
        tm.tools['Help']['About'].icon = about_icon

        # Project
        tm.tools['Project']['Close project'].command = self.close_project

        self.tools_initialized = True

    # def set_interactor_functors(self, about_dlg_func, exit_func):
    #     """Set the interactor functors.

    #     Args:
    #         about_dlg_func (function): function to open the about dialog
    #         exit_func (function): function to exit the model
    #     """
    #     # Keep this list short!
    #     tm = self.tool_manager
    #     tm.tools['File']['Exit'].command = exit_func
    #     tm.tools['Help']['About'].command = about_dlg_func

    # Display commands
    def create_document_window(self, window_type='graphics'):
        """Create a document window."""
        return AppData.create_document_window(window_type=window_type)

    def close_document_window(self):
        """Close the document window."""
        doc = AppData.get_document_window()
        if doc is None:
            return False
        return AppData.close_document_window(doc.window_uuid)

    def refresh(self):
        """Refresh the view."""
        return True

    def frame(self):
        """Frame the view."""
        doc = AppData.get_document_window()
        if doc is None or doc.window_type != 'graphics':
            return False
        if AppData.graphics_view is None:
            return False
        doc_uuid = doc.window_uuid
        doc.camera_orientation = AppData.graphics_view.frame_view(doc_uuid)
        return True

    def plan_view(self):
        """Plan view."""
        return self.set_orientation(bearing=0.0, dip=90.0)

    def front_view(self):
        """Front view."""
        return self.set_orientation(bearing=0.0, dip=0.0)

    def side_view(self):
        """Side view."""
        return self.set_orientation(bearing=90.0, dip=0.0)

    def oblique_view(self):
        """Oblique view."""
        return self.set_orientation(bearing=45, dip=30.0)

    def bottom_view(self):
        """Oblique view."""
        return self.set_orientation(bearing=0.0, dip=-90.0)

    def set_orientation(self, orientation: Orientation = None, bearing: float = None, dip: float = None,
                        at: tuple[float, float, float] = None, width: float = None,
                        height: float = None):
        """Set the orientation.

        Args:
            orientation (Orientation): The orientation to set
            bearing (float): The bearing to set
            dip (float): The dip to set
            at (tuple): The point to set
            width (float): The width to set
            height (float): The height to set
        """
        result = False
        doc = AppData.get_document_window()
        if doc is not None and doc.window_type == 'graphics':
            if orientation is not None:
                doc.camera_orientation = orientation
                result = True
            else:
                if bearing is not None:
                    doc.camera_orientation.bearing = bearing
                    result = True
                if dip is not None:
                    doc.camera_orientation.dip = dip
                    result = True
                if at is not None:
                    doc.camera_orientation.at = at
                    result = True
                if width is not None:
                    doc.camera_orientation.width = width
                    result = True
                if height is not None:
                    doc.camera_orientation.height = height
                    result = True
        return result

    def set_layout(self, layout_name):
        """Set the layout.

        Args:
            layout_name (string): The name of the layout to set
        """
        if layout_name is None or layout_name == '':
            return False
        AppData.doc_windows_layout = layout_name
        return True

    def perform_action(self, tool_uuid, right_click_uuid, current_text=''):
        """Perform an action.

        Args:
            tool_uuid (string): uuid of the tool
            right_click_uuid (string): uuid of the right click
            current_text (string): current
        """
        return self._perform_action_on_tool_dict(self.tool_manager.tools, tool_uuid, right_click_uuid, current_text)

    def _perform_action_on_tool_dict(self, tool_dict, tool_uuid, right_click_uuid, current_text=''):
        """Perform an action on a tool dictionary.

        Args:
            tool_dict (dict): dictionary of tools
            tool_uuid (string): uuid of the tool
            right_click_uuid (string): uuid of the right click
            current_text (string): current text
        """
        for tool_name in tool_dict:
            tool = tool_dict[tool_name]
            if isinstance(tool, dict):
                result, data = self._perform_action_on_tool_dict(tool, tool_uuid, right_click_uuid, current_text)
                if result:
                    return result, data
            else:
                if tool.uuid == tool_uuid:
                    return self._perform_action_on_tool(tool, right_click_uuid, current_text)
        return False, None

    def _perform_action_on_tool(self, tool, right_click_uuid, current_text=''):
        """Perform an action on a tool.

        Args:
            tool (ToolItem): tool to perform the action on
            right_click_uuid (string): uuid of the right click
            current_text (string): current text
        """
        if tool.command is not None:
            if tool.args is not None and tool.args != ():
                return True, tool.command(*tool.args, **tool.kwargs)
            return True, tool.command()
        elif tool.var is not None and tool.item_uuid is not None:
            result = self.set_val(tool.item_uuid, current_text, add_undo=True)
            if result:
                return True, tool.var
            if self.app_data.profile_var and self.app_data.profile_var.uuid == tool.item_uuid:
                self.app_data.set_profile_val(current_text, self)
                return True, self.app_data.profile_var
            return False, None
        elif tool.item_uuid is not None:
            if 'rename' in tool.name.lower():
                result, calc = self.start_named_editing_item(tool.item_uuid)
            else:
                result, calc = self.edit_item(tool.item_uuid)
            if result:
                return True, calc
        elif tool.is_checked is not None:
            if tool.is_checked:
                self.uncheck_all_tools()
            else:
                self.uncheck_all_tools()
                tool.is_checked = True
            return True, tool.is_checked

        return False, None

    def uncheck_all_tools(self):
        """Uncheck all tools."""
        self.uncheck_all_tools_recursive(self.tool_manager.tools)

    def uncheck_all_tools_recursive(self, tool_dict):
        """Uncheck all tools recursively.

        Args:
            tool_dict (dict): dictionary of tools
        """
        for tool_name in tool_dict:
            tool = tool_dict[tool_name]
            if isinstance(tool, dict):
                self.uncheck_all_tools_recursive(tool)
            else:
                if tool.is_checked is not None:
                    tool.is_checked = False
                tool.is_checked = False

    def start_named_editing_item(self, item_uuid):
        """Edit the selected item.

        Args:
            item_uuid (string): The uuid of the selected tree item
        """
        self.set_all_items_not_editing()
        result, item = self.find_item_by_uuid(item_uuid)
        if not result:
            sel_items = self.projects_manager.get_selected_items()
            if len(sel_items) == 1:
                result = True
                item = sel_items[0]
        if result:
            if hasattr(item, 'is_editing'):
                item.is_editing = True
            if hasattr(item, 'tree_data'):
                item.tree_data.is_editing = True
            return True, item

    def edit_item(self, item_uuid):
        """Edit the selected item.

        Args:
            item_uuid (string): The uuid of the selected tree item
        """
        result, old_calc = self.find_item_by_uuid(item_uuid)
        if old_calc is None:
            return False, None
        icon = None
        if old_calc.class_name in self.calcdata_dict:
            icon = self.calcdata_dict[old_calc.class_name].icon
        if old_calc.icon is not None:
            icon = old_calc.icon
        result, calc = self.app_data.edit_item_functor(old_calc, model_name=self.model_name, model_base=self,
                                                       icon=icon)

        if result:
            self.set_calc(calc, old_calc, True)
            return result, calc

        return False, None

    # def get_setting_var(self, name, model=None):
    #     """Returns the variable of a setting with given name or displayed name.

    #     Args:
    #         name (string): the name or displayed name of the setting

    #     Returns:
    #         if the setting was successfully found
    #         var (FHWAVariable): Value of the setting
    #     """
    #     found, var = AppData.app_settings.get_setting_var(name, model=model)
    #     if found:
    #         return found, var

    #     return found, var

    # def set_setting_var(self, name, new_value, model=None):
    #     """Changes a setting with given name or displayed name.

    #     Args:
    #         name (string): the name or displayed name of the setting
    #         new_value (varies): new value to set
    #         model (string): model name

    #     Returns:
    #         if the setting was successfully found
    #         value (vaires): Value of the setting or return_in_case_of_failure
    #     """
    #     return AppData.app_settings.set_setting(name, new_value, model=model)

    def get_first_project_uuid(self):
        """Get the first project uuid."""
        # uuid_list = []
        # for project in self.projects_manager.projects_list:
        #     uuid_list.append(project.uuid)
        # if len(uuid_list) == 0:
        #     self.projects_manager.add_project()
        #     uuid_list.append(self.projects_manager.projects_list[0].uuid)
        # if self.projects_manager.first_project_uuid not in uuid_list:
        #     self.projects_manager.first_project_uuid = self.projects_manager.projects_list[0].uuid
        # TODO: Make changes so that we properly reference the model dict in other places.
        if self.model_name in self.app_data.model_dict:
            return self.app_data.model_dict[self.model_name].projects_manager.first_project_uuid
        return self.projects_manager.first_project_uuid

    def get_project_settings_by_uuid(self, project_uuid):
        """Get the project settings by uuid.

        Args:
            project_uuid (string): uuid of the project
        """
        return self.projects_manager.get_project_settings_by_uuid(project_uuid)

    def add_calcdata_by_name(self, calc_name):
        """Adds a calcdata to the project.

        Args:
            calc_name (string): name of the CalcData to add
        """
        app_data = AppData()
        projects_manager = self.projects_manager
        folder_location_uuid = projects_manager.handle_none_location_uuid(None)
        location_calc_list, project_uuid_list = projects_manager.get_items_with_uuids([folder_location_uuid])

        if self.calcdata_dict[calc_name].item_class:
            calcdata = self.calcdata_dict[calc_name].item_class(app_data=app_data, model_name=self.model_name,
                                                                project_uuid=project_uuid_list[0])
            calcdata.icon = self.calcdata_dict[calc_name].icon
            calcdata.tool_tip = self.calcdata_dict[calc_name].tool_tip
            self.projects_manager.add_calcdata(calcdata, location_calc_list[0])

            return calcdata
        return None

    def add_tree_folder(self, class_name='Folder'):
        """Adds a CalcData to the project.

        Args:
            class_name (string): class name of the Folder
            command_uuid (string): Unique identifier for the command
        """
        folder_location_uuid = None
        projects_manager = self.projects_manager
        selected_items = projects_manager.get_selected_items()
        if len(selected_items) == 1:
            if selected_items[0].class_name in ['Folder', 'Existing', 'Proposed']:
                folder_location_uuid = selected_items[0].uuid
            elif selected_items[0].class_name in ['Project']:
                folder_location = selected_items[0].get_coverage_folder()
                if folder_location is None:
                    folder_location = selected_items[0].get_model_folder()
                folder_location_uuid = folder_location.uuid
            else:
                return False, None
        else:
            return False, None
        folder = projects_manager.add_folder_by_uuid(folder_location_uuid, class_name)
        return True, folder

    def get_name(self):
        """Get the name of the model."""
        return self.model_name

    def get_theme(self):
        """Gets the selected theme."""
        return AppData.app_settings.get_theme()

    def _operate_command(self, command_item, new_value, prev_value):
        """Operate a given command (undo or redo).

        Args:
            command_item (CommandItem): the command to operate
            new_value: the value that we want to set as the new value
            prev_value: the value that we want to set as the previous value
        """
        item_uuid = command_item.uuid
        result, item = self.find_item_by_uuid(item_uuid)
        if result:
            if command_item.command == 'set_calc':
                self.set_calc(new_value, prev_value, add_undo=False)
            else:
                #  uuid, value, command='set_val', add_undo=True
                self.set_val(item_uuid, value=new_value, add_undo=False)

    def undo(self):
        """Undo - Perform an Undo for the last item in undo history."""
        self.command_manager.undo(self._operate_command)

    def redo(self):
        """Redo - Perform an Undo for the last item in redo history."""
        self.command_manager.redo(self._operate_command)

    def copy(self):
        """Copy - Perform a copy from the clipboard."""
        pass

    def cut(self):
        """Cut - Perform a cut from the clipboard."""
        pass

    def paste(self):
        """Paste - Perform an paste from the clipboard."""
        pass

    # End Commands (Undo/Redo)

    def get_all_selected_items(self):
        """Get the selected items.

        Returns:
            list: List of selected items
        """
        return self.projects_manager.get_selected_items()

    def get_items_with_uuids(self, item_uuids):
        """Get the items with the given uuids.

        Args:
            item_uuids (list): list of uuids

        Returns:
            item_list (list): list of items
        """
        item_list, _ = self.projects_manager.get_items_with_uuids(item_uuids)

        if len(item_list) == item_uuids:
            return True, item_list, None

        for i_uuid in item_uuids:
            if i_uuid not in item_list:
                result, item = AppData.app_settings.find_item_by_uuid(i_uuid)
                if result:
                    item_list.append(item)
                else:
                    result, item = self.project_settings.find_item_by_uuid(i_uuid)
                    if result:
                        item_list.append(item)

        if len(item_list) == item_uuids:
            return True, item_list, None

        uuids_not_found = []
        for item in item_list:
            if item.uuid not in item_uuids:
                uuids_not_found.append(item.uuid)

        return False, item_list, uuids_not_found

    def find_item_by_uuid(self, item_uuid):
        """Finds a variable by its uuid.

        Args:
            uuid (string): uuid of the variable

        Returns:
            result: True if item found
            variable: the variable
        """
        if isinstance(item_uuid, str):
            item_uuid = uuid.UUID(item_uuid)
        _, item_list, _ = self.get_items_with_uuids([item_uuid])
        if item_list and len(item_list) > 0:
            return True, item_list[0]

        return False, None

    def set_val(self, uuid, new_value, command='set_val', add_undo=True):
        """Sets the value of a variable.

        Args:
            uuid (string): uuid of the variable
            new_value (varies): value to set
            command (string): command to add to undo list
            add_undo (bool): add to undo list
        """
        result, item = self.find_item_by_uuid(uuid)
        if result:
            prev_value = item.get_val()
            if add_undo:
                undo_command = BatchCommandItem(command=command, batch_uuid=uuid, new_value=new_value,
                                                prev_value=prev_value, name='')
                self.command_manager.add_new_undo_command(undo_command)
                self.command_manager.clear_redo()
            if command == 'set_val':
                item.set_val(new_value)
            elif command == 'set_units':
                _, unit_system = self.settings.get_setting('Selected unit system')
                item.set_selected_unit(unit_system, new_value)
            return True
        return False

    def set_selected_items(self, selected_items_uuid_list):
        """Sets the is_selected item.

        Args:
            selected_items_uuid_list (list): List of uuids of selected items
        """
        self.set_all_items_not_editing()
        return self.projects_manager.set_selected_items(selected_items_uuid_list)

    def set_all_items_selected(self):
        """Sets the is_selected item."""
        self.set_all_items_not_editing()
        self.projects_manager.set_all_items_selected()

    def set_all_items_not_editing(self):
        """Get the selected items.

        Returns:
            list: List of selected items
        """
        return self.projects_manager.set_all_items_not_editing()

    def verify_layout(self):
        """Verify the layout."""
        # This needs to be run before the doc window view models are created
        result, layout = self.tool_manager.verify_layout(AppData.doc_windows_layout, len(AppData.doc_windows))
        if not result:
            AppData.doc_windows_layout = layout

    def update_tool_manager(self, undo_max_history=None):
        """Update the tool manager.

        Args:
            undo_max_history (int): Maximum number of undo items
        """
        # This is run every time the GUI is updated
        if undo_max_history is None:
            undo_max_history = AppData.app_settings.get_setting('Undo list show length')[1]
        self.tool_manager.update_tools(self.projects_manager, self.command_manager, self.file_manager, undo_max_history,
                                       self.open_file, self.app_data)

    # def get_menu_dict(self):
    #     """Returns the menu dictionary."""
    #     self.update_tool_manager()
    #     menu_dict = copy.copy(self.tool_manager.tools)

    #     for menu_group in menu_dict:
    #         items_to_remove = []
    #         for menu_item in menu_dict[menu_group]:
    #             if hasattr(menu_dict[menu_group][menu_item], 'menu_group'):
    #                 if menu_dict[menu_group][menu_item].menu_group is None:
    #                     items_to_remove.append(menu_item)
    #         for item in items_to_remove:
    #             menu_dict[menu_group].pop(item)

    #     return menu_dict

    def get_menu_dict(self, sys_complexity=None):
        """Create a filtered menu dictionary without modifying the original source.

        Args:
            sys_complexity (int, optional): The system complexity level. Defaults to None.
        """
        self.update_tool_manager()
        menu_dict = {}

        for menu_group, menu_items in self.tool_manager.tools.items():
            # Filter the menu items based on the condition and sys_complexity
            filtered_items = {
                menu_item: menu_items[menu_item]
                for menu_item in menu_items
                if not (hasattr(menu_items[menu_item], 'menu_group') and menu_items[menu_item].menu_group is None)
                and (
                    sys_complexity is None
                    or not hasattr(menu_items[menu_item], 'complexity')
                    or menu_items[menu_item].complexity <= sys_complexity
                )
            }

            # Only add the group if it has valid items
            if filtered_items:
                # Adjust the windows menu
                # (make it more user-friendly and only add options available for number of windows)
                if menu_group == 'Windows':
                    menu_dict[menu_group] = filtered_items['main'].copy()  # Create a copy of 'main'
                    if len(AppData.doc_windows) in filtered_items:
                        menu_dict[menu_group].update(filtered_items[len(AppData.doc_windows)])  # Update the copy
                else:
                    menu_dict[menu_group] = filtered_items

        return menu_dict

    def get_context_menu_dict(self, item_uuid):
        """Get the create tools.

        Returns:
            context_menu_dict (dict): A dict of tool items.
        """
        items, _ = self.projects_manager.get_items_with_uuids([item_uuid])
        if items is None or len(items) == 0:
            return False, {}
        item = items[0]
        context_menu_dict = {}
        folder_types = ['Project', 'Folder', 'Existing', 'Proposed']
        if item.class_name in ['Project']:
            context_menu_dict['Project Settings'] = self.tool_manager.tools['Project']['Edit project settings']
            context_menu_dict['Close project'] = self.tool_manager.tools['Project']['Close project']
        if item.class_name not in folder_types:
            context_menu_dict['Edit Calculator'] = self.tool_manager.tools['Calculators']['Edit Calculator']
        if item.class_name not in ['Folder', 'Existing', 'Proposed']:
            context_menu_dict['Edit metadata'] = self.tool_manager.tools['Calculators']['Edit metadata']
        if len(context_menu_dict) > 0:
            context_menu_dict['S1'] = ToolItem('Separator', is_separator=True)
        context_menu_dict['Rename'] = self.tool_manager.tools['Edit']['Rename']
        context_menu_dict['Delete'] = self.tool_manager.tools['Calculators']['Delete Calculator']
        context_menu_dict['Duplicate'] = self.tool_manager.tools['Calculators']['Duplicate Calculator']

        if item.class_name in ['Project', 'Folder', 'Existing', 'Proposed']:
            context_menu_dict['S1'] = ToolItem('Separator', is_separator=True)
            for calc in self.calcdata_dict:
                if calc in self.calcs_with_no_add_tool:
                    continue
                if f'New {calc}' in self.tool_manager.tools['Calculators']:
                    context_menu_dict[f'New {calc}'] = self.tool_manager.tools['Calculators'][f'New {calc}']

        return True, context_menu_dict

    def get_toolbar_dict(self, sys_complexity=None):
        """Returns the toolbar dictionary.

        Args:
            sys_complexity (int, optional): The system complexity level. Defaults to None.
        """
        self.update_tool_manager()
        toolbar_dict = {}
        toolbar_dict['Main Toolbar'] = self.get_main_toolbar_dict()
        toolbar_dict['Profiles Toolbar'] = self.get_profile_toolbar_dict()
        toolbar_dict['Units Toolbar'] = self.get_units_toolbar_dict()
        toolbar_dict['CalcData Toolbar'] = self.get_calculator_toolbar_dict(sys_complexity=sys_complexity)
        orientation_tb = self.get_orientation_toolbar_dict()
        if orientation_tb is not None:
            toolbar_dict['Orientation Toolbar'] = orientation_tb
        display_tb = self.get_display_toolbar_dict()
        if orientation_tb is not None:
            toolbar_dict['Display Toolbar'] = display_tb

        # Toolbars that should be vertical and along the data tree
        center_toolbar_names = ['Orientation Toolbar']

        return toolbar_dict, center_toolbar_names

    def get_main_toolbar_dict(self):
        """Returns the main toolbar dictionary."""
        tm = self.tool_manager
        toolbar_dict = {}

        toolbar_dict['New'] = tm.tools['File']['New']
        toolbar_dict['Add project'] = tm.tools['File']['Add project']
        # toolbar_dict['Close project'] = tm.tools['File']['Close project']
        toolbar_dict['Open'] = tm.tools['File']['Open']
        toolbar_dict['Save'] = tm.tools['File']['Save all']
        toolbar_dict['S1'] = ToolItem('Separator', is_separator=True)
        toolbar_dict['Settings'] = tm.tools['File']['Settings']
        # toolbar_dict['Project Settings'] = tm.tools['Edit']['Edit project settings']
        toolbar_dict['S2'] = ToolItem('Separator', is_separator=True)
        toolbar_dict['Undo'] = tm.tools['Edit']['Undo']
        toolbar_dict['Redo'] = tm.tools['Edit']['Redo']
        toolbar_dict['S3'] = ToolItem('Separator', is_separator=True)
        toolbar_dict['Support window'] = tm.tools['Edit']['Support window']
        toolbar_dict['Show wiki panel'] = tm.tools['Edit']['Show wiki panel']
        toolbar_dict['S4'] = ToolItem('Separator', is_separator=True)
        toolbar_dict['Wiki'] = tm.tools['Help']['Wiki']

        return toolbar_dict

    def get_calculator_toolbar_dict(self, sys_complexity=None):
        """Returns the CalcData toolbar dictionary.

        Args:
            sys_complexity (int, optional): The system complexity level. Defaults to None.
        """
        tm = self.tool_manager
        toolbar_dict = {}

        toolbar_dict['Calculators'] = copy.copy(tm.tools['Calculators'])

        tools_to_remove = []
        for calc_tool in list(toolbar_dict['Calculators']):
            tool = toolbar_dict['Calculators'][calc_tool]
            if (
                tool.add_toolbar is False
                or (
                    sys_complexity is not None
                    and hasattr(tool, 'complexity')
                    and tool.complexity > sys_complexity
                )
            ):
                tools_to_remove.append(calc_tool)

        for tool in tools_to_remove:
            toolbar_dict['Calculators'].pop(tool)

        return toolbar_dict

    def get_profile_toolbar_dict(self):
        """Returns the profile toolbar dictionary."""
        tm = self.tool_manager
        toolbar_dict = {}

        toolbar_dict['Profiles'] = tm.tools['Profiles']['Profiles']
        toolbar_dict['Edit'] = tm.tools['Profiles']['Edit']
        toolbar_dict['Delete'] = tm.tools['Profiles']['Delete']

        return toolbar_dict

    def get_units_toolbar_dict(self):
        """Returns the units toolbar dictionary."""
        tm = self.tool_manager
        toolbar_dict = {}

        toolbar_dict['Units'] = tm.tools['Units']['Units']

        return toolbar_dict

    def get_orientation_toolbar_dict(self):
        """Returns the orientation toolbar dictionary."""
        tm = self.tool_manager
        toolbar_dict = None
        if 'Orientation' in tm.tools:
            toolbar_dict = {}
            toolbar_dict['Orientation'] = tm.tools['Orientation']

        return toolbar_dict

    def get_display_toolbar_dict(self):
        """Returns the display toolbar dictionary."""
        tm = self.tool_manager
        toolbar_dict = None
        if 'Display' in tm.tools:
            toolbar_dict = {}
            toolbar_dict['Display'] = tm.tools['Display']

        return toolbar_dict

    def get_selected_gui_tool(self):
        """Get the selected tool.

        Returns:
            ToolItem: ToolItem object
        """
        return self.tool_manager.get_selected_gui_tool()

    def get_tool_by_uuid(self, uuid):
        """Get the tool by uuid.

        Args:
            uuid (string): Unique identifier for the tool

        Returns:
            ToolItem: ToolItem object
        """
        return self.tool_manager.get_tool_by_uuid(uuid)

    def get_projects_list(self):
        """Returns a list of the projects.

        Returns:
            list of TreeItems: projects list
        """
        return self.projects_manager.get_projects_list()

    def get_statusbar_dict(self):
        """Returns the statusbar dictionary."""
        return {}

    def toggle_show_panel(self, panel_name):
        """Toggle the specified panel."""
        if panel_name.lower() == 'wiki':
            self.toggle_show_wiki_panel()
            return True

        return False

    def toggle_show_wiki_panel(self):
        """Toggle the wiki panel."""
        _, show_wiki = self.get_setting('Show wiki panel')
        self.set_setting('Show wiki panel', not show_wiki)
        self.read_or_save_setting_and_profile_data(save=True)
        return True

    def new(self):
        """Start a new project."""
        self.projects_manager.new()
        self.command_manager = CommandManager()

    def add_project(self):
        """Start a new project."""
        self.projects_manager.add_project()

    def add_or_update_project(self, project):
        """Start a new project."""
        self.projects_manager.add_or_update_project(project=project)

    def close_project(self):
        """Close the current project."""
        self.projects_manager.close_project()

    def select_file_to_open(self, file_types, parent=None):
        """Open a project file.

        Args:
            parent (QWidget): parent of file dialog

        Returns:
            bool: True if the file was successfully opened
            list: list of file paths
        """
        select_file = SelectFile()

        select_file.parent = parent
        select_file.window_title = "Select file to open"
        select_file.starting_path = ""
        select_file.file_types = file_types
        select_file.file_mode = 'existing file'
        select_file.allow_multiple_files = True

        result, file_paths = self.app_data.select_file_functor(select_file)
        return result, file_paths

    def select_file_to_save(self, file_types, parent=None):
        """Open a project file.

        Args:
            parent (QWidget): parent of file dialog

        Returns:
            bool: True if the file was successfully opened
            list: list of file paths
        """
        select_file = SelectFile()

        select_file.parent = parent
        select_file.window_title = "Select file to open"
        select_file.starting_path = ""
        select_file.file_types = file_types
        select_file.file_mode = 'new file'
        select_file.allow_multiple_files = False

        result, file_paths = self.app_data.select_file_functor(select_file)
        return result, file_paths

    def open(self):
        """Open a project file.

        Args:
            file_path (string): path to the project file

        Returns:
            bool: True if the file was successfully opened
        """
        result, file_paths = self.select_file_to_open(self.model_file_types)

        result = False
        for file_path in file_paths:
            if self.open_file(file_path):
                result = True  # If any are True, we want to return True to update the interface
        return result

    def open_file(self, file_path):
        """Open a project file.

        Args:
            file_path (string): path to the project file

        Returns:
            bool: True if the file was successfully opened
        """
        if file_path and os.path.exists(file_path):
            _, max_recent_files = AppData.app_settings.get_setting('Recent files list length')
            self.file_manager.set_current_and_recent_file(file_path, max_recent_files)
            result, data, root, file_warning, unread_var = self.read_manager.read(
                filename=file_path, app_data=self.app_data, model_name=self.model_name)
            if file_warning != '':
                self.show_version_message_box(file_warning, unread_var)
            if result:
                # AppData.calculator_list.extend(calculators)
                # self.projects_manager.extend_calcdatas(calculators)

                project = self.projects_manager.add_or_update_project(root, data)
                self.set_current_and_recent_file(project.uuid, file_path, max_recent_files)
                if len(self.projects_manager.projects_list) == 2 and \
                        len(self.projects_manager.projects_list[0].children) == 0:
                    self.projects_manager.projects_list.pop(0)
                return True
        return False

    def show_version_message_box(self, message, unread_var):
        """Show a message box with the version warning and unread variables.

        Args:
            message (str): The message to display
            unread_var (list): The list of variables that were not read correctly
        """
        msg_box = MessageBox()
        msg_box.window_title = "Warning about selected file"
        # message = "Warning about selected file"
        # 'information', 'warning', 'critical', 'question'
        msg_box.message_type = 'warning'
        # 'ok', 'cancel', 'yes', 'no', 'abort', 'retry', 'ignore', 'close', 'help', 'apply', 'reset'
        msg_box.buttons = ['OK']

        if len(unread_var) > 0:
            message += '\nWould you like to see the variables that failed to read?'
            msg_box.buttons = ['yes', 'close']
        msg_box.message_text = message

        response = self.app_data.message_box_functor(msg_box)

        if response == 'yes':
            msg_box.window_title = 'Data that failed to read correctly'
            message = 'The following cards were not correctly read: '
            for unread_item in unread_var:
                message += unread_item
            msg_box.message_text = message
            msg_box.message_type = 'information'
            msg_box.buttons = ['OK']
            self.app_data.message_box_functor(msg_box)

    def set_current_and_recent_file(self, project_uuid, file_path, max_recent_files=None):
        """Set the current file and recent files.

        Args:
            project_uuid (string): uuid of the project
            file_path (string): path to the project file
            max_recent_files (int): maximum number of recent files
        """
        if not max_recent_files:
            _, max_recent_files = AppData.app_settings.get_setting('Recent files list length')
        self.file_manager.set_current_and_recent_file(file_path, max_recent_files)
        self.projects_manager.set_current_file_for_project(project_uuid, file_path)

    def tell_user_file_is_locked_dialog(self, file_path):
        """Tell the user that the file is locked.

        Args:
            file_path (string): path to the project file
        """
        msg_box = MessageBox()
        msg_box.window_title = "File is locked"
        msg_box.message_type = 'warning'
        msg_box.message_text = f'The file {file_path} is locked. Would you like to try again?'
        msg_box.buttons = ['yes', 'cancel']

        response = self.app_data.message_box_functor(msg_box)

        return response == 'cancel'

    def select_file_save_as(self, parent=None):
        """Open a project file.

        Args:
            file_path (string): path to the project file

        Returns:
            bool: True if the file was successfully opened
        """
        result, file_paths = self.select_file_to_save(self.model_file_types, parent=parent)

        # result = True
        if result:
            for file_path in file_paths:
                return True, file_path
        return False, None

    def save(self, parent, project):
        """Save a project with current filename.

        Args:
            parent (QWidget): parent of file dialog
            project (list): project data

        Returns:
            bool: True if the file was successfully saved
        """
        file_path = project.current_file
        if not file_path:
            return self.save_as(parent, project, file_path)
        return self._save(project, file_path)

    def save_all(self, parent=None):
        """Save all projects.

        Args:
            parent (QWidget): parent of file dialog

        Returns:
            bool: True if the file was successfully saved
        """
        projects_list = self.get_projects_list()
        result = True
        for project in projects_list:
            if not self.save(parent, project):
                result = False

        return result

    def _save(self, project, file_path):
        """Save a project with current filename.

        Args:
            project (list): project data
            file_path (string): path to the project file

        Returns:
            bool: True if the file was successfully saved
        """
        if file_path:
            # Check if the file exists, if so if locked, get user feedback on how to proceed
            file_unlocked = self.write_manager.check_for_file_lock(file_path, True)
            while os.path.exists(file_path) and not file_unlocked:
                cancel = self.tell_user_file_is_locked_dialog(file_path)
                if cancel:
                    return False
                file_unlocked = self.write_manager.check_for_file_lock(file_path, True)

            self.set_current_and_recent_file(project.uuid, file_path)
            self.write_manager.write_to_hdf5_file(project, file_path, True)
            return True
        return False

    def save_as_from_interface(self, parent=None):
        """Save a project as a new file.

        Args:
            parent (QWidget): parent of file dialog

        Returns:
            bool: True if the file was successfully saved
        """
        project = None
        result = True
        if len(self.projects_manager.get_projects_list()) == 1:
            project = self.projects_manager.get_projects_list()[0]
        else:
            sel_items = self.projects_manager.get_selected_items()
            if len(sel_items) != 1:
                return False
            if sel_items[0].class_name in ['Project']:
                project = sel_items[0]
            else:
                if sel_items[0].project_uuid is not None:
                    result, project = self.find_item_by_uuid(sel_items[0].project_uuid)
        if project is None:
            return False

        file_path = project.current_file
        if not self.save_as(parent, project, file_path):
            result = False

        return result

    def save_as(self, parent, project, file_path):
        """Save a project as a new file.

        Args:
            parent (QWidget): starting path
            project (list): project data
            file_path (string): path to the project file

        Returns:
            bool: True if the file was successfully saved
        """
        result, file_path = self.select_file_save_as(parent)
        if not result:
            return False

        return self._save(project, file_path)

    # def add_calcdata(self, command_uuid):
    #     """Adds a CalcData to the project.

    #     Args:
    #         command_uuid (string): Unique identifier for the command
    #     """
    #     folder_location_uuid = self.projects_manager.handle_none_location_uuid(None)
    #     location_calc_list, project_uuid_list = self.projects_manager.get_items_with_uuids([folder_location_uuid])
    #     if location_calc_list is None or len(location_calc_list) == 0:
    #         return False

    #     result, tool = self.tool_manager.get_tool_by_uuid(command_uuid)
    #     calculator_tree_item = self.calcdata_dict[tool.name]
    #     self.projects_manager.add_calcdata(calculator_tree_item.item_class(), location_calc_list[0])

    def add_calcdata_with_calc(self, calc):
        """Adds a CalcData to the project.

        Args:
            calc (CalcData): CalcData to add
        """
        folder_location_uuid = self.projects_manager.handle_none_location_uuid(None)
        location_calc_list, project_uuid_list = self.projects_manager.get_items_with_uuids([folder_location_uuid])
        if location_calc_list is None or len(location_calc_list) == 0:
            return False

        self.projects_manager.add_calcdata(calc, location_calc_list[0])
        return True

    def set_calc(self, new_calc, old_calc, add_undo=True):
        """Set the CalcData by uuid.

        Args:
            new_calc (CalcData): The CalcData that we want to keep values.
            old_calc (CalcData): The CalcData that we want to replace.
            add_undo (bool): Add the command to the undo list.

        Returns:
            True if successful.
        """
        if add_undo:
            calc_cmd = BatchCommandItem(command='set_calc', batch_uuid=new_calc.uuid, new_value=new_calc,
                                        prev_value=old_calc, name=f'Edit {new_calc.name}')
            undo_max_history = AppData.app_settings.get_setting('Undo list length')[1]
            self.command_manager.add_new_undo_command(calc_cmd, undo_max_history)
        result, _, is_setting = self.projects_manager.set_val(new_calc.uuid, new_calc)
        if is_setting:
            settings_file = AppData.app_settings.get_filename()
            file_unlocked = self.write_manager.check_for_file_lock(settings_file, True)
            if not os.path.exists(settings_file) or file_unlocked:
                self.write_manager.write_to_hdf5_file([AppData.app_settings], settings_file, True)

        return result

    def duplicate_selected(self, add_undo=True):
        """Duplicate the selected item(s).

        Args:
            add_undo (bool): Add the command to the undo list.
        """
        selected_items = self.projects_manager.get_selected_items()
        if len(selected_items) == 0:
            return False
        for item in selected_items:
            if item.class_name in ['Project',]:
                continue
            self.duplicate_item(item, add_undo=add_undo)
        return True

    def duplicate_item(self, item, add_undo=True):
        """Duplicate an item (CalcData or folder) by uuid.

        Args:
            calc_uuid (uuid): The uuid of the CalcData.
            add_undo (bool): Add the command to the undo list.

        Returns:
            True if successful.
        """
        item_copy = copy.deepcopy(item)
        self.assign_new_uuid_to_item_and_children_recursive(item_copy)
        item_copy.name = f'{item.name} (copy)'
        parent_uuid = None
        if hasattr(item, 'parent_uuid'):
            parent_uuid = item.parent_uuid
        if hasattr(item, 'tree_data'):
            parent_uuid = item.tree_data.parent_uuid
        if parent_uuid is None:
            return False
        result, parent = self.find_item_by_uuid(parent_uuid)
        if not result:
            return False
        children = None
        if hasattr(parent, 'children'):
            children = parent.children
        elif hasattr(parent, 'tree_data'):
            if hasattr(parent.tree_data, 'children'):
                children = parent.tree_data.children
        if children is None:
            return
        children.append(item_copy)

    def assign_new_uuid_to_item_and_children_recursive(self, item):
        """Assign a new uuid to an item and its children.

        Args:
            item (TreeItem): The item to assign the new uuid to.
        """
        item.uuid = uuid.uuid4()
        children = None
        if hasattr(item, 'children'):
            children = item.children
        elif hasattr(item, 'tree_data'):
            if hasattr(item.tree_data, 'children'):
                children = item.tree_data.children
        if children is None:
            return
        for child in children:
            self.assign_new_uuid_to_item_and_children_recursive(child)

    def delete_selected(self, add_undo=True):
        """Delete the selected item(s).

        Args:
            add_undo (bool): Add the command to the undo list.
        """
        selected_items = self.projects_manager.get_selected_items()
        if len(selected_items) == 0:
            return False
        for item in selected_items:
            self.delete_item(item.uuid, add_undo=add_undo)
        return True

    def delete_item(self, calc_uuid, add_undo=True):
        """Delete an item (CalcData or folder) by uuid.

        Args:
            calc_uuid (uuid): The uuid of the CalcData.
            add_undo (bool): Add the command to the undo list.

        Returns:
            True if successful.
        """
        if add_undo:
            pass
            # calc = self.projects_manager.get_item_by_uuid(calc_uuid)
            # calc_cmd = CommandItem(command='delete_item', uuid=calc_uuid, new_value=new_calc, prev_value=old_calc,
            #                        name=f'Deleted {new_calc.name}')
            # undo_max_history = AppData.app_settings.get_setting('Undo list length')[1]
            # self.command_manager.add_new_undo_command(calc_cmd, undo_max_history)
        self.projects_manager.delete_item(calc_uuid)

    def move_calc(self, source_uuids, target_uuid):
        """Move a CalcData.

        Args:
            source_uuids (string): source uuid
            target_uuid (string): target uuid
        """
        # TODO: Create an option that will copy instead of move
        # TODO: Handle changing projects
        # TODO: Handle multiple source_uuids;
        # if they are children, they should be moved by moving the parent (so those need to be removed from the list)
        # if they are not children, they should be moved individually
        source_uuid = source_uuids[0]
        self.projects_manager.move_calc(source_uuid, target_uuid)

    def set_item_name_by_uuid(self, name, calc_uuid, add_undo=True):
        """Set the rename selected item function.

        Args:
            name (string): The new name of the CalcData.
            calc_uuid (str): The uuid of the CalcData.
            add_undo (bool): Add the command to the undo list.
        """
        result, calc = self.find_item_by_uuid(calc_uuid)
        if not result:
            return False
        if add_undo:
            old_name = calc.name
            calc_cmd = BatchCommandItem(command='set_item_name_by_uuid', batch_uuid=calc_uuid, new_value=name,
                                        prev_value=old_name, name=f'Edited name to {name}')
            undo_max_history = AppData.app_settings.get_setting('Undo list length')[1]
            self.command_manager.add_new_undo_command(calc_cmd, undo_max_history)
        item_name = name
        if hasattr(calc, 'class_name'):
            if calc.class_name == 'Project':
                prefix = 'Project:'
                if item_name.startswith(prefix):
                    item_name = item_name[len(prefix):].lstrip()
        item_name.lstrip()
        item_name.rstrip()
        calc.name = name
        if hasattr(calc, 'input'):
            if 'name' in calc.input:
                calc.input['name'].set_val(name)
        calc.tree_data.is_editing = False

        return True

    def launch_reference_document(self, document):
        """Launch a reference document."""
        pass

    def launch_website(self, website):
        """Launch a website."""
        pass
