"""Classes to Define the model-view of a tool (toolbar or menu)."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from typing import Any, Callable, Dict, Optional, Tuple
from uuid import UUID, uuid4

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


@dataclass
class ToolItem:
    """Data class to hold commands that are executed. Used to keep an undo/redo history."""

    # Back-compat alias for constructor parameter; not part of the public state
    tool_uuid: Optional[UUID] = field(default=None, repr=False, compare=False)

    # Identity
    uuid: UUID = field(default_factory=uuid4)
    item_uuid: Optional[UUID] = None

    # Presentation and grouping
    name: str = ''
    is_separator: bool = False
    icon: Optional[str] = None
    tool_tip: Optional[str] = None
    status_message: Optional[str] = None
    shortcut: Optional[str] = None
    menu_group: str = ''
    add_toolbar: bool = True

    # Command wiring
    command: Optional[Callable[..., Any]] = None
    args: Tuple[Any, ...] = field(default_factory=tuple)
    kwargs: Dict[str, Any] = field(default_factory=dict)

    # State and behavior
    var: Any = None
    is_enabled: bool = True
    is_checked: Optional[bool] = None
    exit: bool = False
    complexity: int = 0

    def __post_init__(self):
        """Finalize defaults and support legacy constructor keyword `tool_uuid`."""
        # Support passing `tool_uuid=` like the previous API
        if self.tool_uuid is not None:
            self.uuid = self.tool_uuid
            self.tool_uuid = None

        # Default status_message to tool_tip if not explicitly set
        if self.status_message is None and self.tool_tip is not None:
            self.status_message = self.tool_tip

    def set_args_and_kwargs(self, *args, **kwargs):
        """Set the args and kwargs for the command.

        Args:
            args (list): List of arguments
            kwargs (dict): Dictionary of keyword arguments
        """
        self.args = args
        self.kwargs = kwargs
