"""Variable dataclass."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
# from pathlib import Path
# import re
import sys
from typing import Any, List, Optional, Tuple  # , Union
from uuid import UUID, uuid4

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


@dataclass
class VariableData:
    """Data class for Variable attributes."""
    # Class-level constants (not instance fields)
    string_types = ['string', 'zip', 'phone', 'email', 'url']
    list_types = ['list', 'float_list', 'int_list', 'string_list', 'bool_list', 'calc_list', 'list_of_lists']
    supported_color_formats = ['rgb']

    # == Core metadata ==
    # Variable Name
    name: str = ''
    # Type: 'bool', 'int', 'float', 'string', 'list', 'float_list', 'int_list', 'string_list', 'bool_list',
    # 'class', 'file', 'image', 'color', 'calc', 'calc_list', 'uuid_dict', 'UserArray', 'table', 'date'
    type: str = ''

    # == Values ==
    # bool if type is bool, int if type is int or list (list size), float if type = float
    value: Any = 0.0
    # Sometimes necessary, for example, when complexity hides variable in UI
    default_value: Any = 0.0
    # If the type is list, this holds the list; if a file, this holds the filters
    # example of file filter: ["Hydraulic Toolbox Files (*.htb)", "All Files (*)"]
    value_options: List[Any] = field(default_factory=list)

    # == Limits and precision ==
    # tuple to give lower and upper limits
    limits: Tuple[float, float] = (0.0, sys.float_info.max)
    # number of significant digits for this variable
    precision: int = 2

    # == Units ==
    # unit type can hold length, speed, time, percent, etc
    unit_type: List = field(default_factory=list)
    # This is the unit that we store/use/compute with internally
    native_unit: str = ""
    # This is the unit that we display the data as to the user in US or both units
    selected_us_unit: str = ""
    # This is the unit that we display the data as to the user in SI
    selected_si_unit: str = ""
    # Units available to the user to select from
    available_us_units: List[List] = field(default_factory=lambda: [[]])
    available_si_units: List[List] = field(default_factory=lambda: [[]])

    # == Help, Notes, and complexity ==
    # A note to give the user additional information about the variable
    note: str = ""
    # Complexity level that displays/hides the variable
    complexity: int = 0
    # URL to help documentation for the variable
    help_url: Optional[str] = None

    # == UI flags and action ==
    # Lock the variable into read-only mode
    read_only: bool = False
    # File mode for file variables: 'folder', 'existing file', 'new file', 'any'
    file_mode: str = 'existing file'
    # Action (used for push button; functor)
    action: Any = None
    # If set to true, it will set the first row as a header or subheader (depending on recurrence_level)
    set_row_as_header: bool = False
    # This will change the color of the row
    # 'approved' = green, 'normal' = (based on row/cell type), 'incomplete' = yellow, 'warning' = orange,
    # 'failure' = red
    data_status: str = 'normal'

    # == Identity ==
    # UUID for the variable, used for undo/redo commands
    uuid: UUID = field(default_factory=uuid4)

    # Nice idea for data validation, but it is causing some errors in hydraulic toolbox -
    # come back to this after those are sorted out
    # def __post_init__(self):
    #     """Post-initialization to finalize variable setup."""
    #     # Make sure we have basics set:
    #     # if self.name == '':
    #     #     self.name = f'Variable {self.uuid}'
    #     # if self.type == '':
    #     #     self.type = 'float'
    #     # elif self.type == 'bool' and not isinstance(self.value, bool):
    #     #     self.value = False
    #     # elif self.type in ['calc_list', 'class']:
    #     #     pass
    #     # elif (self.type in ['int'] or self.type in self.list_types) and not isinstance(self.value, int):
    #     #     self.value = 0
    #     # elif self.type == 'float' and not isinstance(self.value, float):
    #     #     self.value = 0.0
    #     # elif self.type in self.string_types and not isinstance(self.value, str):
    #     #     self.value = ''
    #     # elif self.type in self.list_types and not isinstance(self.value_options, list):
    #     #     self.value_options = []

    #     # Default selected units
    #     if not self.selected_us_unit and self.native_unit:
    #         self.selected_us_unit = self.native_unit

    #     # Compute selected SI unit from native unit and self.native_unit
    #     # _, self.selected_si_unit = ConversionCalc(None).get_si_complementary_unit(self.native_unit)

    #     # Handle default_value mirroring initial value
    #     if self.default_value is None:
    #         self.default_value = self.value

    #     # Ensure value carries name if it's a custom object with a 'name' attribute and not a Path
    #     if not isinstance(self.value, Path) and hasattr(self.value, 'name'):
    #         try:
    #             self.value.name = self.name
    #         except Exception:
    #             pass

    #     # Normalize value_options for list types
    #     if self.type in self.list_types:
    #         if isinstance(self.value_options, list):
    #             if len(self.value_options) > 0:
    #                 self.default_value = self.value_options[0]
    #             else:
    #                 # Keep a numeric default for empty options
    #                 self.default_value = 0.0
    #         else:
    #             # Coerce single option to list
    #             self.value_options = [self.value_options]
