"""SpecifyBoundaryData Class (used in GVF to specify the boundary at both ends."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.GVF.specify_boundary_calc import SpecifyBoundaryCalc
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData


class SpecifyBoundaryData(VariableGroup):
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """

    def __init__(self, boundary_name='upstream', allow_normal_depth=True, allow_channel=True, app_data=None,
                 model_name=None, project_uuid=None):
        """Initializes the Specify Boundary Class.

        Args:
            boundary_name (string): Used in the select to inform the user which end the class is specifying boundary
            allow_normal_depth (bool): Allows the user to select normal depth (not applicable for upstream end
            allow_channel (bool): Allows the user to select channel calculations
            app_data (AppData): application data (settings, etc.).
            model_name (string): name of the model
            project_uuid (uuid): unique identifier for the project
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Specify Boundary'
        self.type = 'SpecifyBoundaryData'

        self.calculator = SpecifyBoundaryCalc()

        # Input
        self.boundary_name = boundary_name
        specify_name = f'Specify {boundary_name} as '
        boundary_list = []
        if allow_normal_depth:
            boundary_list.append('normal depth')
        boundary_list.extend([
            'critical depth', 'constant depth', 'rating curve',
            'specified depths'
        ])
        if allow_channel:
            boundary_list.append('channel calculations')
        self.input['Select'] = Variable(specify_name, "list", 0, boundary_list)
        self.input['Constant depth'] = Variable(
            'Constant depth', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        # Rating Curve
        # self.input['rating curve'] = Variable(
        #     'rating curve', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length)
        wse_color = (70, 118, 202)
        wse_fill_color = (0, 0, 255)

        velocity_color = (255, 140, 0)
        velocity_fill_color = (255, 200, 120)

        self.theme = self.get_theme()
        if self.theme:
            wse_color = self.theme['Plot WSP color']
            wse_fill_color = self.theme['Plot WSP fill color']

            velocity_color = self.theme['Plot velocity color']
            velocity_fill_color = self.theme['Plot velocity fill color']

        rating_curve = {
            'Flow': Variable('Flow', 'float_list', 0, [0.0], precision=2, unit_type=['flow'],
                             native_unit='cfs', us_units=self.us_flow, si_units=self.si_flow),
            'Elevation': Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                  limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                  native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'Velocity': Variable('Velocity', 'float_list', 0, [0.0], precision=2, unit_type=['velocity'],
                                 native_unit='ft/s', us_units=self.us_velocity, si_units=self.si_velocity)
        }
        name = 'Rating curve'
        self.input[name] = Variable(
            name, 'table', TableData(self.theme, name=name, plot_names=['Rating curve'],
                                     input=copy.deepcopy(rating_curve), min_items=3,
                                     app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input[name].get_val().set_plot_series_options(
            name, related_index=0, index=0, name=name, x_axis='Station', y_axis='Elevation',
            line_color=wse_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=wse_fill_color)
        self.input[name].value
        rating_curve_options = self.input[name].value.input['Plot options']['Rating curve'].value.input[
            'Data series'].value.item_list[0].input['Plot series'].value.item_list[0]
        rating_curve_options.input['YY axis'].set_val('Velocity')
        rating_curve_options.input['Line color'].set_val(velocity_color)
        rating_curve_options.input['Fill color'].set_val(velocity_fill_color)
        rating_curve_options.input['Line type'].set_val('dashed')
        # rating_curve_options.input['XX axis'].value = None

        self.input['Specified depths'] = Variable(
            'Specified depths', 'UserArray', UserArray(precision=2, unit_type=['length'], native_unit='ft',
                                                       us_units=[['yd', 'ft', 'in']], si_units=self.si_mid_length))
        self.input['Specified velocities'] = Variable(
            'Specified velocities', 'UserArray', UserArray(precision=2, unit_type=['length'], native_unit='ft/s',
                                                           us_units=[['mph', 'yd/s', 'ft/s', 'in/s']],
                                                           si_units=[['kph', 'm/s', 'mm/s']]))

        manning_n_data = ManningNData(app_data=app_data, model_name=model_name, project_uuid=project_uuid)
        manning_n_data.input['Calculate'].set_val('Head')
        manning_n_data.prepare_for_compute()
        self.manning_n_calc = manning_n_data.calculator
        self.manning_n_calc.input_dict, self.manning_n_calc.plot_dict = manning_n_data.prepare_input_dict()

        # Intermediate calculations
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'allow_normal_depth', 'allow_channel', 'manning_n_calc', 'boundary_name', 'specify_name'
        ])

        self.allow_normal_depth = allow_normal_depth
        self.allow_channel = allow_channel

        self.warnings = []
        self.results = {}

    def get_starting_water_depth_and_velocity(self, normal_depth, normal_velocity, critical_depth, critical_velocity,
                                              flow, index=0):
        """Returns the starting water depth and velocity for the given data.

        Args:
            normal_depth (float): computed normal depth for the channel
            normal_velocity (float): computed average velocity at normal depth
            critical_depth (float): computed critical depth for the flow and geometry
            critical_velocity (float): computed average velocity at critical depth
            flow (float): flowrate of the channel for current computations
            index (int): index of the result that is applicable to this computation
        """
        self.compute_data()
        select = self.input['Select'].get_val()
        return self.calculator.get_starting_water_depth_and_velocity(
            normal_depth, normal_velocity, critical_depth, critical_velocity, flow, index, select)

    def _get_can_compute(self):
        """Determine whether we have enough data for a valid computation."""
        if self.input['Select'].get_val() in ['normal depth', 'critical depth']:
            return True
        elif self.input['Select'].get_val() == 'constant depth':
            if self.input['Constant depth'].get_val() >= 0.0:
                return True
        elif self.input['Select'].get_val() == 'rating curve':
            m = f'Rating curve option is not yet implemented; please select a different condition {self.boundary_name}'
            self.warnings['rating_curve'] = m
            return False
        elif self.input['Select'].get_val() == 'specified depths':
            depths = self.input['Specified depths'].get_val().get_result()
            if len(depths) == 0:
                self.warnings['depth'] = f'Please enter a specified depth {self.boundary_name}'
                return False
            for depth in depths:
                if depth < 0.0:
                    self.warnings['depth'] = \
                        f'Please enter a positive value for all specified depths {self.boundary_name}'
                    return False
            return True
        return False

    def set_specified_depths_and_velocities(self, depths, velocities):
        """Set the specified depths and velocities.

        Args:
            depths (list of depths): water depths
            velocities (list of velocities): water velocities
        """
        if type(depths) is not list:
            depths = [depths]
        self.input['Select'].set_val('specified depths')
        self.input['Specified depths'].get_val().set_user_list(depths)
        self.input['Specified velocities'].get_val().set_user_list(velocities)

    def get_specified_depths_and_velocities(self):
        """Get the specified depths and velocities.

        Returns:
            depths (list of depths): water depths
            velocities (list of velocities): water velocities
        """
        return self.input['Specified depths'].get_val().get_result(), \
            self.input['Specified velocities'].get_val().get_result()

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}
        input_vars['Select'] = self.input['Select']

        if self.input['Select'].get_val() in ['normal depth', 'critical depth']:
            pass  # Nothing else needed
        elif self.input['Select'].get_val() == 'constant depth':
            input_vars['Constant depth'] = self.input['Constant depth']
        elif self.input['Select'].get_val() == 'rating curve':
            input_vars['Rating curve'] = self.input['Rating curve']
        elif self.input['Select'].get_val() == 'specified depths':
            input_vars['Specified depths'] = self.input['Specified depths']
        return input_vars
