"""CalcData for performing Contraction Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_calc import ScourBaseCalc
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class ContractionScourCalc(ScourBaseCalc):
    """A class that defines a contraction scour at a bridge contraction."""

    # Data to interpolate from figure 6.8 in HEC 18 April 2012, page 6.11 (pdf page 147)
    t0_w = [
        0.001, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5
    ]
    t0_d = [
        0.04, 0.056, 0.069, 0.079, 0.09, 0.1, 0.109, 0.118, 0.126, 0.135, 0.21,
        0.29, 0.365, 0.45, 0.535, 0.62, 0.7, 0.78, 0.88, 1.45, 2.17, 4.3, 7.6,
        14
    ]

    t20_w = [
        0.001, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5
    ]
    t20_d = [
        0.03, 0.043, 0.053, 0.062, 0.069, 0.077, 0.083, 0.092, 0.099, 0.106,
        0.169, 0.228, 0.29, 0.36, 0.43, 0.495, 0.57, 0.66, 0.75, 1.27, 2.0,
        4.0, 7.4, 14
    ]

    t40_w = [
        0.001, 0.002, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01,
        0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1, 0.15, 0.2, 0.3,
        0.4, 0.5
    ]
    t40_d = [
        0.025, 0.035, 0.043, 0.05, 0.056, 0.063, 0.069, 0.076, 0.08, 0.088,
        0.14, 0.19, 0.25, 0.31, 0.37, 0.43, 0.51, 0.59, 0.67, 1.2, 1.85, 3.85,
        7.15, 14
    ]

    def __init__(self, clear_my_own_results=True, null_data=None, zero_tol=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): If True, clear the results before computing
            null_data (float): Null data value
            zero_tol (float): Zero tolerance for the data
        """
        super().__init__()

        self.clear_my_own_results = clear_my_own_results

        self.t0_interpolate = Interpolation(ContractionScourCalc.t0_d, ContractionScourCalc.t0_w,
                                            null_data=null_data, zero_tol=zero_tol)
        self.t20_interpolate = Interpolation(ContractionScourCalc.t20_d, ContractionScourCalc.t20_w,
                                             null_data=null_data, zero_tol=zero_tol)
        self.t40_interpolate = Interpolation(ContractionScourCalc.t40_d, ContractionScourCalc.t40_w,
                                             null_data=null_data, zero_tol=zero_tol)

        self.compute_pressure = False

        self.unit_converter = ConversionCalc()

        self.is_upstream_transporting_sediment = False

    def get_can_compute_transporting_sediment(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        _, zero_tol = self.get_data('Zero tolerance', 1e-6)

        result = self.check_float_vars_to_greater_zero(['Average depth upstream (y1)',
                                                        'Average velocity upstream (v1)'])

        if self.d50 < zero_tol:
            self.warnings['Enter D50'] = "Please enter a D50"
            result = False

        self.can_compute_transporting_sediment = result

        return result

    def _get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        if self.gradations:
            self.update_gradation_lists()
        else:
            _, self.d50 = self.get_data('Contracted D50')
            _, self.upstream_d50 = self.get_data('Approach D50')
            _, self.critical_shear_stress = self.get_data('Critical shear stress (τc)')

        if 'Results' not in self.results:
            self.results['Results'] = {}

        self.results['Results']['Approach D50'] = self.upstream_d50
        self.results['Results']['Contracted D50'] = self.d50
        self.results['Results']['Critical shear stress (τc)'] = self.critical_shear_stress

        # Check if it is cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            result = self.check_float_vars_to_greater_zero(
                ['Average depth upstream (y1)', 'Average velocity in contracted section (v2)',
                 "Manning's n (n)"])

        else:
            if not self.get_can_compute_transporting_sediment():
                return False

            self.determine_if_upstream_is_transporting_sediment()

            _, self.compute_pressure = self.get_data('Compute pressure flow')
            self.compute_pressure_flow_prerequisites()

            if self.is_upstream_transporting_sediment:  # Live Bed
                if not self.get_can_compute_live_bed():
                    return False
                if self.applied_shear >= self.critical_shear:
                    pass
                else:
                    # shear is not sufficient
                    # Take the smaller of the depths
                    if not self.get_can_compute_clear_water():
                        return False

            else:  # Clear Water
                if not self.get_can_compute_clear_water():
                    return False

            if self.compute_pressure:
                pressure_result = self.get_can_compute_pressure()
                if not pressure_result:
                    result = False

        return result

    def get_can_compute_live_bed(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        # if unknown is None:
        #     unknown = self.unknown
        result = True

        result, zero_tol = self.get_data('Zero tolerance', 1e-6)

        result = self.check_float_vars_to_greater_zero(['Discharge in approach section (Q1)',
                                                        'Bottom width in approach section (W1)',
                                                        'Bottom width in contracted section (W2)',
                                                        'Depth prior to scour in contracted section (y0)'])

        if self.que < zero_tol:
            self.warnings['Contracted section discharge'] = "Please enter the discharge at the contracted section"
            result = False

        self.can_compute_live_bed = result

        return result

    def get_can_compute_clear_water(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        result, zero_tol = self.get_data('Zero tolerance')

        result = self.check_float_vars_to_greater_zero(['Discharge in contracted section (Q2)',
                                                        'Bottom width in contracted section (W2)',
                                                        'Depth prior to scour in contracted section (y0)'])

        if self.d50 < zero_tol:
            self.warnings['Contracted section D50'] = "Please enter the D50 in the contracted section"
            result = False

        self.can_compute_clear_water = result

        return result

    def get_can_compute_pressure(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        result = self.check_float_vars_to_greater_zero(['Depth prior to scour at bridge (hb + ht or hu)',
                                                        'Vertical size of bridge opening (hb)',
                                                        'Deck thickness (t)'])

        self.can_compute_pressure = result

        return result

    def determine_if_upstream_is_transporting_sediment(self):
        """Determines if the sediment is being transported upstream at the approach section."""
        if self.upstream_d50 <= 0.0 and self.d50 > 0.0:
            self.warnings['Approach D50'] = \
                'The approach D50 is not defined. The surface contracted D50 is used as the approach D50.'
            self.upstream_d50 = self.d50
            if 'Results' not in self.results:
                self.results['Results'] = {}
            self.results['Results']['Approach D50'] = self.upstream_d50

        ku = 11.17
        if self.is_computing_shear_decay:
            thalweg = self.centerline_streambed

            # Get starting soil layer
            self.layer_index, _, _, _ = self.get_layer_index_from_elevation(thalweg)
            ks = self.bh_layers[self.layer_index]['calculator'].results['Shear decay parameters']['Shields number (ks)']

            _, gamma_w = self.get_data('Unit weight of water (γw)')
            gamma_s = self.bh_layers[self.layer_index]['Unit weight of soil (γs)']

            s = gamma_s / gamma_w
            ku = (ks ** (1.0 / 2.0) * (s - 1.0) ** (1.0 / 2.0)) / 0.034
        self.results['Unit conversion constant (Ku)'] = ku

        _, y1 = self.get_data('Average depth upstream (y1)')
        self.critical_velocity = ku * y1 ** (1.0 / 6.0) * self.upstream_d50 ** (1.0 / 3.0)

        _, v1 = self.get_data('Average velocity upstream (v1)')
        if self.critical_velocity > v1:
            self.is_upstream_transporting_sediment = False
        else:
            self.is_upstream_transporting_sediment = True

        self.results['Critical velocity (vc)'] = self.critical_velocity
        self.results['Upstream transporting sediment'] = self.is_upstream_transporting_sediment

        return self.is_upstream_transporting_sediment

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        # Document the source of the equation (if possible):
        #   Give publication, date, page number and link (if there's a bug, that it can be traced back to the source).
        # Assign new variables from the input_dict in similar names as the equation and document units

        # HEC-18, 6th Edition, 2024, Assorted pages; see chapter 8 for details
        self.scour_depth = 0.0

        # Check if the soil is cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            self.horizontal_scour_depth = self.compute_cohesive_scour_depth()
            self.compute_partial_scour_depth()
            self.results['Governing scour method'] = 'Cohesive'
            self.is_upstream_transporting_sediment = False

        else:  # Noncohessive soil
            # Determine if upstream is transporting sediment
            self.determine_if_upstream_is_transporting_sediment()

            self.results['Pressure scour'] = {}
            self.compute_pressure_flow_prerequisites()

            # Compute Live bed and clear water scour depths
            if self.get_can_compute_live_bed():
                self.compute_live_bed_scour_depth()
            if self.get_can_compute_clear_water():
                self.compute_clear_water_scour_depth()

            # Determine whether live bed or clear water governs
            if self.is_upstream_transporting_sediment:
                # Live Bed Check
                if self.applied_shear >= self.critical_shear:
                    # Live Bed
                    self.horizontal_scour_depth = self.live_bed_scour_depth
                    self.results['Governing scour method'] = 'Live bed'
                else:
                    # shear is not sufficient
                    # Take the smaller of the depths
                    self.warnings['scour depth limited by critical shear'] = \
                        'Scour depth is limited by critical shear for D50. Smaller contraction scour estimate is used.'

                    if self.clear_water_scour_depth <= self.live_bed_scour_depth:
                        # Clear Water
                        self.horizontal_scour_depth = self.clear_water_scour_depth
                        self.results['Governing scour method'] = 'Clear water'
                    else:
                        # Live Bed
                        self.horizontal_scour_depth = self.live_bed_scour_depth
                        self.results['Governing scour method'] = 'Live bed'
            else:
                # Clear Water
                self.horizontal_scour_depth = self.clear_water_scour_depth
                self.results['Governing scour method'] = 'Clear water'

        self.results['Scour depth (ys)'] = self.horizontal_scour_depth

        _, compute_pressure = self.get_data('Compute pressure flow')
        if compute_pressure:
            if self.get_can_compute_pressure():
                self.vertical_scour_depth = self.compute_pressure_flow()

                if self.horizontal_scour_depth + self.longterm_scour_depth > self.vertical_scour_depth:
                    self.scour_depth = self.horizontal_scour_depth
                else:
                    self.scour_depth = self.vertical_scour_depth
                    self.results['Governing scour method'] = 'Pressure flow'

                self.results['Scour depth (ys)'] = self.scour_depth
            else:
                self.vertical_scour_depth = 0.0
                self.scour_depth = self.horizontal_scour_depth
        else:
            self.vertical_scour_depth = 0.0
            self.scour_depth = self.horizontal_scour_depth

        if self.is_computing_shear_decay:
            self.compute_shear_decay()
            self.results['Scour depth (ys)'] = self.scour_depth

        self.compute_scour_hole_geometry()

        return True

    def compute_cohesive_scour_depth(self):
        """Compute the scour depth for cohesive soils."""
        _, g = self.get_data('Gravity', 32.2)
        _, ku = self.get_data('Manning constant', 1.486)
        _, y1 = self.get_data('Average depth upstream (y1)')
        _, v2 = self.get_data('Average velocity in contracted section (v2)')
        _, rho_w = self.get_data('Water density (ρw)')
        _, n = self.get_data("Manning's n (n)")
        tc = self.critical_shear_stress

        _, gamma_w = self.get_data('Unit weight of water (γw)')
        _, y0 = self.get_data('Depth prior to scour in contracted section (y0)')

        # Equation 6.6 on page 6.16 of HEC-18
        if rho_w == 0.0 or n == 0.0 or y1 == 0.0:
            self.cohesive_scour_depth = 0.0
        else:
            self.cohesive_scour_depth = 0.94 * y1 * ((1.83 * v2 / math.sqrt(g * y1)) - (ku * (math.sqrt(tc / rho_w)) / (
                g * n * y1 ** (1.0 / 3.0))))

        self.scour_depth = self.cohesive_scour_depth
        self.results['Cohesive scour depth (ys)'] = self.scour_depth
        self.results['Scour depth (ys)'] = self.scour_depth

        self.initialStress = gamma_w * pow(v2 * n / ku, 2.0) * pow(y0, -1.0 / 3.0)
        self.results['Initial shear stress (τ)'] = self.initialStress

        return self.cohesive_scour_depth

    def compute_partial_scour_depth(self):
        """Compute the partial scour depth for cohesive soils."""
        _, t = self.get_data('Duration of flow event (t)', prior_keys=['Scour from single event'])
        _, initial_rate = self.get_data('Initial rate of scour', prior_keys=['Scour from single event'])

        if t > 0.0 and self.cohesive_scour_depth > 0.0:
            self.partial_scour_depth = t / (1 / initial_rate + t / self.cohesive_scour_depth)
            self.results['Partial scour depth'] = self.partial_scour_depth
        else:
            self.partial_scour_depth = 0.0

    def compute_shear_decay(self):
        """Compute the shear decay for contraction scour."""
        if self.gradations is None:
            self.warnings['Gradations not defined'] = 'Gradations must be defined to compute shear decay.'
            return
        if 'Shear decay results' not in self.results:
            self.results['Shear decay results'] = {}
        self.results['Shear decay results']['Scour depth without shear decay (ys)'] = self.scour_depth

        _, q2 = self.get_data('Discharge in contracted section (Q2)')
        _, w2 = self.get_data('Bottom width in contracted section (W2)')
        self.q2c = 0.0
        if w2 > 0.0:
            self.q2c = q2 / w2
        self.results['Shear decay results']['Unit discharge'] = self.q2c
        decay_x = []  # Shear
        decay_y = []  # Elevation
        marker_x = []
        marker_y = []

        thalweg = self.centerline_streambed

        is_live = self.results['Governing scour method'] == 'Live bed'

        # bh_uuid = self.get_data('Selected borehole')
        # bh = self.bh_dict[bh_uuid]
        # self.bh_layers = bh.get_data('Layers').item_list

        # TODO Verify that this gives wse when LTD is applied
        _, y0 = self.get_data('Depth prior to scour in contracted section (y0)')
        wse = thalweg + y0

        # Get starting soil layer
        self.layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear = self.get_layer_index_from_elevation(
            thalweg)
        # mannings_n = self.bh_layers[self.layer_index].results['Shear decay parameters']['Manning n']
        mannings_n = self.bh_layers[self.layer_index]['calculator'].results['Shear decay parameters']['Manning n']
        lower_layer_n, lower_layer_shear, lower_layer_critical = self.get_next_layer_details(
            self.layer_index, wse, self.q2c)
        if len(self.bh_layers) == 1:
            cur_layer_top = thalweg
            cur_layer_bottom = -sys.float_info.max

        scour_elevation = thalweg - self.scour_depth

        cur_shear = sys.float_info.max
        cur_decay_y = thalweg
        _, decay_inc = self.get_data('Shear decay curve increment', 0.5)

        _, min_thickness = self.get_data('Min layer thickness', 2.0)

        # Check if current layer is getting too thin to trust
        cur_thickness = cur_decay_y - cur_layer_bottom
        lower_layer_weaker = False
        if lower_layer_n is not None:
            lower_layer_weaker = lower_layer_critical < cur_critical_shear and cur_thickness < min_thickness and \
                lower_layer_shear > lower_layer_critical

        layers_too_thin = []

        cur_shear = self.compute_shear(wse, cur_decay_y, mannings_n, self.q2c)

        # Determine shear stress until criteria is met (converge on critical shear, reach scour depth (for live), or
        # layer is too thin)
        while cur_shear >= cur_critical_shear and (not is_live or (is_live and scour_elevation < cur_decay_y)) or \
                (cur_shear <= cur_critical_shear and lower_layer_weaker):
            if not cur_shear >= cur_critical_shear and (not is_live or (is_live and scour_elevation < cur_decay_y)) \
                    and self.layer_index not in layers_too_thin:
                layers_too_thin.append(self.layer_index)
                self.warnings[f'Layer {str(self.layer_index)}'] = (f'Layer {str(self.layer_index)} is too thin to '
                                                                   'trust; continuing the shear decay curve.')
            if not cur_layer_top >= cur_decay_y > cur_layer_bottom:  # Handle layer change
                # Determine the shear stress at the bottom of the layer
                cur_shear = self.compute_shear(wse, cur_layer_bottom, mannings_n, self.q2c)
                decay_x.append(cur_shear)
                decay_y.append(cur_layer_bottom)
                marker_x.append(cur_shear)
                marker_y.append(cur_layer_bottom)
                # Update data from the layers
                self.layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear = \
                    self.get_layer_index_from_elevation(cur_decay_y)
                # mannings_n = self.bh_layers[self.layer_index].results['Shear decay parameters']['Manning n')
                mannings_n = self.bh_layers[self.layer_index]['calculator'].manning_n
                lower_layer_n, lower_layer_shear, lower_layer_critical = self.get_next_layer_details(
                    self.layer_index, wse, self.q2c)
                # Determine the shear stress at the top of the new layer
                cur_shear = self.compute_shear(wse, cur_layer_top, mannings_n, self.q2c)
                decay_x.append(cur_shear)
                decay_y.append(cur_layer_top)
                marker_x.append(cur_shear)
                marker_y.append(cur_layer_top)

            cur_shear = self.compute_shear(wse, cur_decay_y, mannings_n, self.q2c)
            decay_x.append(cur_shear)
            decay_y.append(cur_decay_y)
            # If we overshot, correct it:
            if cur_shear < cur_critical_shear:
                _, null_data = self.get_data('Null data')
                _, zero_tol = self.get_data('Zero tolerance')
                shear_interp = Interpolation(list(reversed(decay_x)), list(reversed(decay_y)), null_data=null_data,
                                             zero_tol=zero_tol)
                decay_y[-1], _ = shear_interp.interpolate_y(cur_critical_shear)
                decay_x[-1] = cur_critical_shear

            # Update the current elevation
            cur_decay_y -= decay_inc
            cur_thickness = cur_decay_y - cur_layer_bottom
            if lower_layer_n is not None:  # Check if current layer is getting too thin to trust
                lower_layer_weaker = lower_layer_critical < cur_critical_shear and cur_thickness < min_thickness and \
                    lower_layer_shear > lower_layer_critical

        # Determine the shear at the scour depth (if exited because of scour depth)
        if is_live and scour_elevation > cur_decay_y:
            cur_shear = self.compute_shear(wse, scour_elevation, mannings_n, self.q2c)
            decay_x.append(cur_shear)
            decay_y.append(scour_elevation)

        if len(decay_x) > 0:
            marker_x.insert(0, decay_x[0])
            marker_y.insert(0, decay_y[0])
            marker_x.append(decay_x[-1])
            marker_y.append(decay_y[-1])

            self.results['Shear decay results']['Shear decay curve shear'] = decay_x
            self.results['Shear decay results']['Shear decay curve elevation'] = decay_y
            self.results['Shear decay results']['Shear decay curve shear markers'] = marker_x
            self.results['Shear decay results']['Shear decay curve elevation markers'] = marker_y

            # Set the determined scour depth
            self.scour_depth = thalweg - decay_y[-1]
        else:
            self.results['Shear decay results']['Shear decay curve shear'] = []
            self.results['Shear decay results']['Shear decay curve elevation'] = []
            self.results['Shear decay results']['Shear decay curve shear markers'] = []
            self.results['Shear decay results']['Shear decay curve elevation markers'] = []

        self.results['Shear decay results']['Scour depth with shear decay (ys)'] = self.scour_depth
        if not is_live:
            scour_elevation = None

        self.set_layer_plot_data(decay_x, decay_y, marker_x, marker_y, scour_elevation)

    def compute_pressure_flow_prerequisites(self):
        """Compute the effective flow and effective depth for pressure scour, used in live-bed/clear water."""
        _, q1 = self.get_data('Discharge in approach section (Q1)')
        if q1 is None:
            q1 = 0.0
        _, y1 = self.get_data('Average depth upstream (y1)')

        if self.compute_pressure:
            _, zero_tol = self.get_data('Zero tolerance')
            _, hu = self.get_data('Depth prior to scour at bridge (hb + ht or hu)', prior_keys=[
                'Pressure scour parameters'])
            _, hb = self.get_data('Vertical size of bridge opening (hb)', prior_keys=['Pressure scour parameters'])
            _, t = self.get_data('Deck thickness (t)', prior_keys=['Pressure scour parameters'])

            self.ht = hu - hb

            if self.ht <= 0.0:
                self.warnings['Bridge unsubmerged'] = 'The bridge is not submerged.  Review the upstream depth.'
                return

            if 'Pressure scour' not in self.results:
                self.results["Pressure scour"] = {}
            self.results['Pressure scour']['Distance from the WSE to the low chord of the bridge (ht)'] = self.ht

            self.hw = self.ht - t  # weir flow height
            if self.hw <= 0.0:
                self.hw = 0.0

            self.results['Pressure scour']['Weir flow height (hw)'] = self.hw

            if self.hw > zero_tol:
                self.hue = hb + t  # hue: Effective upstream channel flow depth
                self.que = q1 * (self.hue / y1) ** (8.0 / 7.0)  # Efffecive discharge
            else:
                self.hue = y1  # hue: Effective upstream channel flow depth
                self.que = q1  # Efffecive discharge

            self.results['Pressure scour']['Effective upstream channel flow depth (hue)'] = self.hue
            self.results['Pressure scour']['Effective upstream channel flow (que)'] = self.que
        else:
            self.hue = y1  # hue: Effective upstream channel flow depth
            self.que = q1  # Efffecive discharge

    def compute_pressure_flow(self):
        """Compute the scour depth based on the Pressure Flow method."""
        _, hu = self.get_data('Depth prior to scour at bridge (hb + ht or hu)',
                              prior_keys=['Pressure scour parameters'])
        _, hb = self.get_data('Vertical size of bridge opening (hb)', prior_keys=['Pressure scour parameters'])

        ht = self.ht
        hw = self.hw
        # Equation 6.16, HEC-18 page 6-25 (pdf page 161)
        flow_separation_thickness_t = 0.0
        if self.hue > 0.0 and hb > 0.0:
            flow_separation_thickness_t = 0.5 * (hb * ht / (hu ** 2)) ** 0.2 * (1 - hw / ht) ** -0.1 * hb
        if 'Pressure scour' not in self.results:
            self.results["Pressure scour"] = {}

        self.results['Pressure scour']['Flow separation thickness (t)'] = flow_separation_thickness_t

        _, y2 = self.get_data('Depth prior to scour in contracted section (y0)')
        pressure_scour_depth = y2 + flow_separation_thickness_t - hb

        self.results['Pressure scour']['Pressure scour depth (y2)'] = pressure_scour_depth

        return pressure_scour_depth

    def compute_live_bed_scour_depth(self):
        """Compute the scour depth based on the Live Bed method."""
        self.live_bed_scour_depth = 0.0

        self.results['k exponent description'] = 'User specified'
        _, k = self.get_data('k exponent')
        _, sys_complexity = self.get_data('Complexity')
        _, k_pref = self.get_data('k exponent preference')
        if sys_complexity < 2 or k_pref != 'specify k exponent':
            self.results['k exponent description'] = 'User specified'
            k = self.compute_k_exponent()
        self.results['k exponent'] = k

        # Equation 6.2, HEC-18 page 6.10 (pdf page 146)
        _, q2 = self.get_data('Discharge in contracted section (Q2)')
        _, w1 = self.get_data('Bottom width in approach section (W1)')
        _, w2 = self.get_data('Bottom width in contracted section (W2)')
        _, y0 = self.get_data('Depth prior to scour in contracted section (y0)')

        # These values are determined in the pressure flow method
        # If pressure flow is not computed, we use the y1, q1
        y1 = self.hue
        q1 = self.que

        y2 = 0.0
        if q1 > 0:
            # Average Depth in the contracted section
            # Equation 6.2, HEC-18 page 6.10 (pdf page 146)
            y2 = (q2 / q1)**(6.0 / 7.0) * (w1 / w2)**k * y1

        # Equation 6.3
        # Average scour depth in the contracted section
        ys = y2 - y0
        self.live_bed_scour_depth = ys

        self.results['Average live-bed scour depth (y2)'] = y2
        self.results['Live-bed scour depth (ys)'] = ys

        # Live bed check
        self.compute_parameters_for_live_bed_check(q2, w2, y2)

        return ys

    def compute_parameters_for_live_bed_check(self, q2, w2, y2):
        """Performs a check whether live bed governs or not."""
        # HEC-18 Equation C.4 on page C.5 (pdf page 311)
        ks = 0.039
        stricklers_n = 0.034 * self.upstream_d50**0.1666666666
        velocity = q2 / (w2 * y2)
        _, manning_k = self.get_data('Manning constant', 1.486)
        _, gamma_w = self.get_data('Unit weight of water (γw)')
        gamma_s = None
        if self.gradations is not None:
            _, input_options = self.get_data('Contracted gradation input options',
                                             input_dict=self.gradations.input_dict)
            if input_options != 'Single D50':
                if self.layer_index is None:
                    self.layer_index = self.get_layer_index_from_elevation(self.centerline_streambed)[0]
                gamma_s = self.bh_layers[self.layer_index]['Unit weight of soil (γs)']
        if gamma_s is None:
            _, gamma_s = self.get_data('Unit weight of soil (γs)')

        self.applied_shear = gamma_w * stricklers_n * stricklers_n * \
            velocity * velocity / (manning_k * manning_k) * (y2 ** 0.333333333)

        self.critical_shear = ks * (gamma_s - gamma_w) * self.upstream_d50

        self.results['Applied shear (ϑ0)'] = self.applied_shear
        self.results['Critical shear (ϑc)'] = self.critical_shear

    def compute_clear_water_scour_depth(self):
        """Compute the scour depth based on the Live Bed method."""
        self.clear_water_scour_depth = 0.0

        # Equation 6.4, HEC-18 page 6-12 (pdf page 148)
        ku = 0.0077
        dm = self.d50 * 1.25  # Diameter Of Smallest Nonmoving Particle
        _, q2 = self.get_data('Discharge in contracted section (Q2)')
        _, w2 = self.get_data('Bottom width in contracted section (W2)')
        _, y0 = self.get_data('Depth prior to scour in contracted section (y0)')

        y2 = 0.0
        if dm != 0.0 and w2 != 0.0:
            y2 = ((ku * q2**2) / ((dm**0.66666666666) * w2**2))**0.428571428571

        ys = y2 - y0
        self.clear_water_scour_depth = ys

        self.results['Average clear-water scour depth (y2)'] = y2
        self.results['Clear-water scour depth (ys)'] = ys

        return ys

    def compute_k_exponent(self):
        """Computes the k exponent used in live bed scour.

        Returns:
            k exponent (float): k exponent used in live bed contraction scour equation
        """
        fall_velocity = self.compute_fall_velocity()

        _, g = self.get_data('Gravity', 32.2)
        _, y1 = self.get_data('Average depth upstream (y1)')
        _, s1 = self.get_data('Slope of energy grade line (S1)')
        shear_velocity = math.sqrt(g * y1 * s1)

        self.results['Shear velocity (v*)'] = shear_velocity
        self.results['Fall velocity (t)'] = fall_velocity

        return self.interpolate_k_exponent(fall_velocity, shear_velocity)

    def interpolate_k_exponent(self, fall_velocity, shear_velocity):
        """Determines the k exponent based on shear velocity and fall velocity.

        Args:
            fall velocity (float): shear velocity for d50 at water temperature.
            shear velocity (float): shear velocity for water depth and EGL.

        Returns:
            k exponent (float): k exponent used in live bed contraction scour
        """
        k_exponent = 0.0
        k_exponent_description = ''
        vshear_over_fall_velocity = 0.0
        if fall_velocity != 0.0:
            vshear_over_fall_velocity = shear_velocity / fall_velocity
        if vshear_over_fall_velocity <= 0.0:
            k_exponent_description = 'Calculation failed determining the k exponent.'
        elif vshear_over_fall_velocity < 0.5:
            k_exponent_description = 'Mostly contact bed material discharge.'
            k_exponent = 0.59
        elif vshear_over_fall_velocity < 2.0:
            k_exponent_description = 'Some suspended bed material discharge.'
            k_exponent = 0.64
        else:
            k_exponent_description = 'Mostly suspended bed material discharge.'
            k_exponent = 0.69
        self.results['k exponent description'] = k_exponent_description
        return k_exponent

    def compute_fall_velocity(self):
        """Computes the fall velocity used in the k exponent used in live bed scour.

        Returns:
            fall_velocity (float): The fall velocity of a D50 for a given water temperature
        """
        _, temperature = self.get_data('Temperature of water')
        _, temperature_in_celsius = self.unit_converter.convert_units('°F', '°C', temperature)

        temperature_list = [0.0, 20.0, 40.0]
        omega_list = [
            self.compute_omega_at_t0(self.upstream_d50),
            self.compute_omega_at_t20(self.upstream_d50),
            self.compute_omega_at_t40(self.upstream_d50)
        ]

        _, null_data = self.get_data('Null data')
        _, zero_tol = self.get_data('Zero tolerance')
        t_w_interp = Interpolation(temperature_list, omega_list, null_data=null_data,
                                   zero_tol=zero_tol)
        fall_velocity, _ = t_w_interp.interpolate_y(temperature_in_celsius, False)

        return fall_velocity

    def compute_omega_at_t0(self, d50):
        """Computes the omega for temperature = 0 degrees Celsius.

        Interpolates from figure 6.8 in HEC 18 April 2012, page 6.11 (pdf page 147)

        Args:
            d50 (float): the d50 of the material measured in feet.

        Return:
            omega (float): The omega value at D50 and t = 0
        """
        _, d50_in_mm = self.unit_converter.convert_units('ft', 'mm', d50)
        omega_in_m, _ = self.t0_interpolate.interpolate_y(d50_in_mm, False)
        _, omega = self.unit_converter.convert_units('m', 'ft', omega_in_m)
        return omega

    def compute_omega_at_t20(self, d50):
        """Computes the omega for temperature = 2 degrees Celsius.

        Interpolates from figure 6.8 in HEC 18 April 2012, page 6.11 (pdf page 147)

        Args:
            d50 (float): the d50 of the material measured in feet.

        Return:
            omega (float): The omega value at D50 and t = 20
        """
        _, d50_in_mm = self.unit_converter.convert_units('ft', 'mm', d50)
        omega_in_m, _ = self.t20_interpolate.interpolate_y(d50_in_mm, False)
        _, omega = self.unit_converter.convert_units('m', 'ft', omega_in_m)
        return omega

    def compute_omega_at_t40(self, d50):
        """Computes the omega for temperature = 40 degrees Celsius.

        Interpolates from figure 6.8 in HEC 18 April 2012, page 6.11 (pdf page 147)

        Args:
            d50 (float): the d50 of the material measured in feet.

        Return:
            omega (float): The omega value at D50 and t = 40
        """
        _, d50_in_mm = self.unit_converter.convert_units('ft', 'mm', d50)
        omega_in_m, _ = self.t40_interpolate.interpolate_y(d50_in_mm, False)
        _, omega = self.unit_converter.convert_units('m', 'ft', omega_in_m)
        return omega
