"""Pile Stem Data."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier.Complex.pier_stem_calc import PierStemCalc


class PierStem(VariableGroup):
    """Provides a class that will Handle input for complex pier pile caps."""
    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Specify Pile Caps.

        Args:
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.input = {}

        self.calculator = PierStemCalc()

        pier_shape_options = [
            'Square Nose', 'Round Nose', 'Sharp Nose', 'Circular Cylinder',
            'Group of Cylinders',
        ]

        self.input['Compute pier stem'] = Variable('Compute pier stem scour component', "bool", True)

        self.input['Pier shape (K1)'] = Variable('Pier nose shape (K1)', "list", 1,
                                                 pier_shape_options)

        self.input['Pier width (ap)'] = Variable(
            'Pier width (ap)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='Horizontal size parallel to the bridge, not adjusted for skew')

        self.input['Pier length (Lp)'] = Variable(
            'Pier length (Lp)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='Horizontal size perpendicular to the bridge, not adjusted for skew')

        self.input['Distance between front edge (f)'] = Variable(
            'Distance between front edge of pile cap or footing and pier (f)',
            'float',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='')

        # Intermediate
        # self.compute_prep_functions = []  # Functions to call before compute_data
        # Define variables to copy to the calculator
        self.intermediate_to_copy.extend(['is_hec18_complex', 'is_fdot', 'is_fdot_complex'])

        self.is_hec18_complex = False
        self.is_fdot = False
        self.is_fdot_complex = False

    def get_input_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        if self.is_fdot_complex and not self.input['Compute pier stem'].get_val():
            input_vars['Compute pier stem'] = self.input['Compute pier stem']
            return input_vars

        input_vars = copy.deepcopy(self.input)

        if self.input['Pier shape (K1)'].get_val() in ['Circular Cylinder', 'Group of Cylinders']:
            self.input['Pier length (Lp)'].set_val(self.input['Pier width (ap)'].get_val())
            input_vars.pop('Pier length (Lp)')

        if not self.is_hec18_complex:
            input_vars.pop('Distance between front edge (f)')

        if self.is_fdot:
            input_vars['Pier shape (K1)'].value_options = ['Square Nose', 'Round Nose', 'Sharp Nose']
            input_vars['Pier shape (K1)'].name = 'Pier shape (Ks & Kp)'
        else:
            input_vars['Pier shape (K1)'].name = 'Pier shape (K1)'

        if not self.is_fdot_complex:
            input_vars.pop('Compute pier stem')

        return input_vars
