"""Bridge Scour Hole Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class ScourHoleCalc(Calculator):
    """Provides a class that will define the cross-section of a bridge."""

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        result = self.check_float_vars_to_greater_zero(['Scour depth'])

        return result

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        # Document the source of the equation (if possible):
        #   Give publication, date, page number and link (if there's a bug, that it can be traced back to the source).
        # Assign new variables from the input_dict in similar names as the equation and document units

        # Perform the computation
        _, width_to_depth_ratio = self.get_data('Width to depth ratio')
        _, angle_r = self.get_data('Angle of repose (Θ)')
        _, depth = self.get_data('Scour depth')
        b_width = depth * width_to_depth_ratio

        if self.is_pier_scour_hole:
            if self.get_data('Use pier width')[1]:
                b_width = self.pier_width
                if self.streambed_elevation != self.null_data and len(self.pier_widths) > 0:
                    new_width = self.find_largest_width(depth)
                    if new_width > 0.0:
                        b_width = new_width

        if self.use_specified_b_width:
            b_width = self.b_width

        cotangent_angle = math.cos(angle_r) / math.sin(angle_r)

        t_width = depth * (width_to_depth_ratio + cotangent_angle) + b_width

        self.results['Bottom width'] = b_width
        self.results['Top width'] = t_width
        # Assign the results to the results and plot_dict and log any warnings or errors

        return True

    def find_largest_width(self, depth):
        """Find the largest pier width between streambed elevation and lower elevation.

        Args:
            depth (float): The depth to subtract from streambed elevation.

        Returns:
            float: The largest pier width in the specified range.
        """
        lower_elevation = self.streambed_elevation - depth

        # Determine the width at the streambed elevation and scour depth (the pier may be slowly change width)
        _, null_data = self.get_data('Null data')
        _, zero_tol = self.get_data('Zero tolerance')
        width_interp = Interpolation(self.pier_elevations, self.pier_widths, input_dict=self.input_dict,
                                     null_data=null_data, zero_tol=zero_tol)
        lower_width, _ = width_interp.interpolate_y(lower_elevation)
        streambed_width, _ = width_interp.interpolate_y(self.streambed_elevation)

        max_width = max(lower_width, streambed_width)

        for elevation, width in zip(self.pier_elevations, self.pier_widths):
            if lower_elevation <= elevation <= self.streambed_elevation:
                if width > max_width:
                    max_width = width

        return max_width
