"""Provides a class that will Handle user-selected for boundary conditions."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules
from sortedcontainers import SortedDict

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class InletControlCalc(Calculator):
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """
    culv_shape_to_chan_shape = {
        'Circular': 'circle',
        'Concrete Box': 'box',
        'Elliptical': 'elliptical',
        'Pipe Arch': 'pipe arch',
        'User Defined': 'cross-section',
        'Arch, Open Bottom': 'cross-section',
        'Low-Profile Arch': 'cross-section',
        'High-Profile Arch': 'cross-section',
        'Metal Box': 'cross-section',
        'Concrete Open-Bottom Arch': 'cross-section',
        'South Dakota Concrete Box Culvert': 'box',
        'Horseshoe': 'horseshoe',
    }
    shapes_in_sd_but_can_compute = [
        'Elliptical', 'Pipe Arch',
    ]

    # Interpolation
    # X for interpolation: Q/AD^05; for all interpolation curves
    user_x = [0, 0.25, 0.5, 1.0, 2, 3, 3.5, 4, 5, 6, 7, 8, 9]

    # Chart 51-B Circular and Elliptical Structural Plate Corrugated Metal Conduits
    # Beveled Edge
    # KE = .2
    user_cir_beveled_ke = 0.2
    # SR = .5
    # HW/D
    user_cir_beveled_hwd = [
        0.031, 0.225, 0.33, 0.49, 0.74, 0.96, 1.0825, 1.205, 1.48, 1.81, 2.2, 2.66, 3.16
    ]
    # Mitered
    # KE = .7
    user_cir_mitered_ke = 0.7
    # SR = -.7
    user_cir_mitered_hwd = [
        0.06, 0.285, 0.39, 0.56, 0.84, 1.07, 1.23, 1.42, 1.89, 2.43, 3.05, 3.67, 4.29
    ]
    # Square Edge
    # KE = 0.5
    user_cir_square_ke = 0.5
    # SR = 0.5
    user_cir_square_hwd = [
        0.03, 0.23, 0.33, 0.51, 0.775, 1.025, 1.14, 1.27, 1.59, 1.98, 2.5, 3.07, 3.64
    ]
    # Projecting
    # KE = 0.9
    user_cir_projecting_ke = 0.9
    # SR = 0.5
    user_cir_projecting_hwd = [
        0.04, 0.235, 0.345, 0.52, 0.84, 1.125, 1.2625, 1.4, 1.79, 2.34, 2.99, 3.64, 4.29
    ]

    # Chart 52-B High and low profile Structural Plate Corrugated Metal Arch
    # Beveled Edge
    # KE = .2
    user_arch_beveled_ke = 0.2
    # SR = .5
    # HW/D
    user_arch_beveled_hwd = [
        0.031, 0.1655, 0.3, 0.44, 0.69, 0.88, 1.015, 1.15, 1.47, 1.8, 2.22, 2.66, 3.16
    ]
    # Mitered
    # KE = .7
    user_arch_mitered_ke = 0.7
    # SR = -.7
    user_arch_mitered_hwd = [
        0.04, 0.22, 0.35, 0.5, 0.77, 1.06, 1.255, 1.45, 1.91, 2.46, 3.07, 3.68, 4.29
    ]
    # Square Edge
    # KE = 0.5
    user_arch_square_ke = 0.5
    # SR = 0.5
    user_arch_square_hwd = [
        0.03, 0.155, 0.28, 0.46, 0.73, 0.96, 1.11, 1.26, 1.59, 2.01, 2.51, 3.08, 3.64
    ]
    # Projecting
    # KE = 0.9
    user_arch_projecting_ke = 0.9
    # SR = 0.5
    user_arch_projecting_hwd = [
        0.03, 0.19, 0.3, 0.49, 0.81, 1.11, 1.26, 1.41, 1.82, 2.38, 3.02, 3.66, 4.3
    ]

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        result = True

        culv_shape = self.input_dict['calc_data']['Culvert shape']
        chan_shape = InletControlCalc.culv_shape_to_chan_shape[culv_shape]

        geom = self.input_dict['calc_data']['Geometry']['calculator']
        geom.input_dict['calc_data']['Shape'] = chan_shape
        diameter = geom.input_dict['calc_data']['Diameter']
        span = geom.input_dict['calc_data']['Span']
        rise = geom.input_dict['calc_data']['Rise']

        if chan_shape in ['circle',]:
            rise = diameter
            span = diameter
        elif chan_shape in ['cross-section',]:
            cs_x = self.input_dict['calc_data']['Geometry']['Cross-section']['Station']
            cs_y = self.input_dict['calc_data']['Geometry']['Cross-section']['Elevation']
            rise = max(cs_y) - min(cs_y)
            span = max(cs_x) - min(cs_x)
            geom.input_dict['calc_data']['Span'] = span
            geom.input_dict['calc_data']['Rise'] = rise

        if span <= 0.0:
            self.warnings['Span'] = 'Span is not specified.'
            result = False
        if rise <= 0.0:
            self.warnings['Rise'] = 'Rise is not specified.'
            result = False

        return result

    def _compute_data(self, critical_depth=None, critical_flowarea=None):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.initialize_inlet()

        flows = self.input_dict['calc_data']['Flows']

        self.hw = []
        for flow in flows:
            hw = self.compute_inlet_head_for_flow(flow, critical_depth, critical_flowarea)
            self.hw.append(hw)

        self.results['hw'] = self.hw
        return self.hw

    def initialize_inlet(self):
        """Initialize the inlet calculator."""
        compute_geometry = self.input_dict['calc_data']['Compute geometry']
        specify_dimensions = self.input_dict['calc_data']['Specify dimensions']
        shape = self.get_channel_shape_from_culvert_shape(self.input_dict['calc_data']['Culvert shape'],
                                                          compute_geometry, specify_dimensions)
        geom = self.input_dict['calc_data']['Geometry']['calculator']
        geom.input_dict['calc_data']['Shape'] = shape
        rise = geom.input_dict['calc_data']['Rise']
        span = geom.input_dict['calc_data']['Span']
        diameter = geom.input_dict['calc_data']['Diameter']
        if shape == 'circle':
            rise = diameter
            span = diameter
        geom.input_dict['calc_data']['Depths'] = [rise]
        geom.compute_data()
        full_flow_area = geom.full_flow_area

        self.initialize_inlet_data_for_culvert(shape, span, rise, full_flow_area)

    def get_channel_shape_from_culvert_shape(self, culvert_shape, compute_geometry=False, specify_dimensions=False):
        """Get the channel shape corresponding to the culvert shape.

        Args:
            culvert_shape (str): the culvert shape

        Returns:
            str: the channel shape
        """
        channel_shape = InletControlCalc.culv_shape_to_chan_shape[culvert_shape]
        if (not compute_geometry and not specify_dimensions) and culvert_shape in \
                InletControlCalc.shapes_in_sd_but_can_compute:
            channel_shape = 'cross-section'
        return channel_shape

    def set_ke(self):
        """Sets the KE value, if not specified by the user."""
        if self.input_dict['calc_data']['Inlet type'] in ['manual polynomial']:
            # ke value is user-entered
            return None
        elif self.input_dict['calc_data']['Inlet type'] in ['best available'] and self.polynomial_available:
            # ke value is set with the shape database polynomial coefficients
            return None
        else:
            if self.input_dict['calc_data']['Inlet type'] == 'interpolation: circular or elliptical':
                if self.input_dict['calc_data']['Inlet configuration'] == 'beveled edge':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_cir_beveled_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'mitered to conform to slope':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_cir_mitered_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'square edge with headwall':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_cir_square_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'thin edge projecting':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_cir_projecting_ke
            elif self.input_dict['calc_data']['Inlet type'] == 'interpolation: arch or embedded':
                if self.input_dict['calc_data']['Inlet configuration'] == 'beveled edge':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_arch_beveled_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'mitered to conform to slope':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_arch_mitered_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'square edge with headwall':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_arch_square_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'thin edge projecting':
                    self.input_dict['calc_data']['KE'] = InletControlCalc.user_arch_projecting_ke
        return self.input_dict['calc_data']['KE']

    # def initialize_inlet_data_for_culvert_with_manning_n_calc(self, manning_n_calc):
    #     """Compute and set coefficient and data for the culvert to perform inlet computations.

    #     Args:
    #         manning_n_calc (ManningsNCalculator): The Manning's n calculator to use for computations.
    #     """
    #     manning_n_calc.get_can_compute()
    #     manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].initialize_geometry()
    #     shape = manning_n_calc.input_dict['calc_data']['Geometry']['Shape']
    #     span = manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Span']
    #     rise = manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Rise']
    #     fullflow_area = manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].full_flow_area
    #     slope = manning_n_calc.input_dict['calc_data']['Slope']
    #     self.input_dict['calc_data']['Slope'] = slope

    #     if manning_n_calc.can_compute:
    #         self.initialize_inlet_data_for_culvert(shape, span, rise, fullflow_area)

    def initialize_inlet_data_for_culvert(self, shape, span, rise, fullflow_area):
        """Compute and set coefficient and data for the culvert to perform inlet computations.

        Args:
            shape (str): the shape of the culvert
            span (float): the span of the culvert
            rise (float): the rise of the culvert
            fullflow_area (float): the full flow area of the culvert
            slope (float): the slope of the culvert
            mannings_n_data (ManningsNData): the Manning's n data for the culvert

        Returns:
            bool: True if successful

        """
        self.flow_at_half_rise = 0.0
        self.flow_at_triple_rise = 0.0

        _, g = self.get_data('Gravity')

        self.manning_n_calc.clear_results()

        # Set given data
        self.input_dict['calc_data']['Shape'] = shape
        self.input_dict['calc_data']['Rise'] = rise
        self.input_dict['calc_data']['Span'] = span
        self.input_dict['calc_data']['Full flow area'] = fullflow_area

        # create variables for transition points
        half_rise = rise / 2.0
        triple_rise = 3.0 * rise

        _, null_data = self.get_data('Null data')
        _, zero_tol = self.get_data('Zero tolerance')
        # Add Zero (floor of computation)
        self.flow_unk_sorted = SortedDict({0.0: 0.0})
        self.flow_interp = Interpolation([], [], null_data=null_data, zero_tol=zero_tol)

        # Compute a range of flows with the regression equations to enable our interpolation routines to find the flow
        # at specific headwaters
        new_flow = 1.0
        ih = 0.0
        count = 0
        _, max_iterations = self.get_data('Max number of iterations', 500)
        while (ih < half_rise) and (count < max_iterations):
            ih = self.compute_hw_regression_equations_for_flow(new_flow)
            self.flow_unk_sorted[ih] = new_flow
            new_flow *= 2.0
            count += 1
        if count >= max_iterations:
            self.warnings['Inlet control'] = 'Maximum iterations exceeded computing regression equation flows.'

        # The following methods come from:
        # "Hydraulic Analysis of Culverts by Microcomputer," A Thesis in Civil Engineering by Arthur C. Parola,
        # Jr., May 1987, 84p. Chapter 3, Program Computations, page 20

        # Compute the lower limit velocity head coefficient (kelow) at half rise
        # The velocity head coefficient (kelow) is determined by setting the minimum energy equation equal to the
        # regression equation at one half the rise, which is the lower limit of the regression equation.
        self.flow_at_half_rise = self.compute_flow_for_given_inlet_hw(half_rise)

        # Determine flow conditions at critical depth for given discharge
        self.manning_n_calc.input_dict['calc_data']['Flows'] = [self.flow_at_half_rise]
        self.manning_n_calc.input_dict['calc_data']['Geometry'] = self.input_dict['calc_data']['Geometry']
        self.manning_n_calc.initialize_geometry()
        critical_depth_with_flow_at_half_rise = self.manning_n_calc._compute_critical_depth_from_flow()[0]
        critical_flow_area_with_flow_at_half_rise = self.manning_n_calc.results['Critical flow area'][0]

        if self.flow_at_half_rise == 0.0 or critical_flow_area_with_flow_at_half_rise == 0.0:
            self.kelow = 0.0
        else:
            self.kelow = ((half_rise - critical_depth_with_flow_at_half_rise)) * (2.0 * g) / \
                (self.flow_at_half_rise ** 2 / critical_flow_area_with_flow_at_half_rise ** 2) - 1.0

        # Compute the upper limit coefficient at 3 * rise
        # The coefficient of contraction (cdahi) is determined by setting the orifice equation equal to the regression
        # equation at the upper limit of the regression equation.
        self.flow_at_triple_rise = self.compute_flow_for_given_inlet_hw(triple_rise)

        self.cdahi = 0.0
        if rise > 0.0:
            self.cdahi = self.flow_at_triple_rise / math.sqrt(2.5 * rise)

    def compute_inlet_head_for_flow(self, flow, critical_depth=None, critical_flowarea=None):
        """Computes the inlet head for the given flow.

        Returns:
            bool: True if successful
        """
        ih = 0.0

        _, g = self.get_data('Gravity')

        # The following methods come from:
        # "Hydraulic Analysis of Culverts by Microcomputer," A Thesis in Civil Engineering by Arthur C. Parola, Jr.,
        # May 1987, 84p. Chapter 3, Program Computations, page 20
        # if flow is less than half the rise (below experimental data from regression curves), use the energy
        # equation to determine the inlet head control depth
        if flow < self.flow_at_half_rise:
            # Method to adjust the coefficients to keep flow area from going to zero as we approach zero depth
            shape = self.input_dict['calc_data']['Shape']

            # Determine flow conditions at critical depth for given discharge
            self.manning_n_calc.input_dict['calc_data']['Flows'] = [flow]
            if critical_depth is None:
                self.manning_n_calc.clear_results()
                critical_depth = self.manning_n_calc._compute_critical_depth_from_flow()[-1]
                critical_flowarea = self.manning_n_calc.results['Critical flow area'][-1]
            if (critical_flowarea > 0.0):
                vel_head = flow**2 / (critical_flowarea**2 * (2 * g))
            else:
                vel_head = 0.0

            fract = 1.0
            vhcoef = 1.0
            lmult = 1.0
            if shape not in ['circle', 'box', ]:
                q15 = self.flow_at_half_rise * 0.15
                # q10 = self.flow_at_half_rise * 0.1
                fract = (q15 - flow) / (q15)
                vhcoef = (1 - fract) / (1 + vel_head * fract)
            ih = critical_depth * lmult + (1 + self.kelow) * vel_head * vhcoef
        # if the flow is greater than half the rise and less than 3 times the rise, use the regression equations
        elif flow <= self.flow_at_triple_rise:
            ih = self.compute_hw_regression_equations_for_flow(flow)
        # if the flow is greater than 3 times the rise, use the orifice equation
        else:
            ih = flow**2 / self.cdahi**2 + 0.5 * self.input_dict['calc_data']['Rise']

        return ih

    def compute_flow_for_given_inlet_hw(self, hw):
        """Compute the flow required to reach a given HW.

        Args:
            hw (float): the given headwater depth
        Returns:
            float: flow for the given headwater
        """
        flow_list = list(self.flow_unk_sorted.values())
        hw_list = list(self.flow_unk_sorted.keys())
        if hw in hw_list:
            return self.flow_unk_sorted[hw]
        self.flow_interp.x = hw_list
        self.flow_interp.y = flow_list

        flow_guess, _ = self.flow_interp.interpolate_y(hw)

        _, hw_tol = self.get_data('HW error')
        _, max_loops = self.get_data('Max number of iterations')
        difference = 1.0
        count = 0

        while hw_tol < difference and count < max_loops:
            ih = self.compute_hw_regression_equations_for_flow(flow_guess)
            self.flow_unk_sorted[ih] = flow_guess

            # # Convert the dictionary to a list so we can interpolate from it
            flow_list = list(self.flow_unk_sorted.values())
            hw_list = list(self.flow_unk_sorted.keys())
            self.flow_interp.x = hw_list
            self.flow_interp.y = flow_list
            flow_guess, _ = self.flow_interp.interpolate_y(hw)

            difference = abs(hw - ih)
            count += 1

        return flow_guess

    def compute_hw_regression_equations_for_flow(self, flow):
        """Computes the inlet head from regression equation given only the flow.

        Requires the initialize_inlet_data_for_culvert to be run first

        Args:
            flow (float): Given flow to determine the inlet head

        Returns:
            float: inlet control headwater
        """
        rise = self.input_dict['calc_data']['Rise']
        span = self.input_dict['calc_data']['Span']
        fullflow_area = self.input_dict['calc_data']['Full flow area']
        slope = self.input_dict['calc_data']['Slope']

        return self.compute_hw_regression_equations_for_flow_area_span_rise_slope(flow, fullflow_area, span, rise,
                                                                                  slope)

    # flow_area == full flow area?  I think so...
    def compute_hw_regression_equations_for_flow_area_span_rise_slope(self, flow, flow_area, span, rise, slope):
        """Computes the data possible; stores results in self.

        Returns:
            float: inlet control headwater
        """
        hw = 0.0

        self.set_ke()

        if flow <= 0.0 or flow_area <= 0.0 or rise <= 0.0:
            return hw

        # TODO Find where KE is used in HY-8!
        sr = 0.0  # Slope correction coefficient

        result_note = ""

        if self.input_dict['calc_data']['Inlet type'] == 'manual polynomial' or \
                (self.input_dict['calc_data']['Inlet type'] == 'best available' and self.polynomial_available):
            x = flow / (span * rise**1.5)
            result_note = "Inlet control depth computed using polynomial regression equation."
            if self.use_qad_poly:
                # Compute Q/AD^0.5; some shapes like CONSPAN were developed with Q/AD^0.5
                x = flow / (flow_area * math.sqrt(rise))
                result_note = "Inlet control depth computed using polynomial regression equation with Q/AD^0.5 format."

            a = self.input_dict['calc_data']['A']
            b = self.input_dict['calc_data']['BS']
            c = self.input_dict['calc_data']['C']
            d = self.input_dict['calc_data']['DIP']
            e = self.input_dict['calc_data']['EE']
            f = self.input_dict['calc_data']['F']
            sr = self.input_dict['calc_data']['SR']

            # Polynomial Equation
            hw_d = a + b * x + c * x**2 + d * x**3 + e * x**4 + f * x**5 - sr * slope
            if hw_d < 0.0:
                hw_d = 0.0
        else:
            # Compute Q/AD^0.5
            q_ad = flow / (flow_area * math.sqrt(rise))
            x = InletControlCalc.user_x
            if self.input_dict['calc_data']['Inlet type'] == 'interpolation: circular or elliptical' or \
                    (self.input_dict['calc_data']['Inlet type'] == 'best available' and self.use_cir_ell_interp):
                result_note = "Inlet control depth computed using interpolation curves for circular and elliptical ' \
                    'shapes."
                if self.input_dict['calc_data']['Inlet configuration'].lower() == 'beveled edge':
                    y = InletControlCalc.user_arch_beveled_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'].lower() == 'mitered to conform to slope':
                    y = InletControlCalc.user_arch_mitered_hwd
                    sr = -0.7
                elif self.input_dict['calc_data']['Inlet configuration'].lower() == 'square edge with headwall':
                    y = InletControlCalc.user_arch_square_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'].lower() == 'thin edge projecting':
                    y = InletControlCalc.user_arch_projecting_hwd
                    sr = 0.5
            elif self.input_dict['calc_data']['Inlet type'] == 'interpolation: arch or embedded' or \
                    (self.input_dict['calc_data']['Inlet type'] == 'best available' and not self.use_cir_ell_interp):
                result_note = "Inlet control depth computed using interpolation curves for arch and embedded shapes."
                if self.input_dict['calc_data']['Inlet configuration'].lower() == 'beveled edge':
                    y = InletControlCalc.user_cir_beveled_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'].lower() == 'mitered to conform to slope':
                    y = InletControlCalc.user_cir_mitered_hwd
                    sr = -0.7
                elif self.input_dict['calc_data']['Inlet configuration'].lower() == 'square edge with headwall':
                    y = InletControlCalc.user_cir_square_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'].lower() == 'thin edge projecting':
                    y = InletControlCalc.user_cir_projecting_hwd
                    sr = 0.5

            else:
                return

            self.warnings['Result Note'] = result_note
            _, null_data = self.get_data('Null data')
            _, zero_tol = self.get_data('Zero tolerance')
            interpolate = Interpolation(x, y, null_data=null_data, zero_tol=zero_tol)
            hw_d = interpolate.interpolate_y(q_ad, True)[0] - sr * slope
            if hw_d < 0.0:
                hw_d, _ = interpolate.interpolate_y(q_ad, True)
        hw = hw_d * rise
        return hw
