"""SiteData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules


class SiteCalc(Calculator):
    """Provides a class that will define the site data of a culvert barrel."""

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        _, zero_tol = self.get_data('Zero tolerance')
        if self.input_dict['calc_data']['Culvert type'] == 'straight':
            up_station = self.input_dict['calc_data']['Inlet invert station']
            down_station = self.input_dict['calc_data']['Outlet invert station']
            if math.isclose(up_station, down_station, abs_tol=zero_tol):
                self.warnings['Outlet invert station'] = 'Please enter all stations'
                return False
        elif self.input_dict['calc_data']['Culvert type'] == 'single broken-back':
            up_station = self.input_dict['calc_data']['Inlet invert station']
            break_station = self.input_dict['calc_data']['Break invert station']
            down_station = self.input_dict['calc_data']['Outlet invert station']
            if math.isclose(up_station, break_station, abs_tol=zero_tol) or \
                    math.isclose(break_station, down_station, abs_tol=zero_tol):
                self.warnings['Outlet invert station'] = 'Please enter all stations'
                return False
        elif self.input_dict['calc_data']['Culvert type'] == 'double broken-back':
            up_station = self.input_dict['calc_data']['Inlet invert station']
            upper_break_station = self.input_dict['calc_data']['Upper break invert station']
            lower_break_station = self.input_dict['calc_data']['Lower break invert station']
            down_station = self.input_dict['calc_data']['Outlet invert station']
            if math.isclose(up_station, upper_break_station, abs_tol=zero_tol) or \
                    math.isclose(upper_break_station, lower_break_station, abs_tol=zero_tol) or \
                    math.isclose(lower_break_station, down_station, abs_tol=zero_tol):
                self.warnings['Outlet invert station'] = 'Please enter all stations'
                return False
        elif self.input_dict['calc_data']['Culvert type'] == 'inverted siphon':
            self.warnings['Inverted siphon'] = 'Inverted siphon is not yet implemented'
            return False
        return True

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        _, zero_tol = self.get_data('Zero tolerance')
        if zero_tol < self.embedment_depth:
            if self.input_dict['calc_data']['Entry options'] == 'specify streambed elevation':
                self.input_dict['calc_data']['Inlet invert elevation'] = (
                    self.input_dict['calc_data']['Inlet streambed elevation'] - self.embedment_depth)
                self.input_dict['calc_data']['Outlet invert elevation'] = (
                    self.input_dict['calc_data']['Outlet streambed elevation'] - self.embedment_depth)
            else:
                self.input_dict['calc_data']['Inlet streambed elevation'] = (
                    self.input_dict['calc_data']['Inlet invert elevation'] + self.embedment_depth)
                self.input_dict['calc_data']['Outlet streambed elevation'] = (
                    self.input_dict['calc_data']['Outlet invert elevation'] + self.embedment_depth)
        else:  # Set the streambed elevations to invert elevations
            self.input_dict['calc_data']['Inlet streambed elevation'] = self.input_dict['calc_data'][
                'Inlet invert elevation']
            self.input_dict['calc_data']['Outlet streambed elevation'] = self.input_dict['calc_data'][
                'Outlet invert elevation']

    def get_slope_of_final_barrel(self):
        """Computes and returns the slope of the final barrel.

        Returns:
            slope (float): the slope of the final barrel.
        """
        rise = self.input_dict['calc_data']['Inlet invert elevation'] - self.input_dict['calc_data'][
            'Outlet invert elevation']
        run = self.input_dict['calc_data']['Outlet invert station'] - self.input_dict['calc_data'][
            'Inlet invert station']
        if self.input_dict['calc_data']['Culvert type'] == 'single broken-back':
            rise = self.input_dict['calc_data']['Break invert elevation'] - self.input_dict['calc_data'][
                'Outlet invert elevation']
            run = self.input_dict['calc_data']['Outlet invert station'] - self.input_dict['calc_data'][
                'Break invert station']
        elif self.input_dict['calc_data']['Culvert type'] == 'double broken-back':
            rise = self.input_dict['calc_data']['Lower break invert elevation'] - self.input_dict['calc_data'][
                'Outlet invert elevation']
            run = self.input_dict['calc_data']['Outlet invert station'] - self.input_dict['calc_data'][
                'Lower break invert station']
        slope = 0.0
        if run != 0.0:
            slope = rise / run
        return slope

    def get_length(self):
        """Computes and returns the slope of the final barrel.

        Returns:
            slope (float): the slope of the final barrel.
        """
        run = self.input_dict['calc_data']['Outlet invert station'] - self.input_dict['calc_data'][
            'Inlet invert station']
        if self.input_dict['calc_data']['Culvert type'] == 'single broken-back':
            run = self.input_dict['calc_data']['Outlet invert station'] - self.input_dict['calc_data'][
                'Break invert station']
        elif self.input_dict['calc_data']['Culvert type'] == 'double broken-back':
            run = self.input_dict['calc_data']['Outlet invert station'] - self.input_dict['calc_data'][
                'Lower break invert station']
        return run
