"""Class to retrieve all the data needed to export a fort.13 (standalone or full simulation export)."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util


# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog


class Fort20DataGetter:
    """Class to retrieve all the data needed to export a fort.20 (standalone or full simulation export)."""
    def __init__(self, query, xms_data):
        """Constructor.

        Args:
            query (:obj:`Query`): Object for requesting data from XMS
            xms_data (:obj:`Optional[dict]`): Dict of XMS data to fill
        """
        self._query = query
        self._xms_data = xms_data

    def _retrieve_bc_data(self):
        """Retrieve the simulation data if this is not a full simulation export (haven't already got it)."""
        if 'sim_data' not in self._xms_data:
            try:
                # Get the name of the mesh and it's number of nodes. If we are executing this code, it implies from the
                # Partial Export simulation component command.
                sim_uuid = self._query.parent_item_uuid()
                sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
                bc_comps = tree_util.descendants_of_type(
                    sim_item, unique_name='Mapped_Bc_Component', recurse=False
                )
                self._xms_data['bc_datas'] = [bc_comp.main_file for bc_comp in bc_comps]
            except Exception:
                raise RuntimeError('Unable to retrieve ADCIRC boundary condition data from SMS.')

    def retrieve_data(self):
        """Retrieve data required to export fort.13 from XMS."""
        # Get the simulation data and domain mesh attributes
        XmLog().instance.info('Retrieving ADCIRC boundary condition data from SMS...')
        self._xms_data['bc_datas'] = []
        self._retrieve_bc_data()
