"""Reader for ADCIRC fort.22 wind input files."""

# 1. Standard Python modules
import datetime
import math
from operator import attrgetter
import os
import shlex
import sys
import tempfile
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.constraint import RectilinearGridBuilder
from xms.coverage.windCoverage import (
    BASIN_CARDS_enum, DEPTH_CARDS_enum, DEV_CARDS_enum, RADIUS_CARDS_enum, SUBREGION_CARDS_enum, WIND_CARDS_enum,
    WindCoverage, WindModel_enum, WindNode
)
from xms.data_objects.parameters import Arc, Coverage, Point
from xms.data_objects.parameters import datetime_to_julian, Projection, UGrid
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules
from xms.adcirc.data.adcirc_data import file_exists
from xms.adcirc.feedback.xmlog import XmLog


def convert_magnitude_direction(mag, direction, meteorologic):
    """Compute the X and Y component using the magnitute and direction read.

    Args:
       mag (:obj:`float`): Magnitude
       direction (:obj:`float`): Direction, ex:  float(line[lon])
       meteorologic (:obj:`bool`):  True if in Meteorological, False if Oceanigraphic

    Returns:
       The x and y components of a_mag and direction
    """
    if meteorologic:
        # Convert the direction from Meteorological to Cartesian convention.
        dir_angle = math.radians(270.0 - direction)
    else:
        # Convert the direction from Oceanagraphic to Cartesian convention.
        dir_angle = math.radians(90.0 - direction)
    # Compute the y component using the previously stored magnitude and the direction we just read.
    mag_y = mag * math.sin(dir_angle)
    # Compute the x component using the previously stored magnitude and the direction we just read.
    mag_x = mag * math.cos(dir_angle)
    return mag_x, mag_y


def get_basin_type(text):
    """Convert a two letter basin code from the fort.22 into a windCoverage.BASIN_CARDS_enum value.

    Args
        text (:obj:`str`): Basin code as a text string read from the fort.22. Should be in all caps.

    Returns:
        windCoverage.BASIN_CARDS_enum - The enum value corresponding to text
    """
    if text == "WP":
        return BASIN_CARDS_enum.BASIN_NORTH_PACIFIC
    elif text == "IO":
        return BASIN_CARDS_enum.BASIN_NORTH_INDIAN_OCEAN
    elif text == "SH":
        return BASIN_CARDS_enum.BASIN_SOUTHERN_HEMISPHERE
    elif text == "CP":
        return BASIN_CARDS_enum.BASIN_CENTRAL_PACIFIC
    elif text == "EP":
        return BASIN_CARDS_enum.BASIN_EASTERN_PACIFIC
    elif text == "AL":
        return BASIN_CARDS_enum.BASIN_ATLANTIC
    else:
        return BASIN_CARDS_enum.BASIN_END


def get_subregion_type(text):
    """Convert a one letter subregion code from the fort.22 into a windCoverage.SUBREGION_CARDS_enum value.

    Args
        text (:obj:`str`): Subregion code as a text string read from the fort.22. Should be a capital letter.

    Returns:
        windCoverage.SUBREGION_CARDS_enum - The enum value corresponding to text
    """
    if text == "A":
        return SUBREGION_CARDS_enum.SUBREGION_ARABIAN_SEA
    elif text == "B":
        return SUBREGION_CARDS_enum.SUBREGION_BAY_OF_BENGAL
    elif text == "C":
        return SUBREGION_CARDS_enum.SUBREGION_CENTRAL_PACIFIC
    elif text == "E":
        return SUBREGION_CARDS_enum.SUBREGION_EASTERN_PACIFIC
    elif text == "L":
        return SUBREGION_CARDS_enum.SUBREGION_ATLANTIC
    elif text == "P":
        return SUBREGION_CARDS_enum.SUBREGION_SOUTH_PACIFIC
    elif text == "Q":
        return SUBREGION_CARDS_enum.SUBREGION_SOUTH_INDIAN_OCEAN
    elif text == "S":
        return SUBREGION_CARDS_enum.SUBREGION_WESTERN_PACIFIC
    else:
        return SUBREGION_CARDS_enum.SUBREGION_END


def get_development_type(text):
    """Convert a two letter development code from the fort.22 into a windCoverage.DEV_CARDS_enum value.

    Args
        text (:obj:`str`): Development code as a text string read from the fort.22. Should be in all caps.

    Returns:
        windCoverage.DEV_CARDS_enum - The enum value corresponding to text
    """
    if text == "DB":
        return DEV_CARDS_enum.DEV_DISTURBANCE
    elif text == "TD":
        return DEV_CARDS_enum.DEV_TROPICAL_DEPRESSION
    elif text == "TS":
        return DEV_CARDS_enum.DEV_TROPICAL_STORM
    elif text == "TY":
        return DEV_CARDS_enum.DEV_TYPHOON
    elif text == "ST":
        return DEV_CARDS_enum.DEV_SUPER_TYPHOON
    elif text == "TC":
        return DEV_CARDS_enum.DEV_TROPICAL_CYCLONE
    elif text == "HU":
        return DEV_CARDS_enum.DEV_HURRICANE
    elif text == "SD":
        return DEV_CARDS_enum.DEV_SUBTROPICAL_DEPRESSION
    elif text == "SS":
        return DEV_CARDS_enum.DEV_SUBTROPICAL_STORM
    elif text == "EX":
        return DEV_CARDS_enum.DEV_EXTRATROPICAL_SYSTEMS
    elif text == "IN":
        return DEV_CARDS_enum.DEV_INLAND
    elif text == "DS":
        return DEV_CARDS_enum.DEV_DISSIPATING
    elif text == "LO":
        return DEV_CARDS_enum.DEV_LOW
    elif text == "WV":
        return DEV_CARDS_enum.DEV_TROPICAL_WAVE
    elif text == "ET":
        return DEV_CARDS_enum.DEV_EXTRAPOLATED
    else:
        return DEV_CARDS_enum.DEV_UNKNOWN


def get_radii_type(radii):
    """Convert a two digit integer radii code from the fort.22 into a windCoverage.WIND_CARDS_enum value.

    Args
        radii (:obj:`int`): Radii code as an integer read from the fort.22.

    Returns:
        windCoverage.WIND_CARDS_enum - The enum value corresponding to radii
    """
    if radii == 34:
        return WIND_CARDS_enum.WIND_34
    elif radii == 50:
        return WIND_CARDS_enum.WIND_50
    elif radii == 64:
        return WIND_CARDS_enum.WIND_64
    else:
        return WIND_CARDS_enum.WIND_100


def get_windcode_type(text):
    """Convert a three letter wind radius code from the fort.22 into a windCoverage.RADIUS_CARDS_enum value.

    Args
        text (:obj:`str`): Wind radius code as a text string read from the fort.22. Should be in all caps.

    Returns:
        windCoverage.RADIUS_CARDS_enum - The enum value corresponding to text
    """
    if text == "AAA":
        return RADIUS_CARDS_enum.RADIUS_FULL_CIRCLE
    elif text == "NNS":
        return RADIUS_CARDS_enum.RADIUS_NORTH_SEMICIRCLE
    elif text == "NES":
        return RADIUS_CARDS_enum.RADIUS_NORTHEAST_SEMICIRCLE
    elif text == "EES":
        return RADIUS_CARDS_enum.RADIUS_EAST_SEMICIRCLE
    elif text == "SES":
        return RADIUS_CARDS_enum.RADIUS_SOUTHEAST_SEMICIRCLE
    elif text == "SSS":
        return RADIUS_CARDS_enum.RADIUS_SOUTH_SEMICIRCLE
    elif text == "SWS":
        return RADIUS_CARDS_enum.RADIUS_SOUTHWEST_SEMICIRCLE
    elif text == "WWS":
        return RADIUS_CARDS_enum.RADIUS_WEST_SEMICIRCLE
    elif text == "NWS":
        return RADIUS_CARDS_enum.RADIUS_NORTHWEST_SEMICIRCLE
    elif text == "NNQ":
        return RADIUS_CARDS_enum.RADIUS_NORTH_QUAD
    elif text == "NEQ":
        return RADIUS_CARDS_enum.RADIUS_NORTHEAST_QUAD
    elif text == "EEQ":
        return RADIUS_CARDS_enum.RADIUS_EAST_QUAD
    elif text == "SEQ":
        return RADIUS_CARDS_enum.RADIUS_SOUTHEAST_QUAD
    elif text == "SSQ":
        return RADIUS_CARDS_enum.RADIUS_SOUTH_QUAD
    elif text == "SWQ":
        return RADIUS_CARDS_enum.RADIUS_SOUTHWEST_QUAD
    elif text == "WWQ":
        return RADIUS_CARDS_enum.RADIUS_WEST_QUAD
    elif text == "NWQ":
        return RADIUS_CARDS_enum.RADIUS_NORTHWEST_QUAD
    else:
        return RADIUS_CARDS_enum.RADIUS_END


def get_depth_code(text):
    """Convert a one letter depth code from the fort.22 into a windCoverage.DEPTH_CARDS_enum value.

    Args
        text (:obj:`str`): Depth code as a text string read from the fort.22. Should be a capital letter.

    Returns:
        windCoverage.DEPTH_CARDS_enum - The enum value corresponding to text
    """
    if text == "D":
        return DEPTH_CARDS_enum.SYSDEPTH_DEEP
    elif text == "M":
        return DEPTH_CARDS_enum.SYSDEPTH_MEDIUM
    elif text == "S":
        return DEPTH_CARDS_enum.SYSDEPTH_SHALLOW
    else:
        return DEPTH_CARDS_enum.SYSDEPTH_UNKNOWN


class Fort22Reader:
    """Reads an ADCIRC fort.22 file. Meteorological forcing data."""
    def __init__(self, filename, query, datafromthe15, sim_data, radstress=False):
        """Initializes the reader. The fort.22 contains meteorological forcing data.

        Args:
            filename (:obj:`str`): Full path and _filename of the fort.22 file.
            query (:obj:`xms.api.dmi.Query`): Query for communicating with SMS. Only used here for getting temp files
                to write datasets to.
            datafromthe15 (:obj:`DataForThe22`): Data read from the fort.15 needed to read the fort.22
            sim_data (:obj:`SimData`): Data class of the simulation component. Will update dataset selector UUIDs if
                applicable.
            radstress (:obj:`bool`): Flag for reading the fort.23 file (no scalar data)
        """
        self._filename = filename
        self._query = query
        self._15data = datafromthe15
        self._grid_uuid = self._15data.grid_uuid if self._15data.grid_uuid else str(uuid.uuid4())
        self._cov_uuid = self._15data.cov_uuid if self._15data.cov_uuid else str(uuid.uuid4())
        self._sim_data = sim_data
        self._nws_simple = 0
        self._radstress = radstress
        self._datatypes = None
        self._temp_dir = self._query.xms_temp_directory if self._query else self._15data.temp_dir
        if not self._temp_dir:
            self._temp_dir = tempfile.gettempdir()
        self.built_data = {}
        self._init_atcf_data_types()

    def _init_atcf_data_types(self):
        """Initialize the array of data types for columns of an ATCF file if NWS=8,19,20."""
        self.nws_simple = abs(self._15data.nws_type) % 100
        if self.nws_simple == 8:
            self._datatypes = [
                2, 0, 2, 0, 2, 0, 2, 2, 0, 0, 2, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 2, 2, 0, 2, 0, 0, 0, 0
            ]
        elif self.nws_simple in [19, 20]:
            self._datatypes = [
                2, 0, 2, 0, 2, 0, 2, 2, 0, 0, 2, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 2, 2, 0, 0, 0, 0, 0, 1,
                1, 1, 1, 1
            ]
            if self.nws_simple == 20:
                self._datatypes.extend([1, 1, 1, 1, 1, 1, 1, 1])

    def _get_next_dset_uuid(self):
        """Assign hardcoded UUIDs to datasets if testing."""
        if self._15data.dset_uuids:
            return self._15data.dset_uuids.pop()
        return str(uuid.uuid4())

    def _find_wind_file(self, filename, workdir):
        """Determine if an absolute or relative file exists, and log a warning if it does not.

        Args:
            filename (:obj:`str`): Relative or absolute path to the file
            workdir (:obj:`str`): The directory paths could be relative to

        Returns:
            (:obj:`str`): The normalized absolute file path if found, or empty string if not
        """
        if os.path.exists(filename):  # Given as an absolute path or relative from current working directory
            if not os.path.isabs(filename):
                # At this point, if we have a valid relative path it must be from the current working directory
                return os.path.normpath(os.path.join(os.getcwd(), filename))
            return os.path.normpath(filename)
        else:  # Try a relative path reference
            abs_filename = os.path.join(workdir, filename)
            if os.path.isfile(abs_filename):
                return os.path.normpath(abs_filename)
            else:  # Couldn't find it
                file_exists(filename)  # Warns user
                return ''

    def _get_storm_track_values(self, line):
        """Parse the parameter values from a line in a storm track fort.22 file.

        The storm track files are very sparse, so this function makes parsing of the lines a little more robust.

        Args:
            line (:obj:`list[str]`): One line of input that has been split on commas.

        Returns:
            (:obj:`list`): The parsed values with nonexistent values filled in with defaults
        """
        default_vals = [0, 0.0, '']
        parsed_values = [default_vals[dtype] for dtype in self._datatypes]
        for i in range(len(self._datatypes)):
            try:
                if self._datatypes[i] == 0:
                    parsed_values[i] = int(line[i].strip())
                elif self._datatypes[i] == 1:
                    parsed_values[i] = float(line[i].strip())
                else:
                    parsed_values[i] = line[i].strip()
            except (TypeError, IndexError, ValueError):  # Values can be sparse - whitespace between commas
                pass
        return parsed_values

    def _build_wind_grid(self):
        """Build the wind grid from the definition passed by the fort.15 reader.

        ADCIRC origin is in the middle of the top left cell. SMS origin is the bottom left corner of the grid.
        """
        rect_builder = RectilinearGridBuilder()
        rect_builder.is_2d_grid = True

        # Translate the origin.
        originx = self._15data.lonmin  # adjust to the edge of the cell
        originy = self._15data.latmax
        originy -= self._15data.latinc * self._15data.nwlat
        rect_builder.origin = (originx, originy, 0.0)

        # Compute base grid x and y locations.
        rect_builder.locations_x = [self._15data.loninc * i for i in range(self._15data.nwlon + 1)]
        rect_builder.locations_y = [self._15data.latinc * i for i in range(self._15data.nwlat + 1)]

        # Create the CoGrid file.
        co_grid = rect_builder.build_grid()
        co_grid_file = os.path.join(self._temp_dir, f'{self._grid_uuid}.xmc')
        # Tests fail on CI if binary.
        binary = True if os.environ.get(XmEnv.ENVIRON_RUNNING_TESTS) != 'TRUE' else False
        co_grid.write_to_file(co_grid_file, binary_arrays=binary)

        do_grid = UGrid(co_grid_file, uuid=self._grid_uuid, name='Wind Grid')
        if self._15data.projection:
            proj = self._15data.projection
        else:
            proj = Projection(system='GEOGRAPHIC', vertical_units='METERS')
        do_grid.projection = proj

        self.built_data['wind_grid'] = do_grid

    def _read_nws_dataset(self):
        """Read a fort.22 that is in NWS 1, 2, -2, 4, -4, 5, or -5 format."""
        # Setup the dataset builders.
        vector_uuid = self._get_next_dset_uuid()
        scalar_uuid = self._get_next_dset_uuid()
        if self._radstress:
            vec_dset_name = 'Radstress'
        else:
            vec_dset_name = 'Wind velocity' if self.nws_simple in [4, 5] else 'Wind stress'
        vector_builder = DatasetWriter(
            h5_filename=os.path.join(self._temp_dir, f'{vector_uuid}.h5'),
            name=vec_dset_name,
            dset_uuid=vector_uuid,
            geom_uuid=self._15data.mesh_uuid,
            num_components=2,
            ref_time=self._15data.reftime if self._15data.reftime else None,
            time_units='Seconds'
        )
        scalar_builder = None
        if not self._radstress:
            scalar_builder = DatasetWriter(
                h5_filename=os.path.join(self._temp_dir, f'{scalar_uuid}.h5'),
                name='Atmospheric pressure',
                dset_uuid=scalar_uuid,
                geom_uuid=self._15data.mesh_uuid,
                num_components=1,
                ref_time=self._15data.reftime if self._15data.reftime else None,
                time_units='Seconds'
            )

        # Parse the ASCII file (fort.22 or fort.23)
        cur_time = 0.0
        first_node = None
        timestep_vals_stress = [[0.0, 0.0] for _ in range(self._15data.numnodes)]
        timestep_vals_pressure = []
        if not self._radstress:
            timestep_vals_pressure = [0.0 for _ in range(self._15data.numnodes)]
        first_ts = True
        with open(self._filename, 'r', buffering=100000) as f:
            line = f.readline()
            while line and line.strip():
                line = line.replace(",", "")
                line = line.strip()
                if not line:
                    line = f.readline()
                    continue  # skip blank lines
                line = line.split()
                finishtimestep = False
                # look for the end of a sparse timestep
                if line[0] == '#':
                    line = f.readline()
                    line = line.replace(",", "")
                    line = line.split()
                    if not first_ts:
                        finishtimestep = True
                    else:
                        first_ts = False
                elif not first_node:
                    first_node = 1
                else:
                    nodeid = int(line[0])
                    if nodeid == first_node:
                        finishtimestep = True

                nodeid = int(line[0])

                if finishtimestep:
                    # append the timestep on the two datasets
                    vector_builder.append_timestep(cur_time, timestep_vals_stress)
                    if scalar_builder:
                        scalar_builder.append_timestep(cur_time, timestep_vals_pressure)
                    # reset the values to get ready to read the next timestep
                    timestep_vals_stress = [[0.0, 0.0] for _ in range(self._15data.numnodes)]
                    if scalar_builder:
                        timestep_vals_pressure = [0.0 for _ in range(self._15data.numnodes)]
                    cur_time += self._15data.wtiminc
                if len(line) > 2:
                    timestep_vals_stress[nodeid - 1][0] = float(line[1])
                    timestep_vals_stress[nodeid - 1][1] = float(line[2])
                    if scalar_builder:
                        timestep_vals_pressure[nodeid - 1] = float(line[3])
                line = f.readline()

        # add the last timestep
        vector_builder.append_timestep(cur_time, timestep_vals_stress)
        vector_builder.appending_finished()
        if scalar_builder:
            scalar_builder.append_timestep(cur_time, timestep_vals_pressure)
            scalar_builder.appending_finished()

        # Link up dataset selectors in Model Control dialog
        if not self._radstress:
            # Build the data_objects Datasets to send to XMS
            dsetlist = self.built_data.setdefault('wind_dsets', [])
            dsetlist.extend([vector_builder, scalar_builder])
            self._sim_data.wind.attrs['mesh_wind'] = vector_uuid
            self._sim_data.wind.attrs['mesh_pressure'] = scalar_uuid
        else:
            # Build the data_objects Datasets to send to XMS
            dsetlist = self.built_data.setdefault('wind_dsets', [])
            dsetlist.append(vector_builder)
            self._sim_data.wind.attrs['fort23_uuid'] = vector_uuid

    def _read_nws_3(self):
        """Read a fort.22 that is in NWS3 format (gridded wind data)."""
        self._build_wind_grid()  # Build the wind grid from the definition in the fort.15
        # Setup the dataset builder
        vector_uuid = self._get_next_dset_uuid()
        vector_builder = DatasetWriter(
            h5_filename=os.path.join(self._temp_dir, f'{vector_uuid}.h5'),
            name='Wind velocity',
            dset_uuid=vector_uuid,
            geom_uuid=self._grid_uuid,
            num_components=2,
            ref_time=self._15data.reftime if self._15data.reftime else None,
            time_units='Seconds',
            location='cells'
        )

        # Parse the ASCII file
        cur_time = 0.0
        with open(self._filename, 'r', buffering=100000) as f:
            line = f.readline()  # first timestep time - ignore and calculate from wtiminc and reftime
            while line and line.strip():
                vector_builder.append_timestep(cur_time, self._read_nws3_timestep(f))
                cur_time += self._15data.wtiminc
                line = f.readline()
        vector_builder.appending_finished()

        # Create the data_objects Dataset to send to SMS.
        dsetlist = self.built_data.setdefault('wind_dsets', [])
        dsetlist.append(vector_builder)

        # Link up dataset selector in Model Control dialog
        self._sim_data.wind.attrs['grid_wind'] = vector_uuid

    def _read_nws3_timestep(self, fs):
        """Read a timestep in NWS3 format.

        For each latitude of the grid (starting at the top) and then each longitude (starting at the left)
        the magnitude for each cell is written. Then the directions are written in the same order after all
        the magnitudes.

        Args:
            fs: Open file handle to the fort.22. Next line in the buffer should be the first line of data
                for the timestep
        """
        ts_vals = [[0.0, 0.0] for _ in range(self._15data.nwlon * self._15data.nwlat)]
        # Read the magnitudes
        for lat in range(self._15data.nwlat - 1, -1, -1):
            line = fs.readline().split()
            for lon in range(self._15data.nwlon):
                idx = (lat * self._15data.nwlon) + lon
                ts_vals[idx][0] = float(line[lon])  # store the magnitude for now
        # Read the directions
        for lat in range(self._15data.nwlat - 1, -1, -1):
            line = fs.readline().split()
            for lon in range(self._15data.nwlon):
                idx = (lat * self._15data.nwlon) + lon
                ts_vals[idx][0], ts_vals[idx][1] = convert_magnitude_direction(ts_vals[idx][0], float(line[lon]), True)
        return ts_vals

    def _read_nws_6(self):
        """Read a fort.22 that is in NWS6 or NWS7 format (gridded wind data)."""
        self._build_wind_grid()  # Build the wind grid from the definition in the fort.15
        # Setup the dataset builders.
        vector_uuid = self._get_next_dset_uuid()
        scalar_uuid = self._get_next_dset_uuid()
        vector_builder = DatasetWriter(
            h5_filename=os.path.join(self._temp_dir, f'{vector_uuid}.h5'),
            name='Wind velocity' if self.nws_simple == 6 else 'Surface stress',
            dset_uuid=vector_uuid,
            geom_uuid=self._grid_uuid,
            num_components=2,
            ref_time=self._15data.reftime if self._15data.reftime else None,
            time_units='Seconds',
            location='cells'
        )
        scalar_builder = DatasetWriter(
            h5_filename=os.path.join(self._temp_dir, f'{scalar_uuid}.h5'),
            name='Atmospheric pressure',
            dset_uuid=scalar_uuid,
            geom_uuid=self._grid_uuid,
            num_components=1,
            ref_time=self._15data.reftime if self._15data.reftime else None,
            time_units='Seconds',
            location='cells'
        )

        # Parse the ASCII file.
        cur_time = 0.0
        with open(self._filename, 'r', buffering=100000) as f:
            line = f.readline().split()
            while line:
                vector_data, scalar_data = self._read_nws_6_timestep(f, line)
                scalar_builder.append_timestep(cur_time, scalar_data)
                vector_builder.append_timestep(cur_time, vector_data)
                cur_time += self._15data.wtiminc
                line = f.readline().split()
        scalar_builder.appending_finished()
        vector_builder.appending_finished()

        # Create the data_object Datasets to send to SMS.
        dsetlist = self.built_data.setdefault('wind_dsets', [])
        dsetlist.extend([vector_builder, scalar_builder])
        # Link up dataset selectors in Model Control dialog
        self._sim_data.wind.attrs['grid_wind'] = vector_uuid
        self._sim_data.wind.attrs['grid_pressure'] = scalar_uuid

    def _read_nws_6_timestep(self, fs, line):
        """Read a timestep in NWS6 or NWS7 format.

        Args:
            fs: Open file handle to the fort.22. Next line in the buffer should be the second line of data
                for the timestep
            line (:obj:`list`): The first line of data from this timestep split on whitespace
        """
        # Check for the old format where pressure is in the first column
        x_col = 0
        y_col = 1
        press_col = 2
        if float(line[0]) > 10000:
            press_col = 0
            x_col = 1
            y_col = 2
        vec_vals = [[0.0, 0.0] for _ in range(self._15data.nwlon * self._15data.nwlat)]
        pressure_vals = [0.0 for _ in range(self._15data.nwlon * self._15data.nwlat)]
        # Read the values
        first_time = True
        for lat in range(self._15data.nwlat - 1, -1, -1):
            for lon in range(self._15data.nwlon):
                if not first_time:
                    line = fs.readline().split()
                else:
                    first_time = False
                idx = (lat * self._15data.nwlon) + lon
                vec_vals[idx][0] = float(line[x_col])
                vec_vals[idx][1] = float(line[y_col])
                pressure_vals[idx] = float(line[press_col])
        return vec_vals, pressure_vals

    def _read_nws_storm_track(self):
        """Read a NWS=8,19,21 fort.22 file in ATCF Best Track/Objective Aid/Wind Radii format.

        See https://www.nrlmry.navy.mil/atcf_web/docs/database/new/abrdeck.html
        """
        wind_cov = WindCoverage()
        wind_cov.m_cov = Coverage()
        wind_cov.m_cov.uuid = self._cov_uuid

        if self.nws_simple == 8:
            wind_cov.m_windModel = WindModel_enum.WM_HOLLAND_SYMMETRIC
        else:
            wind_cov.m_windModel = WindModel_enum.WM_HOLLAND_ASYMMETRIC

        cov_pts = {}  # key=(str(ptx), str(pty)) value=coverage point
        node_id = 1
        with open(self._filename, 'r', buffering=100000) as f:
            for line in f:
                line = line.split(',')
                if not line:
                    continue
                line = self._get_storm_track_values(line)

                # Read the coverage level attributes
                if node_id == 1:
                    wind_cov.m_basin = get_basin_type(line[0].upper())  # Basin type
                    if len(line[22]) > 0:
                        wind_cov.m_subregion = get_subregion_type(line[22].upper())
                    wind_cov.m_cycloneNumber = int(line[1])

                # Each line is a point in the storm track coverage.
                # Create the coverage geometry point.
                pty = line[6]  # Parse the latitude
                if pty.upper().endswith("N"):
                    pty = float(pty[:len(pty) - 1])
                else:
                    pty = float(pty[:len(pty) - 1]) * -1.0
                pty /= 10.0  # gets written as tenths of degrees
                ptx = line[7]  # Parse the longitude
                if ptx.upper().endswith("E"):
                    ptx = float(ptx[:len(ptx) - 1])
                else:
                    ptx = float(ptx[:len(ptx) - 1]) * -1.0
                ptx /= 10.0  # gets written as tenths of degrees
                if (str(ptx), str(pty)) not in cov_pts:  # check if we have created a point for this location yet
                    # Create the coverage point
                    cov_pt = Point(ptx, pty, feature_id=node_id)
                    cov_pts[(str(ptx), str(pty))] = cov_pt
                    # Create the nodal attributes.
                    wind_node = WindNode()
                    wind_node.m_id = node_id
                    wind_cov.m_nodeWind.append(wind_node)
                    node_id += 1  # increment node id for the next new point encountered
                node_date = line[2]
                node_year = int(node_date[:4])
                node_month = int(node_date[4:6])
                node_day = int(node_date[6:8])
                node_hr = int(node_date[8:])
                dt = datetime.datetime(year=node_year, month=node_month, day=node_day, hour=node_hr)
                wind_cov.m_nodeWind[-1].m_date = datetime_to_julian(dt)
                wind_cov.m_nodeWind[-1].m_iTechNumOrMinutes = int(line[3])
                wind_cov.m_nodeWind[-1].m_sTech = line[4]
                # wind_node.m_tau = line[5]  # We do not care about Tau
                wind_cov.m_nodeWind[-1].m_iVMax = int(line[8])
                wind_cov.m_nodeWind[-1].m_iMinSeaLevelPressure = int(line[9])
                wind_cov.m_nodeWind[-1].m_eLevelOfDevelopment = get_development_type(line[10].upper())
                radii_key = get_radii_type(int(line[11]))
                wind_cov.m_nodeWind[-1].m_eWindRadiusCode = get_windcode_type(line[12].upper())
                wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[0].m_radius = int(line[13])
                wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[1].m_radius = int(line[14])
                wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[2].m_radius = int(line[15])
                wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[3].m_radius = int(line[16])
                if self.nws_simple == 8:
                    for radii in wind_cov.m_nodeWind[-1].m_radii:
                        if radii != radii_key.value:
                            for i in range(4):
                                wind_cov.m_nodeWind[-1].m_radii[radii].m_data[i].m_radius = -1
                wind_cov.m_nodeWind[-1].m_iPressure = int(line[17])
                wind_cov.m_nodeWind[-1].m_iPressureRadius = int(line[18])
                wind_cov.m_nodeWind[-1].m_iMaxWindRadius = int(line[19])
                wind_cov.m_nodeWind[-1].m_iGusts = int(line[20])
                wind_cov.m_nodeWind[-1].m_iEyeDiameter = int(line[21])
                wind_cov.m_nodeWind[-1].m_iMaxSeas = int(line[23])
                wind_cov.m_nodeWind[-1].m_sInitials = line[24]
                wind_cov.m_nodeWind[-1].m_iStormDirection = int(line[25])
                wind_cov.m_nodeWind[-1].m_iStormSpeed = int(line[26])
                wind_cov.m_nodeWind[-1].m_sStormName = line[27]
                if self.nws_simple == 8:
                    wind_cov.m_nodeWind[-1].m_eDepth = get_depth_code(line[28].upper())
                    wind_cov.m_nodeWind[-1].m_iWaveHeight = int(line[29])
                    wind_cov.m_nodeWind[-1].m_eSeasRadiusCode = get_windcode_type(line[30].upper())
                    wind_cov.m_nodeWind[-1].m_iSeas1 = int(line[31])
                    wind_cov.m_nodeWind[-1].m_iSeas2 = int(line[32])
                    wind_cov.m_nodeWind[-1].m_iSeas3 = int(line[33])
                    wind_cov.m_nodeWind[-1].m_iSeas4 = int(line[34])
                else:
                    # If a line appears in the file for this isotach, it is defined even if none of the radii
                    # are enabled.
                    wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_defined = True
                    # Read max values for all radii even if the radius is disabled. Preserves all data in the file.
                    if int(line[30]) == 1:
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[0].m_on = True
                    wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[0].m_rMax = float(line[34])
                    if int(line[31]) == 1:
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[1].m_on = True
                    wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[1].m_rMax = float(line[35])
                    if int(line[32]) == 1:
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[2].m_on = True
                    wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[2].m_rMax = float(line[36])
                    if int(line[33]) == 1:
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[3].m_on = True
                    wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[3].m_rMax = float(line[37])
                    wind_cov.m_nodeWind[-1].m_hollandB = float(line[38])
                    if self.nws_simple == 20:
                        # Read Holland B and max values for all radii even if the radius is disabled. Preserves all
                        # data in the file.
                        # Column 40-43 is the quadrant-varying Holland B parameter
                        # Column 44-47 are the quadrant-varying Vmax calculated at the top of the planetary boundary
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[0].m_hollandB = float(line[39])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[0].m_vMax = float(line[43])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[1].m_hollandB = float(line[40])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[1].m_vMax = float(line[44])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[2].m_hollandB = float(line[41])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[2].m_vMax = float(line[45])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[3].m_hollandB = float(line[42])
                        wind_cov.m_nodeWind[-1].m_radii[radii_key.value].m_data[3].m_vMax = float(line[46])

        # Convert point map to a list
        cov_pts_lst = [pt for loc, pt in cov_pts.items()]
        # Sort the points by id
        cov_pts_lst.sort(key=attrgetter('id'))
        cov_arcs = []
        # Create the coverage arcs
        for i in range(len(cov_pts_lst)):
            if i > 0:
                cov_arc = Arc(feature_id=i, start_node=cov_pts_lst[i - 1], end_node=cov_pts_lst[i])
                cov_arcs.append(cov_arc)
        wind_cov.m_cov.arcs = cov_arcs
        wind_cov.m_cov.name = 'Storm Track'
        if self._15data.projection:
            wind_cov.m_cov.projection = self._15data.projection
        wind_cov.m_cov.complete()
        self.built_data['wind_cov'] = wind_cov

    def _read_nws_10(self):
        """Nothing to do for now. May change interface later."""
        return

    def _read_nws_11(self):
        """Nothing to do for now. May change interface later."""
        return

    def _read_nws_12(self):
        """Read an NWS=12 fort.22 file (OWI meteorological files)."""
        num_header_lines = 3
        pressure_filenames = []
        wind_filenames = []
        with open(self._filename, 'r') as f:
            lines = f.readlines()

        if len(lines) < num_header_lines:
            XmLog().instance.error('Invalid fort.22 file for NWS=12. Wind data will not be read.')
            return

        num_sets = int(lines[0].split()[0])  # First line is number of file pairs
        self._sim_data.wind.attrs['NWBS'] = int(lines[1].split()[0])  # Second line is NWBS
        self._sim_data.wind.attrs['DWM'] = float(lines[2].split()[0])  # Third line is DWM

        fort22_dir = os.path.dirname(self._filename)
        if not fort22_dir:  # Path is the current working directory if we were passed a relative fort.22 file path.
            fort22_dir = os.getcwd()
        if num_sets > 0:  # Use hard-coded ADCIRC filenames if number of sets is positive.
            pressure_filenames = [os.path.join(fort22_dir, 'fort.221')]
            wind_filenames = [os.path.join(fort22_dir, 'fort.222')]
            if num_sets > 1:
                pressure_filenames.append(self._find_wind_file('fort.223', fort22_dir))
                wind_filenames.append(self._find_wind_file('fort.224', fort22_dir))
                if num_sets > 2:
                    pressure_filenames.append(self._find_wind_file('fort.217', fort22_dir))
                    wind_filenames.append(self._find_wind_file('fort.218', fort22_dir))
        else:  # Number of sets is negative, read filename pairs from the rest of the file.
            for i in range(num_header_lines, len(lines)):
                if not lines[i].startswith('#'):  # skip comment lines
                    lexer = shlex.shlex(lines[i], punctuation_chars=True)
                    lexer.whitespace += ','
                    split_line = list(lexer)
                    if len(split_line) > 1:
                        pressure_file = self._find_wind_file(split_line[0].strip('"'), fort22_dir)
                        pressure_filenames.append(pressure_file)
                        wind_file = self._find_wind_file(split_line[1].strip('"'), fort22_dir)
                        wind_filenames.append(wind_file)

        nws12_files = {
            'Pressure File': xr.DataArray(data=np.array(pressure_filenames, dtype=object)),
            'Wind File': xr.DataArray(data=np.array(wind_filenames, dtype=object)),
        }
        self._sim_data.nws12_files = xr.Dataset(data_vars=nws12_files)

    def _read_nws_15(self):
        """Read an NWS=15 fort.22 file (HWND meteorological files)."""
        file_dir = os.path.dirname(self._filename)
        if not file_dir:  # Path is the current working directory if we were passed a relative fort.22 file path.
            file_dir = os.getcwd()

        with open(self._filename, 'r', buffering=100000) as f:
            is_4_col = False
            f.readline()  # don't care about comment line
            self._sim_data.wind.attrs['DWM'] = float(f.readline().split()[0])
            self._sim_data.wind.attrs['wind_pressure'] = f.readline().split()[0]
            if self._sim_data.wind.attrs['wind_pressure'] == 'specifiedPc':  # Following lines have extra pressure value
                is_4_col = True

            # Build up a Dataset for the file table.
            hours = []
            pressures = []
            ramp_mults = []
            filenames = []
            line = f.readline()
            while line and line.strip():
                split_line = shlex.split(line)
                if len(split_line) >= 4:
                    hours.append(float(split_line[0]))
                    # Set pressure to -1 to emphasize value is ignored by ADCIRC if not specifying central pressure.
                    pressures.append(-1.0 if not is_4_col else float(split_line[1]))
                    ramp_mults.append(float(split_line[2]))
                    filename = self._find_wind_file(split_line[3].replace('"', ''), file_dir)
                    filenames.append(filename)
                line = f.readline()

        # Set the file Dataset on the simulation component's data
        nws15_files = {
            'Hours': xr.DataArray(data=np.array(hours, dtype=np.float64)),
            'Central Pressure': xr.DataArray(data=np.array(pressures, dtype=np.float64)),
            'Ramp Multiplier': xr.DataArray(data=np.array(ramp_mults, dtype=np.float64)),
            'HWND File': xr.DataArray(data=np.array(filenames, dtype=object)),
        }
        self._sim_data.nws15_files = xr.Dataset(data_vars=nws15_files)

    def read_nws_16(self):
        """Read an NWS=16 fort.22 file (GFDL meteorological files)."""
        file_dir = os.path.dirname(self._filename)
        if not file_dir:  # Path is the current working directory if we were passed a relative fort.22 file path.
            file_dir = os.getcwd()

        with open(self._filename, 'r', buffering=100000) as f:
            f.readline()  # don't care about comment line
            self._sim_data.wind.attrs['DWM'] = float(f.readline().split()[0])
            self._sim_data.wind.attrs['max_extrap'] = float(f.readline().split()[0])

            # Build up a Dataset for the file table.
            hours = []
            ramp_mults = []
            filenames = []
            line = f.readline()
            while line and line.strip():
                split_line = shlex.split(line)
                if len(split_line) >= 3:
                    hours.append(float(split_line[0]))
                    ramp_mults.append(float(split_line[1]))
                    filename = self._find_wind_file(split_line[2].replace('"', ''), file_dir)
                    filenames.append(filename)
                line = f.readline()

        # Set the file Dataset on the simulation component's data
        nws16_files = {
            'Hours': xr.DataArray(data=np.array(hours, dtype=np.float64)),
            'Ramp Multiplier': xr.DataArray(data=np.array(ramp_mults, dtype=np.float64)),
            'GFDL File': xr.DataArray(data=np.array(filenames, dtype=object)),
        }
        self._sim_data.nws16_files = xr.Dataset(data_vars=nws16_files)

    def read(self):
        """Top-level function that triggers the read of a fort.22 file."""
        if not os.path.isfile(self._filename) or os.path.getsize(self._filename) == 0:  # pragma: no cover
            if self._radstress:
                sys.stderr.write(f"Error reading fort.23: File not found - {self._filename}")
            else:
                sys.stderr.write(f"Error reading fort.22: File not found - {self._filename}")
            return

        if self._radstress:
            XmLog().instance.info('Reading fort.23...', str(__package__))
        else:
            XmLog().instance.info(f'Reading fort.22 in NWS={self.nws_simple} format...', str(__package__))
        if self.nws_simple in [1, 2, 4, 5] or self._radstress:
            self._read_nws_dataset()
        elif self.nws_simple == 3:
            self._read_nws_3()
        elif self.nws_simple in [6, 7]:  # 7 no longer supported, but we can read them
            self._read_nws_6()
        elif self.nws_simple in [8, 19, 20]:
            self._read_nws_storm_track()
        elif self.nws_simple == 10:
            self._read_nws_10()
        elif self.nws_simple == 11:
            self._read_nws_11()
        elif self.nws_simple == 12:
            self._read_nws_12()
        elif self.nws_simple == 15:
            self._read_nws_15()
        elif self.nws_simple == 16:
            self.read_nws_16()

        if self._radstress:
            XmLog().instance.info('Finished reading fort.23.')
        else:
            XmLog().instance.info('Finished reading fort.22.')
