"""A dialog for hydrodynamic material values."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import webbrowser

# 2. Third party modules
from adhparam.material_transport_properties import MaterialTransportProperties
from adhparam.time_series import TimeSeries
import pandas as pd
from PySide2.QtCore import QDir, Qt, Slot
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QDialogButtonBox, QFileDialog, QGroupBox, QHBoxLayout, QLabel, QPushButton, QSplitter,
    QStackedWidget, QTabWidget, QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory

# 4. Local modules
from xms.adh.data.materials_io import MaterialsIO
from xms.adh.data.transport_constituents_io import TransportConstituentsIO
from xms.adh.gui.param_qt_helper import ParamQtHelper
from xms.adh.gui.time_series_editor import TimeSeriesEditor
from xms.adh.gui.widgets.friction_widget import FrictionWidget
from xms.adh.gui.widgets.material_table_widget import MaterialTableWidget
from xms.adh.gui.widgets.transport_constituent_assignment_widget import TransportConstituentAssignmentWidget
from xms.adh.gui.widgets.transport_constituent_mat_assignment_widget import TransportConstituentMatAssignmentWidget


class MaterialDialog(XmsDlg):
    """A dialog for assigning material properties."""

    def __init__(self, win_cont, title, data, pe_tree):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (QWidget): Parent window
            title (str): Window title
            data (MaterialsIO): Materials object
            pe_tree (TreeNode): The project explorer tree.
        """
        super().__init__(win_cont, 'xms.adh.gui.material_dialog')
        self.data = data
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Material_Properties'
        self.param_helper = ParamQtHelper(self)
        self.widgets = dict()
        self.mat_id_to_page = {}

        self._GENERAL_TAB = 0
        self._FRICTION_TAB = 1
        self._TRANSPORT_TAB = 2

        self.setWindowTitle(title)
        self._setup(pe_tree)
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())
        self.last_import_export_path = None
        self.deleted_mat_ids = []

    def _setup(self, pe_tree):
        """Sets up the dialog.

        Args:
            pe_tree (TreeNode): The project explorer tree.
        """
        # Dialog QVBoxLayout with QTabWidget then QDialogButtonBox
        self._set_layout('', 'top_layout', QVBoxLayout())

        # Create the transport constituent widget for selecting the component.
        self.widgets['transport_select'] = TransportConstituentAssignmentWidget(self, pe_tree, True)
        self.widgets['transport_select'].ui.constituents_table.setVisible(False)
        self.widgets['transport_select'].changed_use_constituents[bool].connect(self._use_constituents_changed)
        self.widgets['transport_select'].changed_constituents[TransportConstituentsIO].connect(
            self._set_material_transport
        )
        self.widgets['top_layout'].addWidget(self.widgets['transport_select'])

        # Create a splitter widget.
        self.widgets['main_splitter'] = QSplitter()
        self.widgets['top_layout'].addWidget(self.widgets['main_splitter'])

        # Add a table for the left side of the dialog listing all materials.
        self.widgets['table_view'] = MaterialTableWidget(self, self.data)
        self.widgets['main_splitter'].addWidget(self.widgets['table_view'])

        self.widgets['main_splitter'].setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )

        # Add buttons for the bottom of the dialog.
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['top_layout'].addWidget(self.widgets['btn_box'])

        # Add a stack widget for the right side of the dialog. The stack will show the currently selected material.
        self.widgets['prop_stack'] = QStackedWidget()
        self.widgets['main_splitter'].addWidget(self.widgets['prop_stack'])

        # Setup the button to change all of the material display options at once.
        icon_size = self.widgets['table_view'].toolbar.iconSize()
        dummy_opt = PolygonOptions()
        icon = DisplayOptionIconFactory.get_icon(dummy_opt, icon_size.height())
        self.widgets['table_view'].toolbar.addAction(icon, 'Change all material textures', self.on_btn_change_all)

        # Set up import and export buttons for the material table.
        self.widgets['import_btn'] = QPushButton('Import...')
        self.widgets['export_btn'] = QPushButton('Export...')
        self.widgets['import_btn'].clicked.connect(self.on_btn_import)
        self.widgets['export_btn'].clicked.connect(self.on_btn_export)
        self.widgets['table_view'].toolbar_layout.addWidget(self.widgets['import_btn'])
        self.widgets['table_view'].toolbar_layout.addWidget(self.widgets['export_btn'])

        # Connect material table signals to slots in this class.
        self.widgets['table_view'].added.connect(self._add_material)
        self.widgets['table_view'].deleted.connect(self._delete_materials)
        self.widgets['table_view'].model.itemChanged.connect(self._update_name)

        # Adjust the table size and selection mode.
        self.widgets['table_view'].table_view.setMinimumSize(250, 500)
        selection_model = self.widgets['table_view'].table_view.selectionModel()
        selection_model.currentChanged.connect(self.on_material_changed)

        # QDialogButtonBox with Ok and Cancel buttons
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)

    def _add_prop_page(self, mat_id):
        """Adds a page for a material onto the stacked widget if one does not exist for the material.

        Args:
            mat_id (int): The id of the material that may need a page.
        """
        if mat_id in self.mat_id_to_page:
            return
        tab_widget = QTabWidget()
        tab_widget.setObjectName('prop_tab_widget')
        prop_box, prop_layout = self._add_property_group_widget(mat_id, 'prop_box')
        fric_prop_box, fric_prop_layout = self._add_property_group_widget(mat_id, 'prop_box_fric')
        trans_prop_box, trans_prop_layout = self._add_property_group_widget(mat_id, 'prop_box_trans')
        tab_widget.addTab(prop_box, 'General')
        tab_widget.addTab(fric_prop_box, 'Friction')
        tab_widget.addTab(trans_prop_box, 'Transport')

        # need a layout on the right
        # create a copy of the param object, when the material changes, set the attrs of the copied param
        # to the values in the selected material
        old_param_objects = set(self.param_helper.param_dict.keys())
        self.param_helper.add_params_to_layout(prop_layout, self.data.materials.material_properties[mat_id])

        mat_type_combo = None
        for param_object, param_info in self.param_helper.param_dict.items():
            if param_object in old_param_objects:
                continue
            if param_info['param_name'] == 'eddy_viscosity_method':
                for widget in param_info['widget_list']:
                    if isinstance(widget, QComboBox):
                        mat_type_combo = widget
                        break
                break

        # set all widget values and hide/show
        self.param_helper.do_param_widgets(None)

        # Use Diffusion Coefficient checkbox
        use_diffusion_checkbox = QCheckBox(prop_box)
        use_diffusion_checkbox.setObjectName('UseDiffusionCoefficient')
        use_diffusion_checkbox.setText('Use Diffusion Coefficient')
        use_diffusion_checkbox.setVisible(False)  # Hidden initially
        prop_layout.addWidget(use_diffusion_checkbox)

        # Set diffusion checkbox to saved value
        material_props = self.data.materials.material_properties[mat_id]
        transport_props = list(material_props.transport_properties.values())
        if transport_props:
            use_diffusion_checkbox.setChecked(transport_props[0].use_diffusion_coefficient)

        # Show/hide use diffusion checkbox based on eddy viscosity method
        def update_diffusion_checkbox_visibility():
            selected_method = mat_type_combo.currentText().lower()
            is_evs = 'evs' in selected_method
            use_diffusion_checkbox.setVisible(is_evs)

        if mat_type_combo:
            mat_type_combo.currentIndexChanged.connect(update_diffusion_checkbox_visibility)
            update_diffusion_checkbox_visibility()

        use_met_condition = QCheckBox(prop_box)
        use_met_condition.setObjectName('UseMeteorological')
        use_met_condition.setText('Include meteorologic condition')
        prop_layout.addWidget(use_met_condition)
        met_layout = QHBoxLayout(prop_box)
        met_label = QLabel(prop_box)
        met_label.setText('Rain or evaporation (as flow per unit area):')
        met_layout.addWidget(met_label)
        met_curve = QPushButton(prop_box)
        met_curve.setText('Edit...')
        met_curve.pressed.connect(self._define_meteorological_curve)
        met_layout.addWidget(met_curve)
        use_met_condition.toggled.connect(met_label.setEnabled)
        use_met_condition.toggled.connect(met_curve.setEnabled)
        if self.data.materials.material_use_meteorological[mat_id] is not None and \
                self.data.materials.material_use_meteorological[mat_id]:
            use_met_condition.setChecked(True)
        else:
            met_label.setEnabled(False)
            met_curve.setEnabled(False)
        prop_layout.addLayout(met_layout)
        prop_layout.addStretch()

        # Create a friction widget for the friction tab.
        friction = FrictionWidget(fric_prop_box, False)
        friction_result = self.data.materials.friction.loc[self.data.materials.friction['STRING_ID'] == mat_id]
        if friction_result.size > 0:
            card_type = friction_result['CARD_2'].iloc[0]
            val1 = float(friction_result['REAL_01'].iloc[0])
            val2 = float(friction_result['REAL_02'].iloc[0])
            val3 = float(friction_result['REAL_03'].iloc[0])
            val4 = float(friction_result['REAL_04'].iloc[0])
            val5 = float(friction_result['REAL_05'].iloc[0])
            friction.set_friction_type(card_type, False, [val1, val2, val3, val4, val5])
        else:
            friction.set_friction_type('', False, [])

        fric_prop_layout.addWidget(friction)

        use_seas_condition = QCheckBox(friction)
        use_seas_condition.setObjectName('UseSeasonal')
        use_seas_condition.setText('Include Seasonal Roughness Adjustment')
        fric_prop_layout.addWidget(use_seas_condition)
        seas_layout = QHBoxLayout(friction)
        seas_label = QLabel(friction)
        seas_label.setText('Roughness adjustment factor (multiplier, variable overtime):')
        seas_layout.addWidget(seas_label)
        seas_curve = QPushButton(friction)
        seas_curve.setText('Edit...')
        seas_curve.pressed.connect(self._define_seasonal_curve)
        seas_layout.addWidget(seas_curve)
        use_seas_condition.toggled.connect(seas_label.setEnabled)
        use_seas_condition.toggled.connect(seas_curve.setEnabled)

        if mat_id in self.data.materials.friction_use_seasonal and \
                self.data.materials.friction_use_seasonal[mat_id] is not None and \
                self.data.materials.friction_use_seasonal[mat_id]:
            use_seas_condition.setChecked(True)
        else:
            seas_label.setEnabled(False)
            seas_curve.setEnabled(False)
        fric_prop_layout.addLayout(seas_layout)
        fric_prop_layout.addStretch()

        # Create a transport constituents widget for the transport tab.
        transport = TransportConstituentMatAssignmentWidget(trans_prop_box)
        transport.setObjectName('transport')
        trans_prop_layout.setContentsMargins(0, 0, 0, 0)
        trans_prop_layout.addWidget(transport)
        transport.set_transport('', self.widgets['transport_select'].constituents)
        if mat_type_combo:
            mat_type_combo.currentTextChanged.connect(transport.filter_model.set_visible_viscosity_method)
            transport.filter_model.set_visible_viscosity_method(mat_type_combo.currentText())
        for row in range(transport.model.rowCount()):
            id_idx = transport.model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_ID_COLUMN)
            constituent_id = int(transport.model.data(id_idx))
            if constituent_id in self.data.materials.material_properties[mat_id].transport_properties:
                constituent = self.data.materials.material_properties[mat_id].transport_properties[constituent_id]
                r_idx = transport.model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_REFINEMENT_COLUMN)
                d_idx = transport.model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_DIFFUSION_COLUMN)
                transport.model.setData(r_idx, constituent.refinement_tolerance)
                transport.model.setData(d_idx, constituent.turbulent_diffusion_rate)
        trans_prop_box.setEnabled(self.widgets['transport_select'].ui.transport_constituents_group.isChecked())

        self.mat_id_to_page[mat_id] = self.widgets['prop_stack'].addWidget(tab_widget)

    def _add_property_group_widget(self, mat_id, object_name):
        """Adds a group box with the material name.

        Args:
            mat_id (int): The id of the material.
            object_name (str): The Qt object name to give the QGroupBox.

        Returns:
            A tuple of (QGroupBox, QVBoxLayout). The group box contains the layout.
        """
        prop_box = QGroupBox()
        prop_box.setObjectName(object_name)
        prop_box.setTitle(self.data.materials.material_properties[mat_id].material_name)
        prop_box.setEnabled(mat_id > 0)
        prop_layout = QVBoxLayout()
        prop_box.setLayout(prop_layout)
        return prop_box, prop_layout

    def _add_material(self, mat_id):
        """Adds a material.

        Args:
            mat_id (int): The id of the material to add.
        """
        self._add_prop_page(mat_id)

    def _delete_materials(self, mat_ids):
        """Deletes materials.

        Args:
            mat_ids (list): List of integer ids of materials that will be deleted.
        """
        self.deleted_mat_ids.extend(mat_ids)
        for mat_id in mat_ids:
            self.mat_id_to_page.pop(mat_id)

    def _update_name(self, item):
        """Updates the titles of group boxes to show the materials name.

        Args:
            item (QStandardItem): The item in the material table that changed its name.
        """
        mat_id = item.data(Qt.UserRole)
        mat_name = item.data(Qt.EditRole)
        if mat_id in self.mat_id_to_page:
            page_idx = self.mat_id_to_page[mat_id]
            tab_widget = self.widgets['prop_stack'].widget(page_idx)
            for tab_idx in range(tab_widget.count()):
                box_widget = tab_widget.widget(tab_idx)
                if box_widget:
                    box_widget.setTitle(mat_name)

    def _set_layout(self, parent_name, layout_name, layout):
        """Adds a layout to the parent.

        Args:
            parent_name (str): Name of parent widget in self.widgets or '' for self
            layout_name (QLay): Name of layout in parent widget
            layout (str): QtLayout to be used
        """
        self.widgets[layout_name] = layout
        if parent_name:
            parent = self.widgets[parent_name]
        else:
            parent = self
        if type(parent) in [QVBoxLayout, QHBoxLayout]:
            parent.addLayout(self.widgets[layout_name])
        else:
            parent.setLayout(self.widgets[layout_name])

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def on_btn_import(self):
        """Imports AdH material properties."""
        file_dir = self.last_import_export_path if self.last_import_export_path else QDir.homePath()
        filename, _ = QFileDialog.getOpenFileName(self, 'Open', dir=file_dir, filter='*.adh_mat')
        if filename:
            self.last_import_export_path = os.path.dirname(filename)
            import_mat = MaterialsIO(filename)
            self.data.materials = import_mat.materials
            self.widgets['table_view'].model.removeRows(0, self.widgets['table_view'].model.rowCount())
            self.widgets['table_view'].setup_table_model()

    def on_btn_export(self):
        """Exports AdH material properties to a file."""
        dir = self.last_import_export_path if self.last_import_export_path else QDir.homePath()
        filename, _ = QFileDialog.getSaveFileName(self, 'Save', dir=dir, filter='*.adh_mat')
        if filename:
            self.last_import_export_path = os.path.dirname(filename)
            self._update_material_name_and_display()
            export_mat = MaterialsIO(filename)
            export_mat.materials = self.data.materials
            export_mat.commit()

    def on_material_changed(self):
        """Called when the current material in the table changes.

        Sets the property stack to shows widgets for the current material.
        """
        current_tab = 0
        if self.widgets['prop_stack'].currentWidget():
            current_tab = max(self.widgets['prop_stack'].currentWidget().currentIndex(), 0)
        # Disable the delete button if the OFF material currently selected.
        index = self.widgets['table_view'].table_view.selectionModel().currentIndex()
        if not index.isValid():
            return
        index_row = index.row()
        index = self.widgets['table_view'].table_view.model().index(index_row, MaterialTableWidget.MAT_NAME_COL)
        enable_delete = True
        # if the 'OFF' material is select then the user can not delete
        if index_row == MaterialsIO.UNASSIGNED_MAT:
            enable_delete = False
        self.widgets['table_view'].toolbar.widgetForAction(
            self.widgets['table_view'].btn_actions[self.widgets['table_view'].delete_icon]
        ).setEnabled(enable_delete)

        mat_id = self.widgets['table_view'].table_view.model().data(index, role=Qt.UserRole)
        if mat_id is None:
            return
        if mat_id not in self.mat_id_to_page:
            self._add_prop_page(mat_id)
        self.widgets['prop_stack'].setCurrentIndex(self.mat_id_to_page[mat_id])
        self.widgets['prop_stack'].currentWidget().setCurrentIndex(current_tab)

    def on_btn_change_all(self):
        """Called when the change all material texture/style button is clicked."""
        dlg = PolygonDisplayOptionsDialog(PolygonOptions(), self)
        dlg.ui.lab_line_color.setVisible(False)
        dlg.ui.color_btn.setVisible(False)
        if dlg and dlg.exec():
            new_opts = dlg.get_options()
            for row in range(self.widgets['table_view'].model.rowCount()):
                idx = self.widgets['table_view'].model.index(row, 0)
                disp_data = self.widgets['table_view'].model.data(idx, Qt.UserRole)
                disp_data.texture = new_opts.texture
                self.widgets['table_view'].model.setData(idx, disp_data, Qt.UserRole)

    def _update_material_name_and_display(self):
        """Updates the material data for either saving or export."""
        row_count = self.widgets['table_view'].model.rowCount()
        fric_drop_list = []
        for row in range(0, row_count):
            display_idx = self.widgets['table_view'].model.index(row, 0)
            name_idx = self.widgets['table_view'].model.index(row, 1)
            mat_id = self.widgets['table_view'].model.data(name_idx, Qt.UserRole)
            mat_name = self.widgets['table_view'].model.data(name_idx, Qt.DisplayRole)
            disp_opts = self.widgets['table_view'].model.data(display_idx, Qt.UserRole)
            self.data.materials.material_properties[mat_id].material_name = mat_name
            self.data.materials.material_display[mat_id] = disp_opts
            # skip this iteration if nothing changed
            if mat_id not in self.mat_id_to_page:
                continue
            page = self.mat_id_to_page[mat_id]
            widget_page = self.widgets['prop_stack'].widget(page)

            # Save seasonal friction data
            use_seas_widget = widget_page.findChildren(QCheckBox, 'UseSeasonal')[0]
            self.data.materials.friction_use_seasonal[mat_id] = use_seas_widget.isChecked()

            # Save meteorological data
            use_met_widget = widget_page.findChildren(QCheckBox, 'UseMeteorological')[0]
            self.data.materials.material_use_meteorological[mat_id] = use_met_widget.isChecked()

            self._save_transport(mat_id, widget_page)
            self._save_friction(mat_id, widget_page, fric_drop_list)
        self._update_transport_constituents_for_save()
        # clean out friction and time series for deleted materials
        for del_mat_id in self.deleted_mat_ids:
            fric_row_index = self.data.materials.friction.index[self.data.materials.friction['STRING_ID'] == del_mat_id]
            if len(fric_row_index):
                fric_drop_list.append(fric_row_index[0])
            if del_mat_id in self.data.materials.time_series:
                self.data.materials.time_series.pop(del_mat_id)
            if del_mat_id in self.data.materials.time_series:
                self.data.materials.time_series.pop(del_mat_id)
        self.data.materials.friction.drop(fric_drop_list, axis=0, inplace=True)
        self.deleted_mat_ids = []

    def _save_friction(self, mat_id, widget_page, fric_drop_list):
        """Saves the friction data from the widgets.

        Args:
            mat_id (int): The material id.
            widget_page (QWidget): The widget page for the material. Contains all tabs.
            fric_drop_list (list): Friction entries to remove later.
        """
        # Save friction data
        fric_widget = widget_page.findChildren(FrictionWidget, 'FrictionWidget')[0]
        fric_type = fric_widget.get_friction_type()
        friction_off = fric_type is None or fric_type == ''
        fric_row_index = self.data.materials.friction.index[self.data.materials.friction['STRING_ID'] == mat_id
                                                            ].tolist()
        if friction_off and len(fric_row_index) > 0:
            # remove the card
            fric_drop_list.append(fric_row_index[0])
        elif not friction_off:
            fric_options = fric_widget.get_friction_values()
            if len(fric_row_index) > 0:
                # update the card
                index = fric_row_index[0]
                self.data.materials.friction.at[index, 'CARD_2'] = fric_type
                for col, val in zip(['REAL_01', 'REAL_02', 'REAL_03', 'REAL_04', 'REAL_05'], fric_options):
                    self.data.materials.friction.at[index, col] = val
            else:
                # add the card
                values = ['FR', fric_type, mat_id]
                values.extend(fric_options)
                # 8 columns in the DataFrame, need to fill blank columns
                for _ in range(8 - len(values)):
                    values.append(None)
                df2 = pd.DataFrame(
                    [values],
                    columns=["CARD", "CARD_2", "STRING_ID", "REAL_01", "REAL_02", "REAL_03", "REAL_04", "REAL_05"]
                )
                self.data.materials.friction = pd.concat([self.data.materials.friction, df2])

    def _save_transport(self, mat_id, widget_page):
        """Saves the transport constituent assignment data from the widgets.

        Args:
            mat_id (int): The material id.
            widget_page (QWidget): The widget page for the material. Contains all tabs.
        """
        # Save transport data
        self.data.materials.material_properties[mat_id].transport_properties = {}
        trans_widget = widget_page.findChild(TransportConstituentMatAssignmentWidget, 'transport')
        id_col = TransportConstituentMatAssignmentWidget.SOURCE_ID_COLUMN
        refine_col = TransportConstituentMatAssignmentWidget.SOURCE_REFINEMENT_COLUMN
        diffuse_col = TransportConstituentMatAssignmentWidget.SOURCE_DIFFUSION_COLUMN
        diffusion_checkbox = widget_page.findChild(QCheckBox, 'UseDiffusionCoefficient')
        use_diffusion = diffusion_checkbox.isChecked() if diffusion_checkbox else False

        for constituent_row in range(trans_widget.model.rowCount()):
            id_idx = trans_widget.model.index(constituent_row, id_col)
            refine_idx = trans_widget.model.index(constituent_row, refine_col)
            diffuse_idx = trans_widget.model.index(constituent_row, diffuse_col)
            constituent_id = int(trans_widget.model.data(id_idx))
            trans_prop = MaterialTransportProperties()
            trans_prop.refinement_tolerance = float(trans_widget.model.data(refine_idx))
            trans_prop.turbulent_diffusion_rate = float(trans_widget.model.data(diffuse_idx))
            trans_prop.use_diffusion_coefficient = use_diffusion
            self.data.materials.material_properties[mat_id].transport_properties[constituent_id] = trans_prop

    def _update_transport_constituents_for_save(self):
        """Updates the transport constituents.

        Goes through all of the materials, removes values for removed constituents, adds default values for added
        constituents.
        """
        # Set if we are using transport constituents.
        use_transport_checked = self.widgets['transport_select'].ui.transport_constituents_group.isChecked()
        self.data.use_transport = 1 if use_transport_checked else 0

        # Look at all transport constituents that are defined.
        transport_comp = self.widgets['transport_select'].constituents
        if transport_comp is None:
            return
        all_constituents = transport_comp.user_constituents['ID'].values.tolist()
        if transport_comp.param_control.salinity:
            all_constituents.append(1)
        if transport_comp.param_control.temperature:
            all_constituents.append(2)
        if transport_comp.param_control.vorticity:
            all_constituents.append(3)
        if all_constituents:
            all_constituents.sort()
            # Go through the defined values for transport constituents for each material.
            for mat_id in self.data.materials.material_properties.keys():
                all_current = list(self.data.materials.material_properties[mat_id].transport_properties.keys())
                for old_id in all_current:
                    if old_id not in all_constituents:
                        self.data.materials.material_properties[mat_id].transport_properties.pop(old_id)
                for con_id in all_constituents:
                    if con_id not in all_current:
                        self.data.materials.material_properties[mat_id].transport_properties[con_id] = \
                            MaterialTransportProperties()

    def _define_meteorological_curve(self):
        """Defines a curve for the rainfall."""
        index = self.widgets['table_view'].table_view.selectionModel().currentIndex()
        if not index.isValid():
            return
        mat_id = self.widgets['table_view'].table_view.model().data(index, role=Qt.UserRole)
        curve_id = self.data.materials.material_meteorological_curve[mat_id]
        if curve_id not in self.data.materials.time_series:
            if curve_id < 0:
                curve_id = 0
                while curve_id in self.data.materials.time_series:
                    curve_id = curve_id + 1
                self._mat_comp.data.materials.material_meteorological_curve[mat_id] = curve_id
            self.data.materials.time_series[curve_id] = TimeSeries()
            df = pd.DataFrame([[0.0, 0.0]], columns=['X', 'Y'])
            self.data.materials.time_series[curve_id].time_series = \
                pd.concat([self.data.materials.time_series[curve_id].time_series, df])
        series = self.data.materials.time_series[curve_id].time_series

        time_series_editor = TimeSeriesEditor(series, icon=self.windowIcon(), parent=self)
        if time_series_editor.run():
            self.data.materials.time_series[curve_id].time_series = time_series_editor.series

    def _define_seasonal_curve(self):
        """Defines a curve for the rainfall."""
        index = self.widgets['table_view'].table_view.selectionModel().currentIndex()
        if not index.isValid():
            return
        mat_id = self.widgets['table_view'].table_view.model().data(index, role=Qt.UserRole)
        if mat_id not in self.data.materials.friction_seasonal_curve:
            self.data.materials.friction_seasonal_curve[mat_id] = -1
        curve_id = self.data.materials.friction_seasonal_curve[mat_id]
        if curve_id not in self.data.materials.time_series:
            if curve_id < 0:
                curve_id = 0
                while curve_id in self.data.materials.time_series:
                    curve_id = curve_id + 1
                self.data.materials.friction_seasonal_curve[mat_id] = curve_id
            self.data.materials.time_series[curve_id] = TimeSeries()
            df = pd.DataFrame([[0.0, 0.0]], columns=['X', 'Y'])
            self.data.materials.time_series[curve_id].time_series = \
                pd.concat([self.data.materials.time_series[curve_id].time_series, df])
        series = self.data.materials.time_series[curve_id].time_series

        time_series_editor = TimeSeriesEditor(series, icon=self.windowIcon(), parent=self)
        if time_series_editor.run():
            self.data.materials.time_series[curve_id].time_series = time_series_editor.series

    @Slot(bool)
    def _use_constituents_changed(self, on):
        """Sets whether a transport constituent is used or not.

        Args:
            on (bool): True is using a transport constituent. Transport constituent widgets will be enabled if true.
        """
        for idx in range(self.widgets['prop_stack'].count()):
            self.widgets['prop_stack'].widget(idx).widget(self._TRANSPORT_TAB).setEnabled(on)

    def set_transport(self, transport_uuid, transport_comp, transport_name, use_transport):
        """Sets the transport constituents data for all materials.

        Args:
            transport_uuid (str): The uuid of the transport constituents component.
            transport_comp (TransportConstituentsIO): The transport constituents data to set.
            transport_name (str): The name of the transport constituents component.
            use_transport (bool): True if using a transport constituent.
        """
        self.widgets['transport_select'].set_transport(transport_uuid, transport_comp, transport_name=transport_name)
        self.widgets['transport_select'].ui.transport_constituents_group.setChecked(use_transport)

    @Slot(TransportConstituentsIO)
    def _set_material_transport(self, transport_comp):
        """Sets the transport constituents for each material.

        Args:
            transport_comp (TransportConstituentsIO): The transport constituents data to set.
        """
        self.data.transport_uuid = self.widgets['transport_select'].constituents_uuid
        self.data.info.attrs['transport_uuid'] = self.data.transport_uuid
        # self.data.materials.material_properties[mat_id].transport_properties = {}
        for idx in range(self.widgets['prop_stack'].count()):
            mat_group = self.widgets['prop_stack'].widget(idx).widget(self._TRANSPORT_TAB)
            transport = mat_group.findChild(TransportConstituentMatAssignmentWidget, 'transport')
            transport.set_transport('', transport_comp)

    def accept(self):
        """Saves the material data."""
        self._update_material_name_and_display()
        return super().accept()

    def reject(self):
        """Cancels out of the dialog."""
        return super().reject()

# def main():
#     """Demonstrates a simple use of the dialog.
#
#     """
#     import sys
#     from PySide2.QtWidgets import (QApplication)
#     from xms.adh.data.materials import Materials
#
#     data = Materials('C:/Temp/AdH_Material_file.nc')
#     app = QApplication(sys.argv)
#     dialog = MaterialDialog(None, 'AdH Material Dialog', data.param_control, None)
#     if dialog.exec():
#         data.param_control = dialog.data
#         data.commit()
#
#     sys.exit(app.exec_())
#
#
# if __name__ == "__main__":
#     main()
