"""A dialog for weir values."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QDialog, QDialogButtonBox)

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
import xms.adh.gui.resources.adhqrc  # noqa F401
from xms.adh.gui.weir_dialog_ui import Ui_WeirDialog


class WeirDialog(QDialog):
    """A dialog for assigning materials to polygons."""
    def __init__(self, win_cont, icon, title):
        """Allows the user to edit arc boundary conditions.

        Args:
            win_cont (QWidget): Parent window
            icon (QIcon): Window icon
            title (str): Window title
            data (MaterialsIO): Materials object
        """
        super().__init__(win_cont)
        self.ui = Ui_WeirDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Boundary_Condition_Assignment'

        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.setWindowIcon(icon)
        self.setWindowTitle(title)
        self._setup()

    def _setup(self):
        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.height.setValidator(QxDoubleValidator())
        self.ui.crest_elevation.setValidator(QxDoubleValidator())
        self.ui.length.setValidator(QxDoubleValidator())
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def set_assignment(self, upstream_arc_id, downstream_arc_id, upstream_point_id, downstream_point_id):
        """Sets the text for the upstream and downstream roles. Also shows a warning if the structure is invalid.

        Args:
            upstream_arc_id (:obj:`tuple` of str and bool): Upstream arc id string and if the ids are valid.
            downstream_arc_id (:obj:`tuple` of str and bool): Downstream arc id string and if the ids are valid.
            upstream_point_id (:obj:`tuple` of str and bool): Upstream point id string and if the ids are valid.
            downstream_point_id (:obj:`tuple` of str and bool): Downstream point id string and if the ids are valid.
        """
        _ID = 0  # noqa N806
        _VALID = 1  # noqa N806
        option_1 = f'Arc {upstream_arc_id[_ID]} as upstream, arc {downstream_arc_id[_ID]} as downstream'
        option_2 = f'Arc {downstream_arc_id[_ID]} as upstream, arc {upstream_arc_id[_ID]} as downstream'
        self.ui.arc_upstream_downstream.addItems([option_1, option_2])
        option_1 = f'Point {upstream_point_id[_ID]} as upstream, point {downstream_point_id[_ID]} as downstream'
        option_2 = f'Point {downstream_point_id[_ID]} as upstream, point {upstream_point_id[_ID]} as downstream'
        self.ui.point_upstream_downstream.addItems([option_1, option_2])
        valid = upstream_arc_id[_VALID] and downstream_arc_id[_VALID] and \
            upstream_point_id[_VALID] and downstream_point_id[_VALID]
        if not valid:
            self.ui.warning_label.show()
            self.ui.warning_label.setStyleSheet("QLabel { color : red; }")
            self.ui.buttonBox.removeButton(self.ui.buttonBox.button(QDialogButtonBox.Ok))
        else:
            self.ui.warning_label.hide()

    def set_name(self, name):
        """Sets the name of the structure.

        Args:
            name (str): Name of the structure.
        """
        self.ui.weir_name.setText(name)

    def set_options(self, length, crest_elev, height):
        """Set the length, elevation, and height of the weir.

        Args:
            length (float): Length of the weir.
            crest_elev (float): Crest elevation of the weir.
            height (float): Height of the weir.
        """
        self.ui.length.setText(str(length))
        self.ui.crest_elevation.setText(str(crest_elev))
        self.ui.height.setText(str(height))

    def get_arc_assignment_changed(self):
        """Get whether the arc assignment has changed.

        Returns:
            (bool): True if arc assignment changed.
        """
        is_first_index = self.ui.arc_upstream_downstream.currentIndex() == 0
        return not is_first_index

    def get_point_assignment_changed(self):
        """Get whether the point assignment has changed.

        Returns:
            (bool): True if point assignment changed.
        """
        is_first_index = self.ui.point_upstream_downstream.currentIndex() == 0
        return not is_first_index

    def get_name(self):
        """Get name of structure.

        Returns:
            (str): Name of structure.
        """
        return self.ui.weir_name.text()

    def get_options(self):
        """Get the length, elevation, and height of the weir.

        Returns:
            (:obj:`tuple` of :obj:`float`): tuple containing weir:
                - length
                - elevation
                - height
        """
        return float(self.ui.length.text()), float(self.ui.crest_elevation.text()), float(self.ui.height.text())

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when the Ok button is clicked."""
        return QDialog.accept(self)

    def reject(self):
        """Called when the Cancel button is clicked."""
        return QDialog.reject(self)
