"""Python wrapping for xms.api._xmsapi.dmi.ProgressLoop."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.api._xmsapi.dmi import ProgressLoop as CProgressLoop
from xms.api.dmi import Context as PyContext


class ProgressLoop:
    """The pure Python wrapper for C++ exposed xms.api._xmsapi.dmi.ProgressLoop objects."""
    def __init__(self, instance=None):
        """Construct the wrapper.

        Note that this is a low-level API object that should not be constructed directly.

        Args:
            instance (xms.api._xmsapi.dmi.ProgressLoop): The C++ object to wrap
        """
        if instance is None:
            self._instance = CProgressLoop()
        else:
            self._instance = instance

    @property
    def command_line_output_file(self):
        """Returns the filesystem path to the progress loop executable's standard output file."""
        return self._instance.GetCommandLineOutputFile()

    @property
    def error_output_file(self):
        """Returns the filesystem path to the progress loop executable's standard error file."""
        return self._instance.GetErrorOutputFile()

    @property
    def plot_db_file(self):
        """Returns the filesystem path to the progress loop's plot database file."""
        return self._instance.GetPlotDBFile()

    @property
    def working_directory_of_process(self):
        """Returns the filesystem path to the progress loop executable's working directory."""
        return self._instance.GetWorkingDirectoryOfProcess()

    @property
    def progress_context(self):
        """Returns the xms.api.dmi.Context associated with the progress script."""
        return PyContext(instance=self._instance.GetProgressContext())

    @progress_context.setter
    def progress_context(self, ctxt):
        """Set the Context associated with the progress script.

        Args:
            ctxt (xms.api.dmi.Context): The progress script Context
        """
        self._instance.SetProgressContext(ctxt._instance)

    def get_plot_data_names(self, plot_name):
        """Retrieves the plot data names for a given plot.

        Args:
            plot_name (str): XML-defined name of the plot to retrieve data names for

        Returns:
            list of str: Plot data names of the specified plot
        """
        return self._instance.GetPlotDataNames(plot_name)

    def get_plot_context(self, plot_name):
        """Retrieves the Context associated with a specified plot.

        Args:
            plot_name (str): XML-defined name of the plot to retrieve Context for

        Returns:
            xms.api.dmi.Context: See description
        """
        return PyContext(instance=self._instance.GetPlotContext(plot_name))

    def get_plot_data_contexts(self, plot_name, plot_data_name):
        """Retrieves the Contexts associated with a specified plot and plot data.

        Args:
            plot_name (str): XML-defined name of the plot to retrieve Contexts for
            plot_data_name (str): XML-defined name of the plot data to retrieve Contexts for

        Returns:
            list of xms.api.dmi.Context: See description
        """
        return [PyContext(instance=ctxt) for ctxt in self._instance.GetPlotDataContexts(plot_name, plot_data_name)]

    def get_plot_data_db_table(self, plot_name, plot_data_name):
        """Retrieves the SQL table names associated with a specified plot and plot data.

        Args:
            plot_name (str): XML-defined name of the plot to retrieve Contexts for
            plot_data_name (str): XML-defined name of the plot data to retrieve Contexts for

        Returns:
            list of str: See description
        """
        return self._instance.GetPlotDataDBTable(plot_name, plot_data_name)

    def set_progress_function(self, functor):
        """Sets the method that will be called with each iteration of the progress loop.

        Args:
            functor (callable): Pointer the the progress function method
        """
        self._instance.SetProgressFunction(functor)

    def start_loop(self):
        """Starts the progress loop with a time interval retrieved from XMS."""
        self._instance.StartLoop()

    def start_constant_loop(self, milliseconds):
        """Starts the progress loop with a specified time interval.

        Args:
            milliseconds (int): The number of milliseconds between progress loop iterations
        """
        self._instance.StartConstantLoop(milliseconds)
