"""The help pane in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QCheckBox, QComboBox, QLabel

# 3. Aquaveo modules

# 4. Local modules


class StructureOvertopTab:
    """The profiles tab in the structure dialog."""
    def __init__(self, parent, widgets):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            widgets (:obj:`dict`): Dictionary of widgets
        """
        self._parent = parent
        self._widgets = widgets
        self._add_widgets()

    def _add_widgets(self):
        """Set up the UI."""
        dlg = self._parent
        vlayout = self._widgets['tab2_scroll_area_vert_layout']
        # check box for srh overtopping
        self._widgets['tog_srh_overtop'] = QCheckBox('Model overtopping (only check if overtopping exists)')
        checked = True if dlg.data.data_dict['srh_overtopping'] else False
        self._widgets['tog_srh_overtop'].setChecked(checked)
        self._widgets['tog_srh_overtop'].stateChanged.connect(self._on_srh_overtop)
        vlayout.addWidget(self._widgets['tog_srh_overtop'])
        # weir length
        dlg.add_floating_point_widget('srh_weir_length', 'Weir Length:', dlg.data.data_dict['srh_weir_length'], vlayout)
        # weir type
        self._widgets['txt_weir_type'] = QLabel("Weir type:")
        vlayout.addWidget(self._widgets['txt_weir_type'])
        weir_types = ['Paved', 'Gravel', 'Single', 'Double', 'Sharp', 'Broad', 'User']
        self._widgets['cbx_weir_type'] = QComboBox()
        self._widgets['cbx_weir_type'].addItems(weir_types)
        self._widgets['cbx_weir_type'].setCurrentIndex(0)
        if dlg.data.data_dict['srh_weir_type'] in weir_types:
            self._widgets['cbx_weir_type'].setCurrentText(dlg.data.data_dict['srh_weir_type'])
        vlayout.addWidget(self._widgets['cbx_weir_type'])
        self._widgets['cbx_weir_type'].currentIndexChanged.connect(self._on_change_weir_type)
        # user weir type Cw
        dlg.add_floating_point_widget('srh_weir_cw', 'Weir Cw:', dlg.data.data_dict['srh_weir_cw'], vlayout)
        dlg.add_floating_point_widget('srh_weir_a', 'Weir a:', dlg.data.data_dict['srh_weir_a'], vlayout)
        dlg.add_floating_point_widget('srh_weir_b', 'Weir b:', dlg.data.data_dict['srh_weir_b'], vlayout)

        vlayout.addStretch()
        self._on_srh_overtop()

    def _on_srh_overtop(self):
        """Handle the checkbox for srh overtopping."""
        flag = self._widgets['tog_srh_overtop'].isChecked()
        widgets = [
            'txt_srh_weir_length', 'edt_srh_weir_length', 'txt_weir_type', 'cbx_weir_type', 'txt_srh_weir_cw',
            'edt_srh_weir_cw', 'txt_srh_weir_a', 'edt_srh_weir_a', 'txt_srh_weir_b', 'edt_srh_weir_b'
        ]
        for w in widgets:
            self._widgets[w].setVisible(flag)
        if flag:
            self._on_change_weir_type()

    def _on_change_weir_type(self):
        """Change to the weir type combobox."""
        weir_type = self._widgets['cbx_weir_type'].currentText()
        flag = False
        if weir_type == 'User':
            flag = True
        widgets = [
            'txt_srh_weir_cw', 'edt_srh_weir_cw', 'txt_srh_weir_a', 'edt_srh_weir_a', 'txt_srh_weir_b', 'edt_srh_weir_b'
        ]
        for w in widgets:
            self._widgets[w].setVisible(flag)
