"""This module reads a grid file into SMS."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math
import os
import sys
import uuid

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.constraint import Numbering, Orientation
from xms.constraint import QuadtreeGridBuilder
from xms.constraint.quadtree import QuadGridCells
from xms.coverage.activity import ActivityCoverage
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.data_objects.parameters import UGrid

# 4. Local modules


class CMSFlowH5GridImporter:
    """Importer for CMS-Flow h5 grid file."""
    def __init__(self, filename=None, projection=None, bathymetry=None, cell_activity_group=''):
        """Constructor for grid importer.

        Args:
            filename (Optional[str]): Path to the H5 grid file
            projection (Optional[Projection]): Projection to assign the grid
            bathymetry (Optional[tuple[str, str]]): The name of the bathymetry dataset file and its H5 group path in
                the file
            cell_activity_group (Optional[str]): Path in the H5 grid file to the cell activity dataset
        """
        self.query = None
        self.filename = filename
        self.projection = projection
        self.bathymetry = bathymetry  # [filename, group_path] if defined
        self.cell_activity_group = cell_activity_group
        self.ugrid_uuid = str(uuid.uuid4())
        self.quad_ugrid = None
        self.activity_array = None
        self.activity_coverage = None

    def read(self):
        """Top-level function that triggers the read of the grid.h5 file.

        A new simulation will be created in SMS. Any accompanying model-native files in the same directory will
        also be written.

        """
        self.get_sms_data()
        self.read_h5_grid_file()
        self.send_sms_data()

    def set_activity(self):
        """Create an activity coverage and set the model on/off cells of the Quadtree UGrid."""
        if not self.bathymetry:  # [filename, group_path] if defined
            return  # No depth dataset defined
        filename, group_path = self.bathymetry
        with h5py.File(filename, 'r') as f:
            if group_path not in f:
                return
            depths = f[group_path][:][0]

        self.quad_ugrid.cell_elevations = [  # Invert depths to be negative elevations
            depth * -1.0 if depth and not math.isclose(depth, -999.0) else depth for depth in depths
        ]
        if self.activity_array is not None:  # Set the model on/off cells and create an activity coverage
            activity_array = self.activity_array.tolist()
            self.quad_ugrid.model_on_off_cells = activity_array
            cov_builder = GridCellToPolygonCoverageBuilder(self.quad_ugrid, activity_array, self.projection, 'Activity')
            new_cov_geom = cov_builder.create_polygons_and_build_coverage()
            self.activity_coverage = ActivityCoverage()
            self.activity_coverage.m_cov = new_cov_geom
            activity = {}
            for poly_val, poly_ids in cov_builder.dataset_polygon_ids.items():
                for poly_id in poly_ids:
                    activity[poly_id] = poly_val != 0
            self.activity_coverage.m_activity = activity

    def read_h5_grid_file(self):
        """Read the h5 grid file into a rectilinear grid."""
        if not os.path.isfile(self.filename):
            sys.stderr.write("Rectilinear xmdf grid file not found - {}".format(self.filename))
            return

        with h5py.File(self.filename, 'r') as file:
            attrs = None
            group = None
            for file_key in file.keys():
                file_obj = file.get(file_key)
                if not isinstance(file_obj, h5py.Dataset):
                    group = file_obj
                    attrs = group.attrs

            if attrs is None or group is None:
                sys.stderr.write("Unable to open rectilinear xmdf grid file - {}".format(self.filename))

            if 'Origin' in attrs.keys():
                origin = attrs['Origin']
            else:
                origin = [0.0, 0.0, 0.0]

            angle = attrs['Bearing'][0]
            num_i = attrs['NumI'][0]
            num_j = attrs['NumJ'][0]
            coords_i = group.get('CoordsI')
            coords_j = group.get('CoordsJ')
            locations_x = [0.0] + coords_i[:num_i].tolist()
            locations_y = [0.0] + coords_j[:num_j].tolist()

            # Look for an activity array dataset.
            if self.cell_activity_group and self.cell_activity_group in file:
                self.activity_array = file[self.cell_activity_group][:]
            elif 'PROPERTIES/CellTypes' in group:  # Check the standard location if not specified
                self.activity_array = group['PROPERTIES/CellTypes'][:]

        builder = QuadtreeGridBuilder()
        numbering = Numbering.kji
        orientation = (Orientation.x_increase, Orientation.y_increase)

        quad_grid_cells = QuadGridCells()
        quad_grid_cells.set_base_grid(num_i, num_j, 1, numbering, False)
        quad_grid_cells.set_all_included()

        builder.is_2d_grid = True
        builder.locations_x = locations_x
        builder.locations_y = locations_y
        builder.origin = origin
        builder.angle = angle
        builder.numbering = numbering
        builder.orientation = orientation
        builder.quad_grid_cells = quad_grid_cells
        self.quad_ugrid = builder.build_grid()
        self.quad_ugrid.uuid = self.ugrid_uuid
        self.set_activity()

    def get_sms_data(self):
        """Get the path and filename of the .cmcards file. Get the simulation Context vertex id."""
        self.query = Query(timeout=300000)

        # Read file is the XMDF solution
        self.filename = self.query.read_file

        if not os.path.isfile(self.filename):
            sys.stderr.write("Grid file not found - {}".format(self.filename))

    def send_sms_data(self):
        """Send the imported data to SMS."""
        xmc_file = os.path.join(self.query.process_temp_directory, f'{str(self.ugrid_uuid)}.xmc')
        self.quad_ugrid.write_to_file(xmc_file)
        ugrid = UGrid(xmc_file, uuid=self.ugrid_uuid, projection=self.projection)
        self.query.add_ugrid(ugrid)
        self.query.send()
