"""This module is for the rubble mound structures hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog

# 4. Local modules
from xms.cmsflow.components.cmsflow_component import CmsflowComponent
from xms.cmsflow.components.id_files import RM_INITIAL_ATT_ID_FILE, RM_INITIAL_COMP_ID_FILE
from xms.cmsflow.data.rm_structures_data import RMStructuresData
from xms.cmsflow.file_io import io_util
from xms.cmsflow.gui.rubble_mound_dlg import RubbleMoundDlg
from xms.cmsflow.merge.merge_rubble_mound import RubbleMoundMerger


class RMStructuresComponent(CmsflowComponent):
    """A hidden Dynamic Model Interface (DMI) component for the CMS-Flow rubble mound structures coverage."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Display Options...', 'open_display_options')
        ]
        self.polygon_commands = [('Assign Rubble Mound Jetty...', 'open_poly_attributes')]
        self.data = RMStructuresData(self.main_file)
        if os.path.exists(self.main_file):
            self.cov_uuid = self.data.info.attrs['cov_uuid']
            # Copy default display options if needed
            comp_dir = os.path.dirname(self.main_file)
            self.rm_poly_id_file = os.path.join(comp_dir, 'rm_poly.display_ids')
            self.disp_opts_file = os.path.join(comp_dir, 'rm_display_options.json')
            if not os.path.exists(self.disp_opts_file):
                # Read the default polygon display options, and save ourselves a copy with a randomized UUID.
                poly_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                poly_default_file = os.path.join(
                    os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                    'rm_display_options.json'
                )
                poly_json_dict = read_display_options_from_json(poly_default_file)
                poly_categories.from_dict(poly_json_dict)
                poly_categories.comp_uuid = self.uuid
                write_display_options_to_json(self.disp_opts_file, poly_categories)
                # Save our display list UUID to the main file
                self.data.info.attrs['polygon_display_uuid'] = poly_categories.uuid
                self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            # Change the display UUID to be something different.
            new_comp = RMStructuresComponent(new_main_file)
            new_uuid = self._update_display_uuids_for_duplicate(new_comp.uuid, new_comp.disp_opts_file)
            new_comp.data.info.attrs['polygon_display_uuid'] = new_uuid
            new_comp.data.commit()
            action_requests.append(self.get_display_refresh_action())
        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        return [], [self.get_display_refresh_action()]

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (:obj:`list` of :obj:`tuple`): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (:obj:`dict`): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        merger = RubbleMoundMerger(self)
        return merger.merge(merge_list)

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        self.cov_uuid = query.parent_item_uuid()
        if not self.cov_uuid:
            return [('ERROR', 'Could not get CMS-Flow rubble mound structures coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.main_file), RM_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), RM_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            io_util.remove(initial_att_file)
            io_util.remove(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.polygon, att_id, comp_id)

        self.data.info.attrs['cov_uuid'] = self.cov_uuid
        self.data.commit()
        # Send the default display list to XMS.
        self.display_option_list.append(
            XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
        return [], []

    def open_poly_attributes(self, query, params, win_cont):
        """Opens the rubble mound polygon dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        comp_id = self._unpack_xms_data(params[0], TargetType.polygon)
        incremented_id = False
        if comp_id not in self.data.polygons.comp_id:  # Create default attributes if polygon has never been assigned.
            comp_id = self.data.add_rm_atts()
            incremented_id = True  # We just got a new component id, so don't duplicate on OK
        single_poly = self.data.polygons.loc[dict(comp_id=[comp_id])]
        dlg = RubbleMoundDlg(
            rm_structures_data=single_poly, multi_msg=self.dlg_message, pe_tree=query.project_tree, parent=win_cont
        )
        if dlg.exec_():
            if incremented_id:  # Was previously unassigned, so we already got a new component id. Just update it.
                self.data.update_rm_polygon(comp_id, dlg.rm_structures_data)
            else:  # Always generate a new component id, old one might be shared if assigned during a multi-select.
                comp_id = self.data.add_rm_atts(dlg.rm_structures_data)
            self._append_comp_id_to_file(self.rm_poly_id_file, comp_id)
            self.data.commit()
            for poly_id in self.selected_att_ids:
                self.update_component_id(TargetType.polygon, poly_id, comp_id)
            # Send back updated display lists to XMS after ActionRequest
            self.display_option_list.append(
                # Update the polygon display list only.
                XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid)
            )
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Opens the display options dialog for boundary condition arcs.

        Args:
            query (:obj:'xmsapi.dmi.Query'): A way to communicate with XMS.
            params (:obj:'list' of :obj:'str'): Generic map of parameters.
            win_cont (:obj:'PySide2.QtWidgets.QWidget'): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Read the default polygon display options, and save ourselves a copy with a randomized UUID.
        poly_categories = CategoryDisplayOptionList()
        poly_json_dict = read_display_options_from_json(self.disp_opts_file)
        poly_categories.from_dict(poly_json_dict)

        # TODO: add categories for any polygon that exists that we have not added to the category

        dlg = CategoryDisplayOptionsDialog([poly_categories], win_cont)
        if dlg.exec_():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                disp_file = self.disp_opts_file
                write_display_options_to_json(disp_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
                )
        return [], []

    def _remove_unused_component_ids(self):
        """Remove unused component ids from the data and display option id files.

        Notes:
            Coverage component ids must be loaded from the Query before calling this.
        """
        # Drop unused component ids from the datasets
        comp_ids = list(self.comp_to_xms.get(self.cov_uuid, {}).get(TargetType.polygon, {}).keys())
        mask = self.data.polygons.comp_id.isin(comp_ids)
        self.data.polygons = self.data.polygons.where(mask, drop=True)
        # Rewrite the display option id file now to clear out non-existent component ids.
        if len(self.data.polygons.comp_id) > 0:
            write_display_option_ids(self.rm_poly_id_file, self.data.polygons.comp_id.data)
        else:  # If we don't have any polygons, just delete the id file.
            io_util.remove(self.rm_poly_id_file)
        self.data.vacuum()  # Reclaim disk space in the NetCDF file
