"""Base class for xarray based component data classes."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from distutils.dir_util import copy_tree
import os
from pathlib import Path

# 2. Third party modules
import h5py
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.core.filesystem import filesystem as io_util

# 4. Local modules


def try_to_compute_relative_path(folder_path, file_path):
    r"""Computes and returns the path of file relative to path.

    Also changes all '\' to '/' so tests work on Windows and Linux.

    Args:
        folder_path (str): A full path included in 'file'.
        file_path (str): A full path to a file

    Returns:
        str: A relative path.
    """
    try:
        return io_util.compute_relative_path(folder_path, file_path)
    except Exception:  # If converting to relative path failed, return the original absolute path.
        return file_path


class MainFileBase:
    """
    A 'root' class for things that take a main-file in their constructor.

    This class exists to break an unfortunate dependency chain between XarrayBase and VisibleCoverageComponentDataBase.
    Classes that inherit VCCDB will typically inherit XB as well. Since VCCDB is abstract, it needs to come last in the
    base class list, or else it will cause errors due to unimplemented abstract methods. Since XB is often used by
    itself, it can't assume its super class takes a main-file, so it can't pass a main-file to its super class, so if it
    appears first in the base class list then VCCDB won't get a main-file.

    In short, neither class can come before the other in the base class list.

    By having both of them inherit from MainFileBase, XB can safely pass its main-file to its super class. When XB is
    used alone, it will send the main-file here, where it will be swallowed. When it's used in combination with VCCDB,
    it will go there, which will send it back here, where it will be swallowed. Either way, everyone is initialized and
    happy.
    """
    def __init__(self, main_file: str | Path):
        """Swallow the main-file parameter."""
        super().__init__()


class XarrayBase(MainFileBase):
    """Base class for xarray based component data classes."""
    def __init__(self, filename):
        """Initializes the data class.

        Args:
            filename (str): The netcdf file (with path) associated with this instance data.
        """
        super().__init__(filename)
        self._filename = filename
        self._info = self.get_dataset('info', False)
        if self._info is None:
            self._info = xr.Dataset()
        if 'FILE_TYPE' not in self._info.attrs:
            self._info.attrs['FILE_TYPE'] = 'UNDEFINED'
        if 'VERSION' not in self._info.attrs:
            self._info.attrs['VERSION'] = '0.0.0'
        # Set the project folder for resolving external file paths
        env_proj_dir = os.path.dirname(os.environ.get(XmEnv.ENVIRON_PROJECT_PATH, ''))
        old_proj_dir = self._info.attrs.get('proj_dir', '')
        if not os.path.isdir(old_proj_dir):
            # If the attr has not been set, or is set to a non-existent folder, update it from the environment.
            # We do not want to always update the path because we may be executing a save-as operation.
            self._info.attrs['proj_dir'] = env_proj_dir

    def get_dataset(self, path, lazy):
        """Get a handle to an xarray dataset inside this data's file.

        Args:
            path: Path in the NetCDF file to the dataset.
            lazy: If False, the entire dataset will immediately be loaded into memory.

        Returns:
            xarray.Dataset: Dataset handle or None on failure
        """
        try:
            if lazy:
                return xr.open_dataset(self._filename, group=path)
            else:
                return xr.load_dataset(self._filename, group=path)
        except Exception:
            return None

    @property
    def info(self):
        """Return the info dataset.

        Returns:
            xarray.Dataset: Dict of parameters
        """
        return self._info

    def commit(self):
        """Save in memory datasets to the NetCDF file."""
        mode = 'a'
        if not os.path.exists(self._filename) or not os.path.isfile(self._filename):
            mode = 'w'
        if self._info is not None:
            if mode != 'w':
                self._drop_h5_groups(['info'])
            self.info.to_netcdf(self._filename, group='info', mode=mode)

    def close(self):
        """Closes the H5 file and does not write any data that is in memory."""
        self._info.close()

    def vacuum(self):
        """Rewrites the file. This recovers space in the file from dropping datasets and groups."""
        pass

    def _drop_h5_groups(self, groups):
        """Delete a group or dataset from an H5 or NetCDF file.

        Need to use H5 calls because the xarray and Python NetCDF libraries do not provide
        a convenient enough way to overwrite existing datasets in a file without overwriting
        the entire file.

        Args:
            groups (list): List of paths in the file to remove
        """
        with h5py.File(self._filename, 'a') as f:
            for group in groups:
                try:
                    del f[group]
                except Exception:
                    pass  # Try to clean up the other groups

    # External file reference stuff
    def update_file_paths(self):
        """Called before resaving an existing project.

        All referenced filepaths should be converted to relative from the project directory. Should already be stored
        in the component main file since this is a resave operation.

        Returns:
            (str): Message on failure, empty string on success

        """
        return ''

    def update_proj_dir(self):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Returns:
            (str): Message on failure, empty string on success

        """
        return ''

    def copy_external_files(self):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Returns:
            (str): Message on failure, empty string on success

        """
        return ''

    def does_file_exist(self, file):
        """Determine if a file in our persistent data still exist.

        If file is not absolute, will check if relative from the project directory exists.

        Args:
         file (str): Relative or absolute file path to check the existence of

        Returns:
            (bool): True if the file exists

        """
        return io_util.does_file_exist(file, self.info.attrs['proj_dir'])

    @staticmethod
    def _convert_filepath(attrs, param_name, proj_dir, old_proj_dir, logger=None):
        """Convert a parameter filepath to be relative to the project save location.

        Args:
            attrs (dict): Attrs of the dataset containing the parameter
            param_name (str): Name of the parameter
            proj_dir (str): Path to the new project save location
            old_proj_dir (str): Path to the old project save location. On a SAVE event, this should be empty string. If
                empty string, any file paths that are already relative will be left alone. If non-empty string, will
                use to recreate absolute paths from any file paths that are already relative. If that absolute path
                does not exist, the value in the attrs will be set to empty string.
            logger (logging.Logger): Logger to print errors to
        Returns:
            (bool): True if the parameter filepath was manipulated.
        """
        try:
            file_path = attrs[param_name].strip()
            if not file_path:
                return False  # Filename has not been defined for this attr

            if not os.path.isabs(file_path):  # File is already relative
                if not os.path.isdir(old_proj_dir):  # No base project to recreate absolute path
                    return False  # If called on resave, only want to update file paths that are already absolute
                # Relative from old project. Convert to absolute then relative from new location.
                file_path = io_util.resolve_relative_path(old_proj_dir, file_path)

            if os.path.exists(file_path):
                attrs[param_name] = try_to_compute_relative_path(proj_dir, file_path)
            else:
                attrs[param_name] = ''  # Clear the file if it no longer exists.
            return True
        except Exception:
            if logger is not None:
                logger.exception('Error updating external paths to be relative from the project directory.')
        return False

    @staticmethod
    def _copy_attr_file(attrs, attr_name, old_project_dir, target_dir, logger=None):
        """Copy a file referenced in a dict to a given location.

        Args:
            attrs (dict): xarray.Dataset attrs containing the file to copy
            attr_name (str): Key of the attr containing the file to copy
            old_project_dir (str): File location of the old project directory
            target_dir (str): File location of the destination folder to copy file to. Should be the new project folder.
            logger (logging.Logger): Logger to print errors to
        """
        try:
            abs_path = attrs[attr_name].strip()
            if not abs_path:
                # If the filename attr is not defined, don't continue because we might end up thinking it is a valid
                # directory.
                return

            if not os.path.isabs(abs_path) and os.path.isdir(old_project_dir):
                # Relative from old project. Convert to absolute then relative to new location.
                abs_path = io_util.resolve_relative_path(old_project_dir, abs_path)

            # In the project_data folder if package save (next to the XMS project file).
            new_file_path = os.path.join(target_dir, os.path.basename(abs_path))
            if os.path.isfile(abs_path):
                io_util.copyfile(abs_path, new_file_path)
                attrs[attr_name] = try_to_compute_relative_path(target_dir, new_file_path)
            elif os.path.isdir(abs_path):
                copy_tree(abs_path, new_file_path)
                attrs[attr_name] = try_to_compute_relative_path(target_dir, new_file_path)
            else:
                attrs[attr_name] = ''  # Clear the file if it no longer exists.
        except Exception:
            if logger is not None:
                logger.exception('Error copying external file(s) to package save location.')

    @staticmethod
    def _copy_table_files(data_variable, old_project_dir, target_dir, logger=None):
        """Copy a file referenced in a dict to a given location.

        Args:
            data_variable (xarray.Variable): The xarray Variable containing the files to copy
            old_project_dir (str): path to old project directory
            target_dir (str): File location of the destination folder to copy file to. Should be the new project folder.
            logger (logging.Logger): Logger to print errors to
        """
        for i in range(data_variable.shape[0]):
            try:
                abs_path = str(data_variable.data[i]).strip()
                if not abs_path:
                    # If the filename attr is not defined, don't continue because we might end up thinking it is a valid
                    # directory.
                    continue

                if not os.path.isabs(abs_path) and os.path.isdir(old_project_dir):
                    # Relative from old project. Convert to absolute then relative to new location.
                    abs_path = io_util.resolve_relative_path(old_project_dir, abs_path)

                # In the project_data folder if package save (next to the XMS project file).
                new_file_path = os.path.join(target_dir, os.path.basename(abs_path))
                if os.path.isfile(abs_path):  # Found an absolute file path, make relative to the new project directory.
                    io_util.copyfile(abs_path, new_file_path)
                    data_variable.data[i] = try_to_compute_relative_path(target_dir, new_file_path)
                elif os.path.isdir(abs_path):
                    copy_tree(abs_path, new_file_path)
                    data_variable.data[i] = try_to_compute_relative_path(target_dir, new_file_path)
                else:
                    data_variable.data[i] = ''  # Clear the file if it no longer exists.
            except Exception:
                if logger is not None:
                    logger.exception('Error copying external file(s) to package save location.')

    @staticmethod
    def _update_table_files(variable, proj_dir, old_proj_dir, logger=None):
        """Update filename in an xarray dataset.

        Args:
            variable (xarray.Variable): xarray Variable containing the filenames. Assumed to be 1-D array of strings
            proj_dir (str): Path to the new project save location
            old_proj_dir (str): Path to the old project save location. On a SAVE event, this should be empty string. If
                empty string, any file paths that are already relative will be left alone. If non-empty string, will
                use to recreate absolute paths from any file paths that are already relative. If that absolute path
                does not exist, the value in the Variable will be set to empty string.
            logger (logging.Logger): Logger to print errors to
        Returns:
            (bool): True if the parameter filepath was manipulated.
        """
        changed_path = False
        for i in range(variable.shape[0]):
            try:
                file_path = str(variable.data[i]).strip()
                if not file_path:
                    continue  # File has not been defined for this row

                if not os.path.isabs(file_path):  # File is already relative
                    if not os.path.isdir(old_proj_dir):  # No base project to recreate absolute path from
                        continue  # If called on resave, only want to update file paths that are already absolute
                    # Relative from old project. Convert to absolute then relative to new location.
                    file_path = io_util.resolve_relative_path(old_proj_dir, file_path)

                if os.path.exists(file_path):  # Found absolute file path, make relative to the project directory.
                    variable.data[i] = try_to_compute_relative_path(proj_dir, file_path)
                else:
                    variable.data[i] = ''  # Clear the file if it no longer exists.
                changed_path = True
            except Exception:
                if logger is not None:
                    logger.exception('Error updating external paths to be relative from the project directory.')
        return changed_path
