"""Get coverage data for a coverage dump file."""

__copyright__ = '(C) Copyright Aquaveo 2020'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DataDumpIOBase

# 4. Local modules
from xms.coverage import coverage_util
from xms.coverage.activity import ActivityCoverage
from xms.coverage.att_table_coverage_dump import AttTableCoverageDump
from xms.coverage.mesh_generator import MeshGeneratorCoverage


def get_coverage_data(file_name: str) -> Optional[DataDumpIOBase]:
    """
    Get coverage data for a dump file.

    Args:
        file_name: The path to the coverage file.

    Returns:
        The coverage dump data.
    """
    file_type = None
    with h5py.File(file_name, 'r') as f:
        the_cov_name = coverage_util.read_coverage_name(f)
        if the_cov_name == '':
            f.close()
            return
        file_type = None
        if 'CoverageType' in f['Map Data'][the_cov_name].attrs.keys():
            coverage_type = f['Map Data'][the_cov_name].attrs['CoverageType']
            if len(coverage_type) == 1:
                file_type = coverage_type[0].decode()
    # match file_type:
    #     case 'ACTIVITY_CLASSIFICATION':
    #         return ActivityCoverage(file_name)
    #     case 'MESH_GENERATION':
    #         return MeshGeneratorCoverage(file_name)
    #     case 'NOCOVERAGE':
    #         return AttTableCoverageDump(file_name)
    # yapf can't handle match/case yet
    if file_type == 'ACTIVITY_CLASSIFICATION':
        return ActivityCoverage(file_name)
    elif file_type == 'MESH_GENERATION':
        return MeshGeneratorCoverage(file_name)
    elif file_type == 'NOCOVERAGE':
        return AttTableCoverageDump(file_name)
    return None
