"""ADCIRC files writer."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.adcirc.data.sim_data import SimData
from xms.adcirc.dmi.fort15_data_getter import Fort15DataGetter
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort15_writer import Fort15Writer

# 4. Local modules

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


class AdcircWriter:
    """Writer class for the ADCIRC from CSTORM."""
    def __init__(self, logger, xms_data, cstorm_data):
        """Constructor.

        Args:
            logger (:obj:`logging.Logger`): the logger
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
            cstorm_data (:obj:`dict`): fills in data for CSTORM config file
        """
        self._logger = logger
        self._xms_data = xms_data
        self._cstorm_data = cstorm_data

    def write(self, wsid):
        """Writes the adcirc simulation for the CSTORM model.

        Args:
            wsid (:obj:`int`): the wave service id used by CSTORM
        """
        self._logger.info('Writing ADCIRC simulation')
        adcirc_sim = self._xms_data.adcirc_sim
        if adcirc_sim is None:
            self._logger.error('No ADCIRC simulation defined. Aborting export.')
            raise RuntimeError
        sim, sim_comp = adcirc_sim
        sim_data = SimData(sim_comp.main_file)
        if sim_data.general.attrs['run_padcirc'] == 0:
            self._logger.error(
                '"Run parallel ADCIRC" not selected in ADCIRC model control. Parallel ADCIRC is '
                'required for CSTORM. Aborting export.'
            )
            raise RuntimeError
        if wsid in [3, 6]:
            if sim_data.wind.attrs['wave_radiation'] != 400:  # STWAVE coupled
                self._logger.error(
                    'Current CSTORM setting require ADCIRC wind wave radiation to be "STWAVE coupled". '
                    'Change setting in ADCIRC Model Control, Wind Tab. Aborting export.'
                )
                raise RuntimeError

        # Find out if we are exporting as a template
        template = self._cstorm_data.get('template', False)
        adc_xms_data = {'sim_item': sim, 'sim_data': sim_data, 'sim_export': True}
        getter = Fort15DataGetter(self._xms_data.query, adc_xms_data)
        getter.retrieve_data()
        adc_writer = Fort15Writer('fort.15', self._xms_data.query, adc_xms_data, template=template)
        XmLog().instance.set_logger(self._logger)
        adc_writer.write()

        # get data for the config file
        self._cstorm_data['adc_reftime'] = sim_data.timing.attrs['ref_date']
        self._cstorm_data['adcgrid'] = '"fort.14"'
        self._cstorm_data['adcprocs'] = sim_data.general.attrs['num_comp_proc']
        self._cstorm_data['writerprocs'] = sim_data.general.attrs['num_io_proc']
        dtdp = sim_data.timing.attrs['DTDP']
        runday = sim_data.timing.attrs['RUNDAY']
        run_seconds = runday * 24 * 60 * 60
        adcfinish = run_seconds / dtdp
        self._cstorm_data['DTDP'] = dtdp
        self._cstorm_data['adcstart'] = 0
        self._cstorm_data['adcfinish'] = int(adcfinish)
