"""Module for handling XMS interprocess communication with EwnCoverageComponent."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
import xms.api.dmi.XmsEnvironment as Xme
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.constraint.ugrid_builder import UGridBuilder
from xms.data_objects.parameters import Arc, Component, Coverage, Point, UGrid

# 4. Local modules


def add_adcirc_bc_coverage(query, adcirc_bc_coverage):
    """Add the output UGrid from inserting EWN features to SMS project explorer.

    Args:
        query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.
        adcirc_bc_coverage (:obj:`xms.data_objects.parameters.coverage`): new adcirc bc coverage

    Returns:
        (:obj:`list[int]`): build vertices
    """
    # Add data_object xms.api Query.
    bc_cov = adcirc_bc_coverage['cov_geom']
    bc_comp = adcirc_bc_coverage['cov_comp']
    arc_json = os.path.join(os.path.dirname(bc_comp.main_file), 'bc_display_options.json')
    pt_json = os.path.join(os.path.dirname(bc_comp.main_file), 'bc_point_display_options.json')
    bc_comp.display_option_list = [
        XmsDisplayMessage(file=arc_json, edit_uuid=bc_cov.uuid),  # arcs
        XmsDisplayMessage(file=pt_json, edit_uuid=bc_cov.uuid),  # points
    ]
    do_comp = Component(
        main_file=bc_comp.main_file, unique_name='Bc_Component', model_name='ADCIRC', comp_uuid=bc_comp.uuid
    )
    query.add_coverage(
        do_coverage=bc_cov,
        model_name='ADCIRC',
        coverage_type='Boundary Conditions',
        components=[do_comp],
        component_keywords=[
            {
                'component_coverage_ids': [bc_comp.uuid, bc_comp.update_ids],
                'display_options': bc_comp.get_display_options()
            }
        ],
    )


def add_output_ugrid(query, ugrid, name='', projection=None):
    """Add the output UGrid from inserting EWN features to SMS project explorer.

    Args:
        query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.
        ugrid (:obj:`xmsconstraint.UGrid`): The output UGrid
        name (:obj:`str`): name of mesh
        projection (:obj:`data_objects.parameters.Projection`): xms projection
    """
    if not ugrid:
        return

    # Create a CoGrid.
    ug_builder = UGridBuilder()
    ug_builder.set_is_2d()
    ug_builder.set_ugrid(ugrid)
    co_grid = ug_builder.build_grid()

    add_output_cogrid(query, co_grid, name, projection)


def add_output_cogrid(query, cogrid, name='', projection=None):
    """Add the output xmsconstraint.cogrid from inserting EWN features to SMS project explorer.

    Args:
        query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.
        cogrid (:obj:`xmsconstraint`): The output UGrid
        name (:obj:`str`): name of mesh
        projection (:obj:`data_objects.parameters.Projection`): xms projection
    """
    if not cogrid:
        return

    # Look for the XMS temp directory.
    temp_dir = Xme.xms_environ_process_temp_directory()
    if not temp_dir:  # Use system temp if XMS environment variable not set.
        import tempfile
        temp_dir = tempfile.gettempdir()

    # Write CoGrid to a file.
    if not cogrid.uuid:
        cogrid.uuid = str(uuid.uuid4())
    co_grid_file = os.path.join(temp_dir, f'{cogrid.uuid}.xmc')
    cogrid.write_to_file(co_grid_file)

    # Create data_objects UGrid.
    new_name = name if name else 'EWN Feature Mesh'
    do_ugrid = UGrid(cogrid_file=co_grid_file, name=new_name, uuid=cogrid.uuid, projection=projection)

    # Add data_object UGrid to xms.api Query.
    query.add_ugrid(do_ugrid)


def add_non_intersecting_transects_coverage(query, non_intersecting_transects, cov_name):
    """Create a coverage of the non-intersecting transects and send to XMS.

    Args:
        query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.
        non_intersecting_transects (:obj:`list`): List of the start and end node x,y,z tuples of the non-intersecting
            transects
        cov_name (:obj:`str`): name of coverage that is created
    """
    # Create arcs from the non-intersecting transect segments.
    pt_id = 1
    arc_id = 1
    arcs = []
    for segment in non_intersecting_transects:
        start_node = Point(x=segment[0][0], y=segment[0][1], z=segment[0][2], feature_id=pt_id)
        pt_id += 1
        end_node = Point(x=segment[1][0], y=segment[1][1], z=segment[1][2], feature_id=pt_id)
        pt_id += 1
        arc = Arc(start_node=start_node, end_node=end_node, feature_id=arc_id)
        arc_id += 1
        arcs.append(arc)

    # Create data_objects Coverage.
    do_cov = Coverage(name=cov_name, uuid=str(uuid.uuid4()), projection=query.display_projection)
    do_cov.arcs = arcs
    do_cov.complete()

    # Add data_object UGrid to xms.api Query.
    query.add_coverage(do_cov)


def add_adh_materials_coverage(query, adh_mat_coverage):
    """Add the ADH materials coverage.

    Args:
        query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.
        adh_mat_coverage (:obj:`xms.data_objects.parameters.coverage`): new adh material coverage

    Returns:
        (:obj:`list[int]`): build vertices
    """
    mat_cov = adh_mat_coverage['cov_geom']
    mat_comp = adh_mat_coverage['cov_comp']
    mat_json = os.path.join(os.path.dirname(mat_comp.main_file), 'material_display_options.json')
    mat_comp.display_option_list = [
        XmsDisplayMessage(file=mat_json, edit_uuid=mat_comp.cov_uuid),  # materials
    ]
    do_comp = Component(
        main_file=mat_comp.main_file, unique_name='MaterialConceptual', model_name='AdH', comp_uuid=mat_comp.uuid
    )
    query.add_coverage(
        do_coverage=mat_cov,
        model_name='AdH',
        coverage_type='Materials',
        components=[do_comp],
        component_keywords=[
            {
                'component_coverage_ids': [mat_comp.uuid, mat_comp.update_ids],
                'display_options': mat_comp.get_display_options()
            }
        ],
    )


def add_transition_poly_coverage(query, transition_coverage):
    """Add the transition polygon coverage.

    Args:
        query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.
        transition_coverage (:obj:`xms.data_objects.parameters.coverage`): new transition polygon coverage
    """
    query.add_coverage(transition_coverage)
