"""DMI component to manage running of a FUNWAVE simulation from SMS."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import argparse
import os
from pathlib import Path
import time

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand, XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.components.bases.run_base import RunBase

# 4. Local modules
from xms.funwave.feedback.solution_import_runner import import_solution_with_feedback


class SimRunner(RunBase):
    """Class that handles running FUNWAVE."""

    def __init__(self, dummy_mainfile=''):
        """Initializes the class.

        Args:
            dummy_mainfile (:obj:`str`): Unused. Just to keep constructor consistent with component classes.
        """
        super().__init__()

    def read_solution(self, query, params, win_cont):
        """Reads the FUNWAVE Solution.

        Args:
            query (:obj:`xms.data_objects.parameters.Query`): A Query object to communicate with SMS.
            params (:obj:`dict`): Generic map of parameters
            win_cont (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the tuple being the message
                level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        import_solution_with_feedback(input_txt_file=params[0]['input_txt_file'], geom_uuid=params[0]['geom_uuid'],
                                      query=query, parent=win_cont)
        return [], []

    def get_executables(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for
            query (:obj:`xms.data_objects.parameters.Query`): A Query object to communicate with GMS.
            filelocation (:obj:`str`): The location of input files for the simulation

        Returns:
            (:obj:`list[xms.api.dmi.ExecutableCommand]`): The executable objects to run and the action requests that
            go with it
        """
        input_txt_file = os.path.join(filelocation, 'input.funwave')
        if not os.path.isfile(input_txt_file):
            XmEnv.report_error(
                f'Unable to run FUNWAVE because "{input_txt_file}" was not found. Ensure the simulation has been '
                f'saved.'
            )
            return []

        entry_point = os.path.normpath(
            os.path.join(os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'run_model.py')
        )
        cmd = ExecutableCommand(executable=entry_point, model='FUNWAVE', executable_order=0, display_name='FUNWAVE',
                                run_weight=100, progress_script='xml_entry_points/funwave_progress.py',
                                executable_is_script=True)
        cmd.add_commandline_arg(f'-i {input_txt_file}')
        cmd.add_solution_file(self.get_solution_load_actions(sim, query, filelocation)[0])
        return [cmd]

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for
            query (:obj:`xms.data_objects.parameters.Query`): A Query object to communicate with SMS
            filelocation (:obj:`str`): The location of input files for the simulation

        Returns:
            (:obj:`list[xms.api.dmi.ActionRequest]`): The solution load ActionRequests for the simulation
        """
        input_txt_file = os.path.join(filelocation, 'input.funwave')
        geom_uuid = self._get_grid_uuid(sim, query)
        if geom_uuid is not None:
            load_sol = ActionRequest(modality='MODAL', class_name='SimRunner', module_name='xms.funwave.dmi.sim_runner',
                                     method_name='read_solution', main_file=input_txt_file,
                                     parameters={'input_txt_file': input_txt_file,
                                                 'geom_uuid': geom_uuid})
            return [load_sol]
        return []

    def _get_grid_uuid(self, sim, query):
        """Get the UUID of the simulation grid.

        Args:
            sim (:obj:`xms.data_objects.parameters.Simulation`): The Simulation you want to load the solution for
            query (:obj:`xms.data_objects.parameters.Query`): A Query object to communicate with SMS

        Returns:
            The UUID of the simulation's linked grid.
        """
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, sim.uuid)
        grids = tree_util.descendants_of_type(tree_root=sim_node, xms_types=['TI_CGRID2D_PTR'], allow_pointers=True)
        if not grids:
            grids = tree_util.descendants_of_type(tree_root=sim_node, xms_types=['TI_CGRID2D_PTR'], allow_pointers=True)
        if grids:
            return grids[0].uuid


def launch_model():
    """Entry point for launching the FUNWAVE model executable."""
    parser = argparse.ArgumentParser(description='FUNWAVE Runner')
    parser.add_argument('-i', '--input_txt_file', help='FUNWAVE input file', required=True)
    args = parser.parse_args()
    input_txt_file = args.input_txt_file.strip()
    run_folder = os.path.dirname(input_txt_file)
    log_file = Path(input_txt_file).parent / 'LOG.txt'
    if log_file.is_file():
        os.remove(log_file)

    output_file = XmEnv.xms_environ_stdout_file()
    with open(output_file, 'a') as stdout:
        stdout.write(f'Run Folder: {run_folder}\n')
        stdout.write(f'Input File: {input_txt_file}\n')
        stdout.write(f'Log File: {log_file}\n')
        stdout.write('Looking for simulation completion in LOG.txt...\n')
        stdout.flush()
        running = True
        while running:
            time.sleep(1.0)
            if log_file.is_file():
                log_text = open(log_file, 'r').read()
                if log_text.find('Normal Termination!') > 0:
                    stdout.write('Simulation completed!\n')
                    running = False
