"""Dataset control_reader."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
from pathlib import Path
from typing import Optional
import uuid

# 2. Third party modules
import numpy

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.datasets.dataset_writer import DatasetWriter
from xms.gmi.data.generic_model import Type

# 4. Local modules
from xms.funwave.data.model import get_variables_by_name
import xms.funwave.file_io.io_util as io_util


def _file_word_generator(file: str):
    """Generator to iterate over words in a file.

    Args:
        file(:obj:`str`): The file path.

    Returns:
        Generator for words of file.
    """
    for line in open(file):
        for word in line.split():
            yield word


def get_dataset_values(values: dict[str, any], depth_file: str) -> dict[str, str]:
    """Get the variables with dataset values.

    Args:
        values(:obj:`dict[str, any]`: The input.funwave values.
        depth_file (str): filename of the depth file

    Returns:
        (:obj:`dict[str, str]`):Variable name and file path for dataset values.
    """
    dataset_values = {}
    variables_by_name = get_variables_by_name()
    for name, value in values.items():
        is_dataset = variables_by_name[name].parameter_type == Type.DATASET
        if is_dataset:
            dataset_values[name] = value
    if depth_file is not None and depth_file != '':
        dataset_values['DEPTH_FILE'] = depth_file
    return dataset_values


def get_grid_dimensions(values: dict[str, any]) -> tuple[Optional[int], Optional[int]]:
    """Get the grid dimensions.

    Args:
        values(:obj:`dict[str, any]`): The input.funwave values.

    Returns:
        (:obj:`tuple`)
    """
    n_size = None
    m_size = None
    variables = get_variables_by_name()
    if 'Mglob' in variables:
        m_size = values['Mglob']
    if 'Nglob' in variables:
        n_size = values['Nglob']
    return n_size, m_size


def read_datasets(to_read: dict[str, str], input_txt_path: Path, value_count: int,
                  geom_uuid: str) -> list[tuple[str, DatasetWriter, str, numpy.array]]:
    """Read data files and create datasets.

    Args:
        to_read(:obj:`dict[str, str]`): Files to read with variable name and file path.
        input_txt_path(:obj:`Path`): The path to the input.funwave file.
        value_count(:obj:`int`): The number of values to read.
        geom_uuid(:obj:`str`): The geometry UUID.

    Returns:
        (:obj:`list[tuple(str, DatasetWriter, str, list)]`): The datasets with variable name and dataset writer.
    """
    datasets = []
    for name, file_path in to_read.items():
        if file_path is None or file_path == '':
            continue
        file = str(input_txt_path / file_path)
        values = numpy.fromiter(_file_word_generator(file), dtype=float, count=value_count)
        dataset_uuid = str(uuid.uuid4())
        dataset_name = None
        if name in get_variables_by_name():
            dataset_name = get_variables_by_name()[name].parameter_name
        if dataset_name is None:
            dataset_name = name.lower()
        dataset_writer = DatasetWriter(name=dataset_name, dset_uuid=dataset_uuid, geom_uuid=geom_uuid, location='cells',
                                       dtype='d')
        dataset_writer.write_xmdf_dataset([0.0], [values])
        datasets.append((name, dataset_writer, dataset_uuid, values))
    return datasets


def build_datasets(values: dict[str, any], input_txt_path: Path,
                   geom_uuid: str, depth_file: str) -> list[tuple[str, DatasetWriter, str, numpy.array]]:
    """Build datasets for items in values from input.funwave.

    Args:
        values(:obj:`dict[str, any]`): Dictionary of values read in input.funwave.
        input_txt_path(:obj:`Path`): The path to the input.funwave file.
        geom_uuid(:obj:`str`): The geometry UUID.
        depth_file (str): filename of the depth file

    Returns:
       (:obj:`list[tuple(str, DatasetWriter, str, np.array)]`): List of datasets.
    """
    n_size, m_size = get_grid_dimensions(values)
    to_read = get_dataset_values(values, depth_file)
    if n_size is not None and m_size is not None:
        value_count = n_size * m_size
        datasets = read_datasets(to_read, input_txt_path.parent, value_count, geom_uuid)
        return datasets
    return []


class DatasetReader:
    """Class to read a FUNWAVE input file."""

    def __init__(self, filename: str, query):
        """Constructor.

        Args:
            filename: The input file.
            query (:obj:`Query`): The XMS interprocess communicator.
        """
        self._logger = logging.getLogger('xms.funwave')
        self._filename = Path(os.path.normpath(filename))
        self._query = query

    def _read_dataset(self):
        """Read an input.txt file."""
        try:
            self._logger.info(f'Parsing input.txt file: {io_util.logging_filename(self._filename)}')

            project_tree = self._query.copy_project_tree()

            active_sim = None
            sims = tree_util.descendants_of_type(tree_root=project_tree, unique_name='SimComponent',
                                                 model_name='FUNWAVE')
            for sim in sims:
                if sim.is_active_item:
                    active_sim = sim
            if active_sim is None and len(sims) > 0:
                active_sim = sims[0]

            grid = tree_util.descendants_of_type(tree_root=active_sim, xms_types=['TI_CGRID2D_PTR'],
                                                 allow_pointers=True, only_first=True)
            geom_uuid = grid.uuid

            to_read = {}
            to_read[self._filename.stem] = self._filename.name
            [[_, dataset, _, _]] = read_datasets(to_read, self._filename.parent, grid.num_cells, geom_uuid)

            self._query.add_dataset(dataset)

            self._logger.info('Finished!')

        except Exception:
            self._logger.exception('Unexpected error reading input.funwave file.')

    def send(self, query):
        """Send all the imported data back to SMS.

        Args:
            query (:obj:`Query`): The XMS interprocess communicator
        """
        if query:
            query.send()

    def read(self):
        """Read all the FUNWAVE simulations."""
        if not os.path.isfile(self._filename):
            self._logger.error(f'Unable to find input file: {self._filename}')
        else:
            self._read_dataset()
